// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Package matchertest provides mock matcher for testing.
package matchertest

import (
	"context"
	"os"
	"testing"

	"github.com/google/osv-scalibr/extractor"
	"github.com/google/osv-scalibr/guidedremediation/internal/vulns"
	"github.com/ossf/osv-schema/bindings/go/osvschema"
	"gopkg.in/yaml.v3"
)

// MockVulnerabilityMatcher is a mock vulnerability matcher for testing.
type MockVulnerabilityMatcher []*osvschema.Vulnerability

// MatchVulnerabilities returns all vulnerabilities in the mockVulnerabilityMatcher that affect each of the Package items.
func (mvc MockVulnerabilityMatcher) MatchVulnerabilities(ctx context.Context, pkgs []*extractor.Package) ([][]*osvschema.Vulnerability, error) {
	result := make([][]*osvschema.Vulnerability, len(pkgs))
	for i, pkg := range pkgs {
		for _, vuln := range mvc {
			if vulns.IsAffected(vuln, pkg) {
				result[i] = append(result[i], vuln)
			}
		}
	}
	return result, nil
}

type mockVulns struct {
	Vulns []*osvschema.Vulnerability `yaml:"vulns"`
}

// NewMockVulnerabilityMatcher creates a mock vulnerability matcher for testing.
// It loads vulnerability data from a YAML file specified by vulnsYAML.
func NewMockVulnerabilityMatcher(t *testing.T, vulnsYAML string) MockVulnerabilityMatcher {
	t.Helper()
	f, err := os.Open(vulnsYAML)
	if err != nil {
		t.Fatalf("failed opening mock vulns: %v", err)
	}
	defer f.Close()
	dec := yaml.NewDecoder(f)

	var vulns mockVulns
	if err := dec.Decode(&vulns); err != nil {
		t.Fatalf("failed decoding mock vulns: %v", err)
	}
	return MockVulnerabilityMatcher(vulns.Vulns)
}
