package megolm_test

import (
	"testing"

	"github.com/stretchr/testify/assert"

	"maunium.net/go/mautrix/crypto/goolm/megolm"
)

var startData [megolm.RatchetParts * megolm.RatchetPartLength]byte

func init() {
	startValue := []byte("0123456789ABCDEF0123456789ABCDEF")
	copy(startData[:], startValue)
	copy(startData[32:], startValue)
	copy(startData[64:], startValue)
	copy(startData[96:], startValue)
}

func TestAdvance(t *testing.T) {
	m, err := megolm.New(0, startData)
	assert.NoError(t, err)

	expectedData := [megolm.RatchetParts * megolm.RatchetPartLength]byte{
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46,
		0xba, 0x9c, 0xd9, 0x55, 0x74, 0x1d, 0x1c, 0x16, 0x23, 0x23, 0xec, 0x82, 0x5e, 0x7c, 0x5c, 0xe8,
		0x89, 0xbb, 0xb4, 0x23, 0xa1, 0x8f, 0x23, 0x82, 0x8f, 0xb2, 0x09, 0x0d, 0x6e, 0x2a, 0xf8, 0x6a,
	}
	m.Advance()
	assert.Equal(t, m.Data[:], expectedData[:], "result after advancing the ratchet is not as expected")

	//repeat with complex advance
	m.Data = startData
	expectedData = [megolm.RatchetParts * megolm.RatchetPartLength]byte{
		0x54, 0x02, 0x2d, 0x7d, 0xc0, 0x29, 0x8e, 0x16, 0x37, 0xe2, 0x1c, 0x97, 0x15, 0x30, 0x92, 0xf9,
		0x33, 0xc0, 0x56, 0xff, 0x74, 0xfe, 0x1b, 0x92, 0x2d, 0x97, 0x1f, 0x24, 0x82, 0xc2, 0x85, 0x9c,
		0x70, 0x04, 0xc0, 0x1e, 0xe4, 0x9b, 0xd6, 0xef, 0xe0, 0x07, 0x35, 0x25, 0xaf, 0x9b, 0x16, 0x32,
		0xc5, 0xbe, 0x72, 0x6d, 0x12, 0x34, 0x9c, 0xc5, 0xbd, 0x47, 0x2b, 0xdc, 0x2d, 0xf6, 0x54, 0x0f,
		0x31, 0x12, 0x59, 0x11, 0x94, 0xfd, 0xa6, 0x17, 0xe5, 0x68, 0xc6, 0x83, 0x10, 0x1e, 0xae, 0xcd,
		0x7e, 0xdd, 0xd6, 0xde, 0x1f, 0xbc, 0x07, 0x67, 0xae, 0x34, 0xda, 0x1a, 0x09, 0xa5, 0x4e, 0xab,
		0xba, 0x9c, 0xd9, 0x55, 0x74, 0x1d, 0x1c, 0x16, 0x23, 0x23, 0xec, 0x82, 0x5e, 0x7c, 0x5c, 0xe8,
		0x89, 0xbb, 0xb4, 0x23, 0xa1, 0x8f, 0x23, 0x82, 0x8f, 0xb2, 0x09, 0x0d, 0x6e, 0x2a, 0xf8, 0x6a,
	}
	m.AdvanceTo(0x1000000)
	assert.Equal(t, m.Data[:], expectedData[:], "result after advancing the ratchet is not as expected")

	expectedData = [megolm.RatchetParts * megolm.RatchetPartLength]byte{
		0x54, 0x02, 0x2d, 0x7d, 0xc0, 0x29, 0x8e, 0x16, 0x37, 0xe2, 0x1c, 0x97, 0x15, 0x30, 0x92, 0xf9,
		0x33, 0xc0, 0x56, 0xff, 0x74, 0xfe, 0x1b, 0x92, 0x2d, 0x97, 0x1f, 0x24, 0x82, 0xc2, 0x85, 0x9c,
		0x55, 0x58, 0x8d, 0xf5, 0xb7, 0xa4, 0x88, 0x78, 0x42, 0x89, 0x27, 0x86, 0x81, 0x64, 0x58, 0x9f,
		0x36, 0x63, 0x44, 0x7b, 0x51, 0xed, 0xc3, 0x59, 0x5b, 0x03, 0x6c, 0xa6, 0x04, 0xc4, 0x6d, 0xcd,
		0x5c, 0x54, 0x85, 0x0b, 0xfa, 0x98, 0xa1, 0xfd, 0x79, 0xa9, 0xdf, 0x1c, 0xbe, 0x8f, 0xc5, 0x68,
		0x19, 0x37, 0xd3, 0x0c, 0x85, 0xc8, 0xc3, 0x1f, 0x7b, 0xb8, 0x28, 0x81, 0x6c, 0xf9, 0xff, 0x3b,
		0x95, 0x6c, 0xbf, 0x80, 0x7e, 0x65, 0x12, 0x6a, 0x49, 0x55, 0x8d, 0x45, 0xc8, 0x4a, 0x2e, 0x4c,
		0xd5, 0x6f, 0x03, 0xe2, 0x44, 0x16, 0xb9, 0x8e, 0x1c, 0xfd, 0x97, 0xc2, 0x06, 0xaa, 0x90, 0x7a,
	}
	m.AdvanceTo(0x1041506)
	assert.Equal(t, m.Data[:], expectedData[:], "result after advancing the ratchet is not as expected")
}

func TestAdvanceWraparound(t *testing.T) {
	m, err := megolm.New(0xffffffff, startData)
	assert.NoError(t, err)
	m.AdvanceTo(0x1000000)
	assert.EqualValues(t, 0x1000000, m.Counter, "counter not correct")

	m2, err := megolm.New(0, startData)
	assert.NoError(t, err)
	m2.AdvanceTo(0x2000000)
	assert.EqualValues(t, 0x2000000, m2.Counter, "counter not correct")
	assert.Equal(t, m.Data, m2.Data, "result after wrapping the ratchet is not as expected")
}

func TestAdvanceOverflowByOne(t *testing.T) {
	m, err := megolm.New(0xffffffff, startData)
	assert.NoError(t, err)
	m.AdvanceTo(0x0)
	assert.EqualValues(t, 0x0, m.Counter, "counter not correct")

	m2, err := megolm.New(0xffffffff, startData)
	assert.NoError(t, err)
	m2.Advance()
	assert.EqualValues(t, 0x0, m2.Counter, "counter not correct")
	assert.Equal(t, m.Data, m2.Data, "result after wrapping the ratchet is not as expected")
}

func TestAdvanceOverflow(t *testing.T) {
	m, err := megolm.New(0x1, startData)
	assert.NoError(t, err)
	m.AdvanceTo(0x80000000)
	m.AdvanceTo(0x0)
	assert.EqualValues(t, 0x0, m.Counter, "counter not correct")

	m2, err := megolm.New(0x1, startData)
	assert.NoError(t, err)
	m2.AdvanceTo(0x0)
	assert.EqualValues(t, 0x0, m2.Counter, "counter not correct")
	assert.Equal(t, m.Data, m2.Data, "result after wrapping the ratchet is not as expected")
}
