//
// Copyright 2021, Sander van Harmelen
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package gitlab

import (
	"net/http"
)

type (
	ProjectBadgesServiceInterface interface {
		ListProjectBadges(pid any, opt *ListProjectBadgesOptions, options ...RequestOptionFunc) ([]*ProjectBadge, *Response, error)
		GetProjectBadge(pid any, badge int64, options ...RequestOptionFunc) (*ProjectBadge, *Response, error)
		AddProjectBadge(pid any, opt *AddProjectBadgeOptions, options ...RequestOptionFunc) (*ProjectBadge, *Response, error)
		EditProjectBadge(pid any, badge int64, opt *EditProjectBadgeOptions, options ...RequestOptionFunc) (*ProjectBadge, *Response, error)
		DeleteProjectBadge(pid any, badge int64, options ...RequestOptionFunc) (*Response, error)
		PreviewProjectBadge(pid any, opt *ProjectBadgePreviewOptions, options ...RequestOptionFunc) (*ProjectBadge, *Response, error)
	}

	// ProjectBadgesService handles communication with the project badges
	// related methods of the GitLab API.
	//
	// GitLab API docs: https://docs.gitlab.com/api/project_badges/
	ProjectBadgesService struct {
		client *Client
	}
)

var _ ProjectBadgesServiceInterface = (*ProjectBadgesService)(nil)

// ProjectBadge represents a project badge.
//
// GitLab API docs:
// https://docs.gitlab.com/api/project_badges/#list-all-badges-of-a-project
type ProjectBadge struct {
	ID               int64  `json:"id"`
	Name             string `json:"name"`
	LinkURL          string `json:"link_url"`
	ImageURL         string `json:"image_url"`
	RenderedLinkURL  string `json:"rendered_link_url"`
	RenderedImageURL string `json:"rendered_image_url"`
	// Kind represents a project badge kind. Can be empty, when used PreviewProjectBadge().
	Kind string `json:"kind"`
}

// ListProjectBadgesOptions represents the available ListProjectBadges()
// options.
//
// GitLab API docs:
// https://docs.gitlab.com/api/project_badges/#list-all-badges-of-a-project
type ListProjectBadgesOptions struct {
	ListOptions
	Name *string `url:"name,omitempty" json:"name,omitempty"`
}

// ListProjectBadges gets a list of a project's badges and its group badges.
//
// GitLab API docs:
// https://docs.gitlab.com/api/project_badges/#list-all-badges-of-a-project
func (s *ProjectBadgesService) ListProjectBadges(pid any, opt *ListProjectBadgesOptions, options ...RequestOptionFunc) ([]*ProjectBadge, *Response, error) {
	return do[[]*ProjectBadge](s.client,
		withPath("projects/%s/badges", ProjectID{pid}),
		withAPIOpts(opt),
		withRequestOpts(options...),
	)
}

// GetProjectBadge gets a project badge.
//
// GitLab API docs:
// https://docs.gitlab.com/api/project_badges/#get-a-badge-of-a-project
func (s *ProjectBadgesService) GetProjectBadge(pid any, badge int64, options ...RequestOptionFunc) (*ProjectBadge, *Response, error) {
	return do[*ProjectBadge](s.client,
		withPath("projects/%s/badges/%d", ProjectID{pid}, badge),
		withRequestOpts(options...),
	)
}

// AddProjectBadgeOptions represents the available AddProjectBadge() options.
//
// GitLab API docs:
// https://docs.gitlab.com/api/project_badges/#add-a-badge-to-a-project
type AddProjectBadgeOptions struct {
	LinkURL  *string `url:"link_url,omitempty" json:"link_url,omitempty"`
	ImageURL *string `url:"image_url,omitempty" json:"image_url,omitempty"`
	Name     *string `url:"name,omitempty" json:"name,omitempty"`
}

// AddProjectBadge adds a badge to a project.
//
// GitLab API docs:
// https://docs.gitlab.com/api/project_badges/#add-a-badge-to-a-project
func (s *ProjectBadgesService) AddProjectBadge(pid any, opt *AddProjectBadgeOptions, options ...RequestOptionFunc) (*ProjectBadge, *Response, error) {
	return do[*ProjectBadge](s.client,
		withMethod(http.MethodPost),
		withPath("projects/%s/badges", ProjectID{pid}),
		withAPIOpts(opt),
		withRequestOpts(options...),
	)
}

// EditProjectBadgeOptions represents the available EditProjectBadge() options.
//
// GitLab API docs:
// https://docs.gitlab.com/api/project_badges/#edit-a-badge-of-a-project
type EditProjectBadgeOptions struct {
	LinkURL  *string `url:"link_url,omitempty" json:"link_url,omitempty"`
	ImageURL *string `url:"image_url,omitempty" json:"image_url,omitempty"`
	Name     *string `url:"name,omitempty" json:"name,omitempty"`
}

// EditProjectBadge updates a badge of a project.
//
// GitLab API docs:
// https://docs.gitlab.com/api/project_badges/#edit-a-badge-of-a-project
func (s *ProjectBadgesService) EditProjectBadge(pid any, badge int64, opt *EditProjectBadgeOptions, options ...RequestOptionFunc) (*ProjectBadge, *Response, error) {
	return do[*ProjectBadge](s.client,
		withMethod(http.MethodPut),
		withPath("projects/%s/badges/%d", ProjectID{pid}, badge),
		withAPIOpts(opt),
		withRequestOpts(options...),
	)
}

// DeleteProjectBadge removes a badge from a project. Only project's
// badges will be removed by using this endpoint.
//
// GitLab API docs:
// https://docs.gitlab.com/api/project_badges/#remove-a-badge-from-a-project
func (s *ProjectBadgesService) DeleteProjectBadge(pid any, badge int64, options ...RequestOptionFunc) (*Response, error) {
	_, resp, err := do[none](s.client,
		withMethod(http.MethodDelete),
		withPath("projects/%s/badges/%d", ProjectID{pid}, badge),
		withRequestOpts(options...),
	)
	return resp, err
}

// ProjectBadgePreviewOptions represents the available PreviewProjectBadge() options.
//
// GitLab API docs:
// https://docs.gitlab.com/api/project_badges/#preview-a-badge-from-a-project
type ProjectBadgePreviewOptions struct {
	LinkURL  *string `url:"link_url,omitempty" json:"link_url,omitempty"`
	ImageURL *string `url:"image_url,omitempty" json:"image_url,omitempty"`
}

// PreviewProjectBadge returns how the link_url and image_url final URLs would be after
// resolving the placeholder interpolation.
//
// GitLab API docs:
// https://docs.gitlab.com/api/project_badges/#preview-a-badge-from-a-project
func (s *ProjectBadgesService) PreviewProjectBadge(pid any, opt *ProjectBadgePreviewOptions, options ...RequestOptionFunc) (*ProjectBadge, *Response, error) {
	return do[*ProjectBadge](s.client,
		withPath("projects/%s/badges/render", ProjectID{pid}),
		withAPIOpts(opt),
		withRequestOpts(options...),
	)
}
