// Copyright 2026 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package gmailpostmastertools provides access to the Gmail Postmaster Tools API.
//
// For product documentation, see: https://developers.google.com/workspace/gmail/postmaster
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/gmailpostmastertools/v2"
//	...
//	ctx := context.Background()
//	gmailpostmastertoolsService, err := gmailpostmastertools.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate.
// To restrict scopes, use [google.golang.org/api/option.WithScopes]:
//
//	gmailpostmastertoolsService, err := gmailpostmastertools.NewService(ctx, option.WithScopes(gmailpostmastertools.PostmasterTrafficReadonlyScope))
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	gmailpostmastertoolsService, err := gmailpostmastertools.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	gmailpostmastertoolsService, err := gmailpostmastertools.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package gmailpostmastertools // import "google.golang.org/api/gmailpostmastertools/v2"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "gmailpostmastertools:v2"
const apiName = "gmailpostmastertools"
const apiVersion = "v2"
const basePath = "https://gmailpostmastertools.googleapis.com/"
const basePathTemplate = "https://gmailpostmastertools.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://gmailpostmastertools.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// Get email traffic metrics, manage domains, and manage domain users for the
	// domains you have registered with Postmaster Tools
	PostmasterScope = "https://www.googleapis.com/auth/postmaster"

	// View and manage the domains you have registered with Postmaster Tools
	PostmasterDomainScope = "https://www.googleapis.com/auth/postmaster.domain"

	// Get email traffic metrics for the domains you have registered with
	// Postmaster Tools
	PostmasterTrafficReadonlyScope = "https://www.googleapis.com/auth/postmaster.traffic.readonly"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/postmaster",
		"https://www.googleapis.com/auth/postmaster.domain",
		"https://www.googleapis.com/auth/postmaster.traffic.readonly",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.DomainStats = NewDomainStatsService(s)
	s.Domains = NewDomainsService(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	DomainStats *DomainStatsService

	Domains *DomainsService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewDomainStatsService(s *Service) *DomainStatsService {
	rs := &DomainStatsService{s: s}
	return rs
}

type DomainStatsService struct {
	s *Service
}

func NewDomainsService(s *Service) *DomainsService {
	rs := &DomainsService{s: s}
	rs.DomainStats = NewDomainsDomainStatsService(s)
	return rs
}

type DomainsService struct {
	s *Service

	DomainStats *DomainsDomainStatsService
}

func NewDomainsDomainStatsService(s *Service) *DomainsDomainStatsService {
	rs := &DomainsDomainStatsService{s: s}
	return rs
}

type DomainsDomainStatsService struct {
	s *Service
}

// BaseMetric: Specifies the base metric to query, which can be a predefined
// standard metric or a user-defined custom metric (if supported in the
// future).
type BaseMetric struct {
	// StandardMetric: A predefined standard metric.
	//
	// Possible values:
	//   "STANDARD_METRIC_UNSPECIFIED" - Unspecified standard metric. This value
	// should not be used directly.
	//   "FEEDBACK_LOOP_ID" - Predefined metric for Feedback Loop (FBL) id.
	//   "FEEDBACK_LOOP_SPAM_RATE" - Predefined metric for Feedback Loop (FBL) spam
	// rate. Filter must be of type feedback_loop_id = "" where is one valid
	// feedback loop ids.
	//   "SPAM_RATE" - Predefined metric for spam rate.
	//   "AUTH_SUCCESS_RATE" - The success rate of authentication mechanisms (DKIM,
	// SPF, DMARC). Filter must be of type auth_type = "" where is one of: [spf,
	// dkim, dmarc]
	//   "TLS_ENCRYPTION_MESSAGE_COUNT" - The rate of messages that were TLS
	// encrypted in transit Filter must be of type traffic_direction = "" where is
	// one of: [inbound, outbound]
	//   "TLS_ENCRYPTION_RATE" - The rate of messages that were TLS encrypted in
	// transit Filter must be of type traffic_direction = "" where is one of:
	// [inbound, outbound]
	//   "DELIVERY_ERROR_COUNT" - The total count of delivery errors encountered
	// (temporary or permanent rejects). The `filter` field supports a limited
	// syntax. Supported formats are: * Empty: No filter is applied. * `error_type`
	// = "" * `error_type` = "" AND `error_reason` = "" If an empty filter is
	// provided, the metric will be aggregated across all error types and reasons.
	// If only `error_type` is specified, the metric will be aggregated across all
	// reasons for that type. Supported values: * reject * temp_fail Supported
	// values depend on the : * For 'reject': [bad_attachment,
	// bad_or_missing_ptr_record, ip_in_rbls, low_domain_reputation,
	// low_ip_reputation, spammy_content, stamp_policy_error, other] * For
	// 'temp_fail': [anomalous_traffic_pattern, other]
	//   "DELIVERY_ERROR_RATE" - Delivery error rate for the specified delivery
	// error type. The `filter` field supports a limited syntax. Supported formats
	// are: * Empty: No filter is applied. * `error_type` = "" * `error_type` = ""
	// AND `error_reason` = "" If an empty filter is provided, the metric will be
	// aggregated across all error types and reasons. If only `error_type` is
	// specified, the metric will be aggregated across all reasons for that type.
	// Supported values: * reject * temp_fail Supported values depend on the : *
	// For 'reject': [bad_attachment, bad_or_missing_ptr_record, ip_in_rbls,
	// low_domain_reputation, low_ip_reputation, spammy_content,
	// stamp_policy_error, other] * For 'temp_fail': [anomalous_traffic_pattern,
	// other]
	StandardMetric string `json:"standardMetric,omitempty"`
	// ForceSendFields is a list of field names (e.g. "StandardMetric") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "StandardMetric") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s BaseMetric) MarshalJSON() ([]byte, error) {
	type NoMethod BaseMetric
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// BatchQueryDomainStatsRequest: Request message for BatchQueryDomainStats.
type BatchQueryDomainStatsRequest struct {
	// Requests: Required. A list of individual query requests. Each request can be
	// for a different domain. A maximum of 100 requests can be included in a
	// single batch.
	Requests []*QueryDomainStatsRequest `json:"requests,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Requests") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Requests") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s BatchQueryDomainStatsRequest) MarshalJSON() ([]byte, error) {
	type NoMethod BatchQueryDomainStatsRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// BatchQueryDomainStatsResponse: Response message for BatchQueryDomainStats.
type BatchQueryDomainStatsResponse struct {
	// Results: A list of responses, one for each query in the
	// BatchQueryDomainStatsRequest. The order of responses will correspond to the
	// order of requests.
	Results []*BatchQueryDomainStatsResult `json:"results,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Results") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Results") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s BatchQueryDomainStatsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod BatchQueryDomainStatsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// BatchQueryDomainStatsResult: Represents the result of a single
// QueryDomainStatsRequest within a batch.
type BatchQueryDomainStatsResult struct {
	// Error: The error status if the individual query failed.
	Error *Status `json:"error,omitempty"`
	// Response: The successful response for the individual query.
	Response *QueryDomainStatsResponse `json:"response,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Error") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Error") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s BatchQueryDomainStatsResult) MarshalJSON() ([]byte, error) {
	type NoMethod BatchQueryDomainStatsResult
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ComplianceRowData: Data for a single row of the compliance status table.
type ComplianceRowData struct {
	// Requirement: The compliance requirement.
	//
	// Possible values:
	//   "COMPLIANCE_REQUIREMENT_UNSPECIFIED" - Unspecified.
	//   "SPF" - Whether the sender has properly configured SPF.
	//   "DKIM" - Whether the sender has properly configured DKIM.
	//   "SPF_AND_DKIM" - Whether the sender has properly configured both SPF and
	// DKIM.
	//   "DMARC_POLICY" - Whether the sender has configured DMARC policy.
	//   "DMARC_ALIGNMENT" - Whether the From: header is aligned with DKIM or SPF
	//   "MESSAGE_FORMATTING" - Whether messages are correctly formatted according
	// to RFC 5322.
	//   "DNS_RECORDS" - Whether the domain has forward and reverse DNS records.
	//   "ENCRYPTION" - Whether messages has TLS encryption.
	//   "USER_REPORTED_SPAM_RATE" - Whether the sender is below a threshold for
	// user-reported spam rate.
	//   "ONE_CLICK_UNSUBSCRIBE" - Whether the sender sufficiently supports
	// one-click unsubscribe. Note that the user-facing requirement is "one-click
	// unsubscribe", but we require satisfaction of multiple "unsubscribe support"
	// rules.
	//   "HONOR_UNSUBSCRIBE" - Whether the sender honors user-initiated unsubscribe
	// requests.
	Requirement string `json:"requirement,omitempty"`
	// Status: The compliance status for the requirement.
	Status *ComplianceStatus `json:"status,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Requirement") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Requirement") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ComplianceRowData) MarshalJSON() ([]byte, error) {
	type NoMethod ComplianceRowData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ComplianceStatus: The status of a sender compliance requirement.
type ComplianceStatus struct {
	// Status: Output only. The compliance status.
	//
	// Possible values:
	//   "STATE_UNSPECIFIED" - Unspecified.
	//   "COMPLIANT" - The compliance requirement is met, and the sender is deemed
	// compliant.
	//   "NEEDS_WORK" - The compliance requirement is unmet, and the sender needs
	// to do work to achieve compliance.
	Status string `json:"status,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Status") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Status") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ComplianceStatus) MarshalJSON() ([]byte, error) {
	type NoMethod ComplianceStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Date: Represents a whole or partial calendar date, such as a birthday. The
// time of day and time zone are either specified elsewhere or are
// insignificant. The date is relative to the Gregorian Calendar. This can
// represent one of the following: * A full date, with non-zero year, month,
// and day values. * A month and day, with a zero year (for example, an
// anniversary). * A year on its own, with a zero month and a zero day. * A
// year and month, with a zero day (for example, a credit card expiration
// date). Related types: * google.type.TimeOfDay * google.type.DateTime *
// google.protobuf.Timestamp
type Date struct {
	// Day: Day of a month. Must be from 1 to 31 and valid for the year and month,
	// or 0 to specify a year by itself or a year and month where the day isn't
	// significant.
	Day int64 `json:"day,omitempty"`
	// Month: Month of a year. Must be from 1 to 12, or 0 to specify a year without
	// a month and day.
	Month int64 `json:"month,omitempty"`
	// Year: Year of the date. Must be from 1 to 9999, or 0 to specify a date
	// without a year.
	Year int64 `json:"year,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Day") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Day") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Date) MarshalJSON() ([]byte, error) {
	type NoMethod Date
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DateList: A set of specific dates.
type DateList struct {
	// Dates: Required. The list of specific dates for which to retrieve data.
	Dates []*Date `json:"dates,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Dates") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Dates") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DateList) MarshalJSON() ([]byte, error) {
	type NoMethod DateList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DateRange: A single date range defined by a start and end date.
type DateRange struct {
	// End: Required. The inclusive end date of the date range.
	End *Date `json:"end,omitempty"`
	// Start: Required. The inclusive start date of the date range.
	Start *Date `json:"start,omitempty"`
	// ForceSendFields is a list of field names (e.g. "End") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "End") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DateRange) MarshalJSON() ([]byte, error) {
	type NoMethod DateRange
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DateRanges: A set of date ranges.
type DateRanges struct {
	// DateRanges: Required. The list of date ranges for which to retrieve data.
	DateRanges []*DateRange `json:"dateRanges,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DateRanges") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DateRanges") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DateRanges) MarshalJSON() ([]byte, error) {
	type NoMethod DateRanges
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Domain: Information about a domain registered by the user.
type Domain struct {
	// CreateTime: Output only. Immutable. The timestamp at which the domain was
	// added to the user's account.
	CreateTime string `json:"createTime,omitempty"`
	// LastVerifyTime: The timestamp at which the domain was last verified by the
	// user.
	LastVerifyTime string `json:"lastVerifyTime,omitempty"`
	// Name: Identifier. The resource name of the domain. Format:
	// `domains/{domain_name}`, where domain_name is the fully qualified domain
	// name (i.e., mymail.mydomain.com).
	Name string `json:"name,omitempty"`
	// Permission: Output only. User's permission of this domain.
	//
	// Possible values:
	//   "PERMISSION_UNSPECIFIED" - Unspecified permission.
	//   "READER" - User has read access to the domain.
	//   "OWNER" - User has owner access to the domain.
	//   "NONE" - User has no access to the domain.
	Permission string `json:"permission,omitempty"`
	// VerificationState: Output only. Information about a user's verification
	// history and properties for the domain.
	//
	// Possible values:
	//   "VERIFICATION_STATE_UNSPECIFIED" - Unspecified.
	//   "UNVERIFIED" - The domain is unverified.
	//   "VERIFIED" - The domain is verified.
	VerificationState string `json:"verificationState,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CreateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Domain) MarshalJSON() ([]byte, error) {
	type NoMethod Domain
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DomainComplianceData: Compliance data for a given domain.
type DomainComplianceData struct {
	// DomainId: Domain that this data is for.
	DomainId string `json:"domainId,omitempty"`
	// HonorUnsubscribeVerdict: Unsubscribe honoring compliance verdict.
	HonorUnsubscribeVerdict *HonorUnsubscribeVerdict `json:"honorUnsubscribeVerdict,omitempty"`
	// OneClickUnsubscribeVerdict: One-click unsubscribe compliance verdict.
	OneClickUnsubscribeVerdict *OneClickUnsubscribeVerdict `json:"oneClickUnsubscribeVerdict,omitempty"`
	// RowData: Data for each of the rows of the table. Each message contains all
	// the data that backs a single row.
	RowData []*ComplianceRowData `json:"rowData,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DomainId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DomainId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DomainComplianceData) MarshalJSON() ([]byte, error) {
	type NoMethod DomainComplianceData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DomainComplianceStatus: Compliance status for a domain.
type DomainComplianceStatus struct {
	// ComplianceData: Compliance data for the registrable domain part of the
	// domain in `name`. For example, if `name` is
	// `domains/example.com/complianceStatus`, this field contains compliance data
	// for `example.com`.
	ComplianceData *DomainComplianceData `json:"complianceData,omitempty"`
	// Name: Identifier. The resource name of the domain's compliance status.
	// Format: `domains/{domain_id}/complianceStatus`.
	Name string `json:"name,omitempty"`
	// SubdomainComplianceData: Compliance data calculated specifically for the
	// subdomain in `name`. This field is only populated if the domain in `name` is
	// a subdomain that differs from its registrable domain (e.g.,
	// `sub.example.com`), and if compliance data is available for that specific
	// subdomain.
	SubdomainComplianceData *DomainComplianceData `json:"subdomainComplianceData,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ComplianceData") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ComplianceData") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DomainComplianceStatus) MarshalJSON() ([]byte, error) {
	type NoMethod DomainComplianceStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DomainStat: Email statistics for a domain for a specified time period or
// date.
type DomainStat struct {
	// Date: Optional. The specific date for these stats, if granularity is DAILY.
	// This field is populated if the QueryDomainStatsRequest specified a DAILY
	// aggregation granularity.
	Date *Date `json:"date,omitempty"`
	// Metric: The user-defined name from MetricDefinition.name in the request,
	// used to correlate this result with the requested metric.
	Metric string `json:"metric,omitempty"`
	// Name: Output only. The resource name of the DomainStat resource. Format:
	// domains/{domain}/domainStats/{domain_stat} The `{domain_stat}` segment is an
	// opaque, server-generated ID. We recommend using the `metric` field to
	// identify queried metrics instead of parsing the name.
	Name string `json:"name,omitempty"`
	// Value: The value of the corresponding metric.
	Value *StatisticValue `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Date") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Date") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DomainStat) MarshalJSON() ([]byte, error) {
	type NoMethod DomainStat
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// HonorUnsubscribeVerdict: Compliance verdict for whether a sender meets the
// unsubscribe honoring compliance requirement.
type HonorUnsubscribeVerdict struct {
	// Reason: The specific reason for the compliance verdict. Must be empty if the
	// status is compliant.
	//
	// Possible values:
	//   "REASON_UNSPECIFIED" - Unspecified.
	//   "NOT_HONORING" - The sender does not honor unsubscribe requests.
	//   "NOT_HONORING_TOO_FEW_CAMPAIGNS" - The sender does not honor unsubscribe
	// requests and consider to increase the number of relevant campaigns.
	//   "NOT_HONORING_TOO_MANY_CAMPAIGNS" - The sender does not honor unsubscribe
	// requests and consider to reduce the number of relevant campaigns.
	Reason string `json:"reason,omitempty"`
	// Status: The compliance status.
	Status *ComplianceStatus `json:"status,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Reason") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Reason") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s HonorUnsubscribeVerdict) MarshalJSON() ([]byte, error) {
	type NoMethod HonorUnsubscribeVerdict
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListDomainsResponse: Response message for ListDomains.
type ListDomainsResponse struct {
	// Domains: The domains that have been registered by the user.
	Domains []*Domain `json:"domains,omitempty"`
	// NextPageToken: Token to retrieve the next page of results, or empty if there
	// are no more results in the list.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Domains") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Domains") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListDomainsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListDomainsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MetricDefinition: Defines a specific metric to query, including a
// user-defined name, the base metric type, and optional filters.
type MetricDefinition struct {
	// BaseMetric: Required. The underlying metric to query.
	BaseMetric *BaseMetric `json:"baseMetric,omitempty"`
	// Filter: Optional. Optional filters to apply to the metric.
	Filter string `json:"filter,omitempty"`
	// Name: Required. The user-defined name for this metric. This name will be
	// used as the key for this metric's value in the response.
	Name string `json:"name,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BaseMetric") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BaseMetric") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MetricDefinition) MarshalJSON() ([]byte, error) {
	type NoMethod MetricDefinition
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// OneClickUnsubscribeVerdict: Compliance verdict for whether a sender meets
// the one-click unsubscribe compliance requirement.
type OneClickUnsubscribeVerdict struct {
	// Reason: The specific reason for the compliance verdict. Must be empty if the
	// status is compliant.
	//
	// Possible values:
	//   "REASON_UNSPECIFIED" - Unspecified.
	//   "NO_UNSUB_GENERAL" - Sender does not support one-click unsubscribe for the
	// majority of their messages.
	//   "NO_UNSUB_SPAM_REPORTS" - Sender does not support one-click unsubscribe
	// for most messages that are manually reported as spam.
	//   "NO_UNSUB_PROMO_SPAM_REPORTS" - Sender does not support one-click
	// unsubscribe for most promotional messages that are manually reported as
	// spam. This classification of messages is a subset of those encompassed by
	// `NO_UNSUB_SPAM_REPORTS`.
	Reason string `json:"reason,omitempty"`
	// Status: The compliance status.
	Status *ComplianceStatus `json:"status,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Reason") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Reason") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s OneClickUnsubscribeVerdict) MarshalJSON() ([]byte, error) {
	type NoMethod OneClickUnsubscribeVerdict
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// QueryDomainStatsRequest: Request message for QueryDomainStats.
type QueryDomainStatsRequest struct {
	// AggregationGranularity: Optional. The granularity at which to aggregate the
	// statistics. If unspecified, defaults to DAILY.
	//
	// Possible values:
	//   "AGGREGATION_GRANULARITY_UNSPECIFIED" - Unspecified granularity. Defaults
	// to DAILY.
	//   "DAILY" - Statistics are aggregated on a daily basis. Each DomainStats
	// entry in the response will correspond to a single day.
	//   "OVERALL" - Statistics are aggregated over the entire requested time
	// period. Each DomainStats entry in the response will represent the total for
	// the period.
	AggregationGranularity string `json:"aggregationGranularity,omitempty"`
	// MetricDefinitions: Required. The specific metrics to query. You can define a
	// custom name for each metric, which will be used in the response.
	MetricDefinitions []*MetricDefinition `json:"metricDefinitions,omitempty"`
	// PageSize: Optional. The maximum number of DomainStats resources to return in
	// the response. The server may return fewer than this value. If unspecified, a
	// default value of 10 will be used. The maximum value is 200.
	PageSize int64 `json:"pageSize,omitempty"`
	// PageToken: Optional. The next_page_token value returned from a previous List
	// request, if any. If the aggregation granularity is DAILY, the page token
	// will be the encoded date + "/" + metric name. If the aggregation granularity
	// is OVERALL, the page token will be the encoded metric name.
	PageToken string `json:"pageToken,omitempty"`
	// Parent: Required. The parent resource name where the stats are queried.
	// Format: domains/{domain}
	Parent string `json:"parent,omitempty"`
	// TimeQuery: Required. The time range or specific dates for which to retrieve
	// the metrics.
	TimeQuery *TimeQuery `json:"timeQuery,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AggregationGranularity") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AggregationGranularity") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s QueryDomainStatsRequest) MarshalJSON() ([]byte, error) {
	type NoMethod QueryDomainStatsRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// QueryDomainStatsResponse: Response message for QueryDomainStats.
type QueryDomainStatsResponse struct {
	// DomainStats: The list of domain statistics. Each DomainStat object contains
	// the value for a metric requested in the QueryDomainStatsRequest.
	DomainStats []*DomainStat `json:"domainStats,omitempty"`
	// NextPageToken: Token to retrieve the next page of results, or empty if there
	// are no more results in the list.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "DomainStats") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DomainStats") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s QueryDomainStatsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod QueryDomainStatsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// StatisticValue: The actual value of a statistic.
type StatisticValue struct {
	// DoubleValue: Double value.
	DoubleValue float64 `json:"doubleValue,omitempty"`
	// FloatValue: Float value.
	FloatValue float64 `json:"floatValue,omitempty"`
	// IntValue: Integer value.
	IntValue int64 `json:"intValue,omitempty,string"`
	// StringList: List of string values.
	StringList *StringList `json:"stringList,omitempty"`
	// StringValue: String value.
	StringValue string `json:"stringValue,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DoubleValue") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DoubleValue") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s StatisticValue) MarshalJSON() ([]byte, error) {
	type NoMethod StatisticValue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *StatisticValue) UnmarshalJSON(data []byte) error {
	type NoMethod StatisticValue
	var s1 struct {
		DoubleValue gensupport.JSONFloat64 `json:"doubleValue"`
		FloatValue  gensupport.JSONFloat64 `json:"floatValue"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.DoubleValue = float64(s1.DoubleValue)
	s.FloatValue = float64(s1.FloatValue)
	return nil
}

// Status: The `Status` type defines a logical error model that is suitable for
// different programming environments, including REST APIs and RPC APIs. It is
// used by gRPC (https://github.com/grpc). Each `Status` message contains three
// pieces of data: error code, error message, and error details. You can find
// out more about this error model and how to work with it in the API Design
// Guide (https://cloud.google.com/apis/design/errors).
type Status struct {
	// Code: The status code, which should be an enum value of google.rpc.Code.
	Code int64 `json:"code,omitempty"`
	// Details: A list of messages that carry the error details. There is a common
	// set of message types for APIs to use.
	Details []googleapi.RawMessage `json:"details,omitempty"`
	// Message: A developer-facing error message, which should be in English. Any
	// user-facing error message should be localized and sent in the
	// google.rpc.Status.details field, or localized by the client.
	Message string `json:"message,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Code") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Code") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Status) MarshalJSON() ([]byte, error) {
	type NoMethod Status
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// StringList: Represents a list of strings.
type StringList struct {
	// Values: The string values.
	Values []string `json:"values,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Values") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Values") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s StringList) MarshalJSON() ([]byte, error) {
	type NoMethod StringList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TimeQuery: The date ranges or specific dates for which you want to retrieve
// data.
type TimeQuery struct {
	// DateList: A list of specific dates.
	DateList *DateList `json:"dateList,omitempty"`
	// DateRanges: A list of date ranges.
	DateRanges *DateRanges `json:"dateRanges,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DateList") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DateList") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TimeQuery) MarshalJSON() ([]byte, error) {
	type NoMethod TimeQuery
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type DomainStatsBatchQueryCall struct {
	s                            *Service
	batchquerydomainstatsrequest *BatchQueryDomainStatsRequest
	urlParams_                   gensupport.URLParams
	ctx_                         context.Context
	header_                      http.Header
}

// BatchQuery: Executes a batch of QueryDomainStats requests for multiple
// domains. Returns PERMISSION_DENIED if you don't have permission to access
// DomainStats for any of the requested domains.
func (r *DomainStatsService) BatchQuery(batchquerydomainstatsrequest *BatchQueryDomainStatsRequest) *DomainStatsBatchQueryCall {
	c := &DomainStatsBatchQueryCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.batchquerydomainstatsrequest = batchquerydomainstatsrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DomainStatsBatchQueryCall) Fields(s ...googleapi.Field) *DomainStatsBatchQueryCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DomainStatsBatchQueryCall) Context(ctx context.Context) *DomainStatsBatchQueryCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DomainStatsBatchQueryCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DomainStatsBatchQueryCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.batchquerydomainstatsrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/domainStats:batchQuery")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "gmailpostmastertools.domainStats.batchQuery", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "gmailpostmastertools.domainStats.batchQuery" call.
// Any non-2xx status code is an error. Response headers are in either
// *BatchQueryDomainStatsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *DomainStatsBatchQueryCall) Do(opts ...googleapi.CallOption) (*BatchQueryDomainStatsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &BatchQueryDomainStatsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "gmailpostmastertools.domainStats.batchQuery", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type DomainsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves detailed information about a domain registered by you.
// Returns NOT_FOUND if the domain is not registered by you. Domain represents
// the metadata of a domain that has been registered within the system and
// linked to a user.
//
//   - name: The resource name of the domain. Format: `domains/{domain_name}`,
//     where domain_name is the fully qualified domain name (i.e.,
//     mymail.mydomain.com).
func (r *DomainsService) Get(name string) *DomainsGetCall {
	c := &DomainsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DomainsGetCall) Fields(s ...googleapi.Field) *DomainsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *DomainsGetCall) IfNoneMatch(entityTag string) *DomainsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DomainsGetCall) Context(ctx context.Context) *DomainsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DomainsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DomainsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "gmailpostmastertools.domains.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "gmailpostmastertools.domains.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Domain.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *DomainsGetCall) Do(opts ...googleapi.CallOption) (*Domain, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Domain{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "gmailpostmastertools.domains.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type DomainsGetComplianceStatusCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetComplianceStatus: Retrieves the compliance status for a given domain.
// Returns PERMISSION_DENIED if you don't have permission to access compliance
// status for the domain.
//
//   - name: The resource name of the domain's compliance status to retrieve.
//     Format: `domains/{domain_id}/complianceStatus`.
func (r *DomainsService) GetComplianceStatus(name string) *DomainsGetComplianceStatusCall {
	c := &DomainsGetComplianceStatusCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DomainsGetComplianceStatusCall) Fields(s ...googleapi.Field) *DomainsGetComplianceStatusCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *DomainsGetComplianceStatusCall) IfNoneMatch(entityTag string) *DomainsGetComplianceStatusCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DomainsGetComplianceStatusCall) Context(ctx context.Context) *DomainsGetComplianceStatusCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DomainsGetComplianceStatusCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DomainsGetComplianceStatusCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "gmailpostmastertools.domains.getComplianceStatus", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "gmailpostmastertools.domains.getComplianceStatus" call.
// Any non-2xx status code is an error. Response headers are in either
// *DomainComplianceStatus.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *DomainsGetComplianceStatusCall) Do(opts ...googleapi.CallOption) (*DomainComplianceStatus, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &DomainComplianceStatus{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "gmailpostmastertools.domains.getComplianceStatus", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type DomainsListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Retrieves a list of all domains registered by you, along with their
// corresponding metadata. The order of domains in the response is unspecified
// and non-deterministic. Newly registered domains will not necessarily be
// added to the end of this list.
func (r *DomainsService) List() *DomainsListCall {
	c := &DomainsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// PageSize sets the optional parameter "pageSize": Requested page size. Server
// may return fewer domains than requested. If unspecified, the default value
// for this field is 10. The maximum value for this field is 200.
func (c *DomainsListCall) PageSize(pageSize int64) *DomainsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": The next_page_token value
// returned from a previous List request, if any.
func (c *DomainsListCall) PageToken(pageToken string) *DomainsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DomainsListCall) Fields(s ...googleapi.Field) *DomainsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *DomainsListCall) IfNoneMatch(entityTag string) *DomainsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DomainsListCall) Context(ctx context.Context) *DomainsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DomainsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DomainsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/domains")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "gmailpostmastertools.domains.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "gmailpostmastertools.domains.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListDomainsResponse.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *DomainsListCall) Do(opts ...googleapi.CallOption) (*ListDomainsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListDomainsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "gmailpostmastertools.domains.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *DomainsListCall) Pages(ctx context.Context, f func(*ListDomainsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type DomainsDomainStatsQueryCall struct {
	s                       *Service
	parent                  string
	querydomainstatsrequest *QueryDomainStatsRequest
	urlParams_              gensupport.URLParams
	ctx_                    context.Context
	header_                 http.Header
}

// Query: Retrieves a list of domain statistics for a given domain and time
// period. Returns statistics only for dates where data is available. Returns
// PERMISSION_DENIED if you don't have permission to access DomainStats for the
// domain.
//
//   - parent: The parent resource name where the stats are queried. Format:
//     domains/{domain}.
func (r *DomainsDomainStatsService) Query(parent string, querydomainstatsrequest *QueryDomainStatsRequest) *DomainsDomainStatsQueryCall {
	c := &DomainsDomainStatsQueryCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.querydomainstatsrequest = querydomainstatsrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DomainsDomainStatsQueryCall) Fields(s ...googleapi.Field) *DomainsDomainStatsQueryCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DomainsDomainStatsQueryCall) Context(ctx context.Context) *DomainsDomainStatsQueryCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DomainsDomainStatsQueryCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DomainsDomainStatsQueryCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.querydomainstatsrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/domainStats:query")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "gmailpostmastertools.domains.domainStats.query", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "gmailpostmastertools.domains.domainStats.query" call.
// Any non-2xx status code is an error. Response headers are in either
// *QueryDomainStatsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *DomainsDomainStatsQueryCall) Do(opts ...googleapi.CallOption) (*QueryDomainStatsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &QueryDomainStatsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "gmailpostmastertools.domains.domainStats.query", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *DomainsDomainStatsQueryCall) Pages(ctx context.Context, f func(*QueryDomainStatsResponse) error) error {
	c.ctx_ = ctx
	defer func(pt string) { c.querydomainstatsrequest.PageToken = pt }(c.querydomainstatsrequest.PageToken)
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.querydomainstatsrequest.PageToken = x.NextPageToken
	}
}
