// Package debugger provides the Chrome DevTools Protocol
// commands, types, and events for the Debugger domain.
//
// Debugger domain exposes JavaScript debugging capabilities. It allows
// setting and removing breakpoints, stepping through execution, exploring stack
// traces, etc.
//
// Generated by the cdproto-gen command.
package debugger

// Code generated by cdproto-gen. DO NOT EDIT.

import (
	"context"
	"encoding/base64"

	"github.com/chromedp/cdproto/cdp"
	"github.com/chromedp/cdproto/runtime"
)

// ContinueToLocationParams continues execution until specific location is
// reached.
type ContinueToLocationParams struct {
	Location         *Location                          `json:"location"` // Location to continue to.
	TargetCallFrames ContinueToLocationTargetCallFrames `json:"targetCallFrames,omitempty,omitzero"`
}

// ContinueToLocation continues execution until specific location is reached.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-continueToLocation
//
// parameters:
//
//	location - Location to continue to.
func ContinueToLocation(location *Location) *ContinueToLocationParams {
	return &ContinueToLocationParams{
		Location: location,
	}
}

// WithTargetCallFrames [no description].
func (p ContinueToLocationParams) WithTargetCallFrames(targetCallFrames ContinueToLocationTargetCallFrames) *ContinueToLocationParams {
	p.TargetCallFrames = targetCallFrames
	return &p
}

// Do executes Debugger.continueToLocation against the provided context.
func (p *ContinueToLocationParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandContinueToLocation, p, nil)
}

// DisableParams disables debugger for given page.
type DisableParams struct{}

// Disable disables debugger for given page.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-disable
func Disable() *DisableParams {
	return &DisableParams{}
}

// Do executes Debugger.disable against the provided context.
func (p *DisableParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandDisable, nil, nil)
}

// EnableParams enables debugger for the given page. Clients should not
// assume that the debugging has been enabled until the result for this command
// is received.
type EnableParams struct {
	MaxScriptsCacheSize float64 `json:"maxScriptsCacheSize,omitempty,omitzero"` // The maximum size in bytes of collected scripts (not referenced by other heap objects) the debugger can hold. Puts no limit if parameter is omitted.
}

// Enable enables debugger for the given page. Clients should not assume that
// the debugging has been enabled until the result for this command is received.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-enable
//
// parameters:
func Enable() *EnableParams {
	return &EnableParams{}
}

// WithMaxScriptsCacheSize the maximum size in bytes of collected scripts
// (not referenced by other heap objects) the debugger can hold. Puts no limit
// if parameter is omitted.
func (p EnableParams) WithMaxScriptsCacheSize(maxScriptsCacheSize float64) *EnableParams {
	p.MaxScriptsCacheSize = maxScriptsCacheSize
	return &p
}

// EnableReturns return values.
type EnableReturns struct {
	DebuggerID runtime.UniqueDebuggerID `json:"debuggerId,omitempty,omitzero"` // Unique identifier of the debugger.
}

// Do executes Debugger.enable against the provided context.
//
// returns:
//
//	debuggerID - Unique identifier of the debugger.
func (p *EnableParams) Do(ctx context.Context) (debuggerID runtime.UniqueDebuggerID, err error) {
	// execute
	var res EnableReturns
	err = cdp.Execute(ctx, CommandEnable, p, &res)
	if err != nil {
		return "", err
	}

	return res.DebuggerID, nil
}

// EvaluateOnCallFrameParams evaluates expression on a given call frame.
type EvaluateOnCallFrameParams struct {
	CallFrameID           CallFrameID       `json:"callFrameId"`                    // Call frame identifier to evaluate on.
	Expression            string            `json:"expression"`                     // Expression to evaluate.
	ObjectGroup           string            `json:"objectGroup,omitempty,omitzero"` // String object group name to put result into (allows rapid releasing resulting object handles using releaseObjectGroup).
	IncludeCommandLineAPI bool              `json:"includeCommandLineAPI"`          // Specifies whether command line API should be available to the evaluated expression, defaults to false.
	Silent                bool              `json:"silent"`                         // In silent mode exceptions thrown during evaluation are not reported and do not pause execution. Overrides setPauseOnException state.
	ReturnByValue         bool              `json:"returnByValue"`                  // Whether the result is expected to be a JSON object that should be sent by value.
	GeneratePreview       bool              `json:"generatePreview"`                // Whether preview should be generated for the result.
	ThrowOnSideEffect     bool              `json:"throwOnSideEffect"`              // Whether to throw an exception if side effect cannot be ruled out during evaluation.
	Timeout               runtime.TimeDelta `json:"timeout,omitempty,omitzero"`     // Terminate execution after timing out (number of milliseconds).
}

// EvaluateOnCallFrame evaluates expression on a given call frame.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-evaluateOnCallFrame
//
// parameters:
//
//	callFrameID - Call frame identifier to evaluate on.
//	expression - Expression to evaluate.
func EvaluateOnCallFrame(callFrameID CallFrameID, expression string) *EvaluateOnCallFrameParams {
	return &EvaluateOnCallFrameParams{
		CallFrameID:           callFrameID,
		Expression:            expression,
		IncludeCommandLineAPI: false,
		Silent:                false,
		ReturnByValue:         false,
		GeneratePreview:       false,
		ThrowOnSideEffect:     false,
	}
}

// WithObjectGroup string object group name to put result into (allows rapid
// releasing resulting object handles using releaseObjectGroup).
func (p EvaluateOnCallFrameParams) WithObjectGroup(objectGroup string) *EvaluateOnCallFrameParams {
	p.ObjectGroup = objectGroup
	return &p
}

// WithIncludeCommandLineAPI specifies whether command line API should be
// available to the evaluated expression, defaults to false.
func (p EvaluateOnCallFrameParams) WithIncludeCommandLineAPI(includeCommandLineAPI bool) *EvaluateOnCallFrameParams {
	p.IncludeCommandLineAPI = includeCommandLineAPI
	return &p
}

// WithSilent in silent mode exceptions thrown during evaluation are not
// reported and do not pause execution. Overrides setPauseOnException state.
func (p EvaluateOnCallFrameParams) WithSilent(silent bool) *EvaluateOnCallFrameParams {
	p.Silent = silent
	return &p
}

// WithReturnByValue whether the result is expected to be a JSON object that
// should be sent by value.
func (p EvaluateOnCallFrameParams) WithReturnByValue(returnByValue bool) *EvaluateOnCallFrameParams {
	p.ReturnByValue = returnByValue
	return &p
}

// WithGeneratePreview whether preview should be generated for the result.
func (p EvaluateOnCallFrameParams) WithGeneratePreview(generatePreview bool) *EvaluateOnCallFrameParams {
	p.GeneratePreview = generatePreview
	return &p
}

// WithThrowOnSideEffect whether to throw an exception if side effect cannot
// be ruled out during evaluation.
func (p EvaluateOnCallFrameParams) WithThrowOnSideEffect(throwOnSideEffect bool) *EvaluateOnCallFrameParams {
	p.ThrowOnSideEffect = throwOnSideEffect
	return &p
}

// WithTimeout terminate execution after timing out (number of milliseconds).
func (p EvaluateOnCallFrameParams) WithTimeout(timeout runtime.TimeDelta) *EvaluateOnCallFrameParams {
	p.Timeout = timeout
	return &p
}

// EvaluateOnCallFrameReturns return values.
type EvaluateOnCallFrameReturns struct {
	Result           *runtime.RemoteObject     `json:"result,omitempty,omitzero"`           // Object wrapper for the evaluation result.
	ExceptionDetails *runtime.ExceptionDetails `json:"exceptionDetails,omitempty,omitzero"` // Exception details.
}

// Do executes Debugger.evaluateOnCallFrame against the provided context.
//
// returns:
//
//	result - Object wrapper for the evaluation result.
//	exceptionDetails - Exception details.
func (p *EvaluateOnCallFrameParams) Do(ctx context.Context) (result *runtime.RemoteObject, exceptionDetails *runtime.ExceptionDetails, err error) {
	// execute
	var res EvaluateOnCallFrameReturns
	err = cdp.Execute(ctx, CommandEvaluateOnCallFrame, p, &res)
	if err != nil {
		return nil, nil, err
	}

	return res.Result, res.ExceptionDetails, nil
}

// GetPossibleBreakpointsParams returns possible locations for breakpoint.
// scriptId in start and end range locations should be the same.
type GetPossibleBreakpointsParams struct {
	Start              *Location `json:"start"`                  // Start of range to search possible breakpoint locations in.
	End                *Location `json:"end,omitempty,omitzero"` // End of range to search possible breakpoint locations in (excluding). When not specified, end of scripts is used as end of range.
	RestrictToFunction bool      `json:"restrictToFunction"`     // Only consider locations which are in the same (non-nested) function as start.
}

// GetPossibleBreakpoints returns possible locations for breakpoint. scriptId
// in start and end range locations should be the same.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-getPossibleBreakpoints
//
// parameters:
//
//	start - Start of range to search possible breakpoint locations in.
func GetPossibleBreakpoints(start *Location) *GetPossibleBreakpointsParams {
	return &GetPossibleBreakpointsParams{
		Start:              start,
		RestrictToFunction: false,
	}
}

// WithEnd end of range to search possible breakpoint locations in
// (excluding). When not specified, end of scripts is used as end of range.
func (p GetPossibleBreakpointsParams) WithEnd(end *Location) *GetPossibleBreakpointsParams {
	p.End = end
	return &p
}

// WithRestrictToFunction only consider locations which are in the same
// (non-nested) function as start.
func (p GetPossibleBreakpointsParams) WithRestrictToFunction(restrictToFunction bool) *GetPossibleBreakpointsParams {
	p.RestrictToFunction = restrictToFunction
	return &p
}

// GetPossibleBreakpointsReturns return values.
type GetPossibleBreakpointsReturns struct {
	Locations []*BreakLocation `json:"locations,omitempty,omitzero"` // List of the possible breakpoint locations.
}

// Do executes Debugger.getPossibleBreakpoints against the provided context.
//
// returns:
//
//	locations - List of the possible breakpoint locations.
func (p *GetPossibleBreakpointsParams) Do(ctx context.Context) (locations []*BreakLocation, err error) {
	// execute
	var res GetPossibleBreakpointsReturns
	err = cdp.Execute(ctx, CommandGetPossibleBreakpoints, p, &res)
	if err != nil {
		return nil, err
	}

	return res.Locations, nil
}

// GetScriptSourceParams returns source for the script with given id.
type GetScriptSourceParams struct {
	ScriptID runtime.ScriptID `json:"scriptId"` // Id of the script to get source for.
}

// GetScriptSource returns source for the script with given id.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-getScriptSource
//
// parameters:
//
//	scriptID - Id of the script to get source for.
func GetScriptSource(scriptID runtime.ScriptID) *GetScriptSourceParams {
	return &GetScriptSourceParams{
		ScriptID: scriptID,
	}
}

// GetScriptSourceReturns return values.
type GetScriptSourceReturns struct {
	ScriptSource string `json:"scriptSource,omitempty,omitzero"` // Script source (empty in case of Wasm bytecode).
	Bytecode     string `json:"bytecode,omitempty,omitzero"`     // Wasm bytecode.
}

// Do executes Debugger.getScriptSource against the provided context.
//
// returns:
//
//	scriptSource - Script source (empty in case of Wasm bytecode).
//	bytecode - Wasm bytecode.
func (p *GetScriptSourceParams) Do(ctx context.Context) (scriptSource string, bytecode []byte, err error) {
	// execute
	var res GetScriptSourceReturns
	err = cdp.Execute(ctx, CommandGetScriptSource, p, &res)
	if err != nil {
		return "", nil, err
	}

	// decode
	var dec []byte
	dec, err = base64.StdEncoding.DecodeString(res.Bytecode)
	if err != nil {
		return "", nil, err
	}
	return res.ScriptSource, dec, nil
}

// DisassembleWasmModuleParams [no description].
type DisassembleWasmModuleParams struct {
	ScriptID runtime.ScriptID `json:"scriptId"` // Id of the script to disassemble
}

// DisassembleWasmModule [no description].
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-disassembleWasmModule
//
// parameters:
//
//	scriptID - Id of the script to disassemble
func DisassembleWasmModule(scriptID runtime.ScriptID) *DisassembleWasmModuleParams {
	return &DisassembleWasmModuleParams{
		ScriptID: scriptID,
	}
}

// DisassembleWasmModuleReturns return values.
type DisassembleWasmModuleReturns struct {
	StreamID            string                `json:"streamId,omitempty,omitzero"`            // For large modules, return a stream from which additional chunks of disassembly can be read successively.
	TotalNumberOfLines  int64                 `json:"totalNumberOfLines,omitempty,omitzero"`  // The total number of lines in the disassembly text.
	FunctionBodyOffsets []int64               `json:"functionBodyOffsets,omitempty,omitzero"` // The offsets of all function bodies, in the format [start1, end1, start2, end2, ...] where all ends are exclusive.
	Chunk               *WasmDisassemblyChunk `json:"chunk,omitempty,omitzero"`               // The first chunk of disassembly.
}

// Do executes Debugger.disassembleWasmModule against the provided context.
//
// returns:
//
//	streamID - For large modules, return a stream from which additional chunks of disassembly can be read successively.
//	totalNumberOfLines - The total number of lines in the disassembly text.
//	functionBodyOffsets - The offsets of all function bodies, in the format [start1, end1, start2, end2, ...] where all ends are exclusive.
//	chunk - The first chunk of disassembly.
func (p *DisassembleWasmModuleParams) Do(ctx context.Context) (streamID string, totalNumberOfLines int64, functionBodyOffsets []int64, chunk *WasmDisassemblyChunk, err error) {
	// execute
	var res DisassembleWasmModuleReturns
	err = cdp.Execute(ctx, CommandDisassembleWasmModule, p, &res)
	if err != nil {
		return "", 0, nil, nil, err
	}

	return res.StreamID, res.TotalNumberOfLines, res.FunctionBodyOffsets, res.Chunk, nil
}

// NextWasmDisassemblyChunkParams disassemble the next chunk of lines for the
// module corresponding to the stream. If disassembly is complete, this API will
// invalidate the streamId and return an empty chunk. Any subsequent calls for
// the now invalid stream will return errors.
type NextWasmDisassemblyChunkParams struct {
	StreamID string `json:"streamId"`
}

// NextWasmDisassemblyChunk disassemble the next chunk of lines for the
// module corresponding to the stream. If disassembly is complete, this API will
// invalidate the streamId and return an empty chunk. Any subsequent calls for
// the now invalid stream will return errors.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-nextWasmDisassemblyChunk
//
// parameters:
//
//	streamID
func NextWasmDisassemblyChunk(streamID string) *NextWasmDisassemblyChunkParams {
	return &NextWasmDisassemblyChunkParams{
		StreamID: streamID,
	}
}

// NextWasmDisassemblyChunkReturns return values.
type NextWasmDisassemblyChunkReturns struct {
	Chunk *WasmDisassemblyChunk `json:"chunk,omitempty,omitzero"` // The next chunk of disassembly.
}

// Do executes Debugger.nextWasmDisassemblyChunk against the provided context.
//
// returns:
//
//	chunk - The next chunk of disassembly.
func (p *NextWasmDisassemblyChunkParams) Do(ctx context.Context) (chunk *WasmDisassemblyChunk, err error) {
	// execute
	var res NextWasmDisassemblyChunkReturns
	err = cdp.Execute(ctx, CommandNextWasmDisassemblyChunk, p, &res)
	if err != nil {
		return nil, err
	}

	return res.Chunk, nil
}

// GetStackTraceParams returns stack trace with given stackTraceId.
type GetStackTraceParams struct {
	StackTraceID *runtime.StackTraceID `json:"stackTraceId"`
}

// GetStackTrace returns stack trace with given stackTraceId.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-getStackTrace
//
// parameters:
//
//	stackTraceID
func GetStackTrace(stackTraceID *runtime.StackTraceID) *GetStackTraceParams {
	return &GetStackTraceParams{
		StackTraceID: stackTraceID,
	}
}

// GetStackTraceReturns return values.
type GetStackTraceReturns struct {
	StackTrace *runtime.StackTrace `json:"stackTrace,omitempty,omitzero"`
}

// Do executes Debugger.getStackTrace against the provided context.
//
// returns:
//
//	stackTrace
func (p *GetStackTraceParams) Do(ctx context.Context) (stackTrace *runtime.StackTrace, err error) {
	// execute
	var res GetStackTraceReturns
	err = cdp.Execute(ctx, CommandGetStackTrace, p, &res)
	if err != nil {
		return nil, err
	}

	return res.StackTrace, nil
}

// PauseParams stops on the next JavaScript statement.
type PauseParams struct{}

// Pause stops on the next JavaScript statement.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-pause
func Pause() *PauseParams {
	return &PauseParams{}
}

// Do executes Debugger.pause against the provided context.
func (p *PauseParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandPause, nil, nil)
}

// RemoveBreakpointParams removes JavaScript breakpoint.
type RemoveBreakpointParams struct {
	BreakpointID BreakpointID `json:"breakpointId"`
}

// RemoveBreakpoint removes JavaScript breakpoint.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-removeBreakpoint
//
// parameters:
//
//	breakpointID
func RemoveBreakpoint(breakpointID BreakpointID) *RemoveBreakpointParams {
	return &RemoveBreakpointParams{
		BreakpointID: breakpointID,
	}
}

// Do executes Debugger.removeBreakpoint against the provided context.
func (p *RemoveBreakpointParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandRemoveBreakpoint, p, nil)
}

// RestartFrameParams restarts particular call frame from the beginning. The
// old, deprecated behavior of restartFrame is to stay paused and allow further
// CDP commands after a restart was scheduled. This can cause problems with
// restarting, so we now continue execution immediately after it has been
// scheduled until we reach the beginning of the restarted frame. To stay
// back-wards compatible, restartFrame now expects a mode parameter to be
// present. If the mode parameter is missing, restartFrame errors out. The
// various return values are deprecated and callFrames is always empty. Use the
// call frames from the Debugger#paused events instead, that fires once V8
// pauses at the beginning of the restarted function.
type RestartFrameParams struct {
	CallFrameID CallFrameID      `json:"callFrameId"`             // Call frame identifier to evaluate on.
	Mode        RestartFrameMode `json:"mode,omitempty,omitzero"` // The mode parameter must be present and set to 'StepInto', otherwise restartFrame will error out.
}

// RestartFrame restarts particular call frame from the beginning. The old,
// deprecated behavior of restartFrame is to stay paused and allow further CDP
// commands after a restart was scheduled. This can cause problems with
// restarting, so we now continue execution immediately after it has been
// scheduled until we reach the beginning of the restarted frame. To stay
// back-wards compatible, restartFrame now expects a mode parameter to be
// present. If the mode parameter is missing, restartFrame errors out. The
// various return values are deprecated and callFrames is always empty. Use the
// call frames from the Debugger#paused events instead, that fires once V8
// pauses at the beginning of the restarted function.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-restartFrame
//
// parameters:
//
//	callFrameID - Call frame identifier to evaluate on.
func RestartFrame(callFrameID CallFrameID) *RestartFrameParams {
	return &RestartFrameParams{
		CallFrameID: callFrameID,
	}
}

// WithMode the mode parameter must be present and set to 'StepInto',
// otherwise restartFrame will error out.
func (p RestartFrameParams) WithMode(mode RestartFrameMode) *RestartFrameParams {
	p.Mode = mode
	return &p
}

// Do executes Debugger.restartFrame against the provided context.
func (p *RestartFrameParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandRestartFrame, p, nil)
}

// ResumeParams resumes JavaScript execution.
type ResumeParams struct {
	TerminateOnResume bool `json:"terminateOnResume"` // Set to true to terminate execution upon resuming execution. In contrast to Runtime.terminateExecution, this will allows to execute further JavaScript (i.e. via evaluation) until execution of the paused code is actually resumed, at which point termination is triggered. If execution is currently not paused, this parameter has no effect.
}

// Resume resumes JavaScript execution.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-resume
//
// parameters:
func Resume() *ResumeParams {
	return &ResumeParams{
		TerminateOnResume: false,
	}
}

// WithTerminateOnResume set to true to terminate execution upon resuming
// execution. In contrast to Runtime.terminateExecution, this will allows to
// execute further JavaScript (i.e. via evaluation) until execution of the
// paused code is actually resumed, at which point termination is triggered. If
// execution is currently not paused, this parameter has no effect.
func (p ResumeParams) WithTerminateOnResume(terminateOnResume bool) *ResumeParams {
	p.TerminateOnResume = terminateOnResume
	return &p
}

// Do executes Debugger.resume against the provided context.
func (p *ResumeParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandResume, p, nil)
}

// SearchInContentParams searches for given string in script content.
type SearchInContentParams struct {
	ScriptID      runtime.ScriptID `json:"scriptId"`      // Id of the script to search in.
	Query         string           `json:"query"`         // String to search for.
	CaseSensitive bool             `json:"caseSensitive"` // If true, search is case sensitive.
	IsRegex       bool             `json:"isRegex"`       // If true, treats string parameter as regex.
}

// SearchInContent searches for given string in script content.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-searchInContent
//
// parameters:
//
//	scriptID - Id of the script to search in.
//	query - String to search for.
func SearchInContent(scriptID runtime.ScriptID, query string) *SearchInContentParams {
	return &SearchInContentParams{
		ScriptID:      scriptID,
		Query:         query,
		CaseSensitive: false,
		IsRegex:       false,
	}
}

// WithCaseSensitive if true, search is case sensitive.
func (p SearchInContentParams) WithCaseSensitive(caseSensitive bool) *SearchInContentParams {
	p.CaseSensitive = caseSensitive
	return &p
}

// WithIsRegex if true, treats string parameter as regex.
func (p SearchInContentParams) WithIsRegex(isRegex bool) *SearchInContentParams {
	p.IsRegex = isRegex
	return &p
}

// SearchInContentReturns return values.
type SearchInContentReturns struct {
	Result []*SearchMatch `json:"result,omitempty,omitzero"` // List of search matches.
}

// Do executes Debugger.searchInContent against the provided context.
//
// returns:
//
//	result - List of search matches.
func (p *SearchInContentParams) Do(ctx context.Context) (result []*SearchMatch, err error) {
	// execute
	var res SearchInContentReturns
	err = cdp.Execute(ctx, CommandSearchInContent, p, &res)
	if err != nil {
		return nil, err
	}

	return res.Result, nil
}

// SetAsyncCallStackDepthParams enables or disables async call stacks
// tracking.
type SetAsyncCallStackDepthParams struct {
	MaxDepth int64 `json:"maxDepth"` // Maximum depth of async call stacks. Setting to 0 will effectively disable collecting async call stacks (default).
}

// SetAsyncCallStackDepth enables or disables async call stacks tracking.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-setAsyncCallStackDepth
//
// parameters:
//
//	maxDepth - Maximum depth of async call stacks. Setting to 0 will effectively disable collecting async call stacks (default).
func SetAsyncCallStackDepth(maxDepth int64) *SetAsyncCallStackDepthParams {
	return &SetAsyncCallStackDepthParams{
		MaxDepth: maxDepth,
	}
}

// Do executes Debugger.setAsyncCallStackDepth against the provided context.
func (p *SetAsyncCallStackDepthParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandSetAsyncCallStackDepth, p, nil)
}

// SetBlackboxExecutionContextsParams replace previous blackbox execution
// contexts with passed ones. Forces backend to skip stepping/pausing in scripts
// in these execution contexts. VM will try to leave blackboxed script by
// performing 'step in' several times, finally resorting to 'step out' if
// unsuccessful.
type SetBlackboxExecutionContextsParams struct {
	UniqueIDs []string `json:"uniqueIds"` // Array of execution context unique ids for the debugger to ignore.
}

// SetBlackboxExecutionContexts replace previous blackbox execution contexts
// with passed ones. Forces backend to skip stepping/pausing in scripts in these
// execution contexts. VM will try to leave blackboxed script by performing
// 'step in' several times, finally resorting to 'step out' if unsuccessful.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-setBlackboxExecutionContexts
//
// parameters:
//
//	uniqueIDs - Array of execution context unique ids for the debugger to ignore.
func SetBlackboxExecutionContexts(uniqueIDs []string) *SetBlackboxExecutionContextsParams {
	return &SetBlackboxExecutionContextsParams{
		UniqueIDs: uniqueIDs,
	}
}

// Do executes Debugger.setBlackboxExecutionContexts against the provided context.
func (p *SetBlackboxExecutionContextsParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandSetBlackboxExecutionContexts, p, nil)
}

// SetBlackboxPatternsParams replace previous blackbox patterns with passed
// ones. Forces backend to skip stepping/pausing in scripts with url matching
// one of the patterns. VM will try to leave blackboxed script by performing
// 'step in' several times, finally resorting to 'step out' if unsuccessful.
type SetBlackboxPatternsParams struct {
	Patterns      []string `json:"patterns"`      // Array of regexps that will be used to check script url for blackbox state.
	SkipAnonymous bool     `json:"skipAnonymous"` // If true, also ignore scripts with no source url.
}

// SetBlackboxPatterns replace previous blackbox patterns with passed ones.
// Forces backend to skip stepping/pausing in scripts with url matching one of
// the patterns. VM will try to leave blackboxed script by performing 'step in'
// several times, finally resorting to 'step out' if unsuccessful.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-setBlackboxPatterns
//
// parameters:
//
//	patterns - Array of regexps that will be used to check script url for blackbox state.
func SetBlackboxPatterns(patterns []string) *SetBlackboxPatternsParams {
	return &SetBlackboxPatternsParams{
		Patterns:      patterns,
		SkipAnonymous: false,
	}
}

// WithSkipAnonymous if true, also ignore scripts with no source url.
func (p SetBlackboxPatternsParams) WithSkipAnonymous(skipAnonymous bool) *SetBlackboxPatternsParams {
	p.SkipAnonymous = skipAnonymous
	return &p
}

// Do executes Debugger.setBlackboxPatterns against the provided context.
func (p *SetBlackboxPatternsParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandSetBlackboxPatterns, p, nil)
}

// SetBlackboxedRangesParams makes backend skip steps in the script in
// blackboxed ranges. VM will try leave blacklisted scripts by performing 'step
// in' several times, finally resorting to 'step out' if unsuccessful. Positions
// array contains positions where blackbox state is changed. First interval
// isn't blackboxed. Array should be sorted.
type SetBlackboxedRangesParams struct {
	ScriptID  runtime.ScriptID  `json:"scriptId"` // Id of the script.
	Positions []*ScriptPosition `json:"positions"`
}

// SetBlackboxedRanges makes backend skip steps in the script in blackboxed
// ranges. VM will try leave blacklisted scripts by performing 'step in' several
// times, finally resorting to 'step out' if unsuccessful. Positions array
// contains positions where blackbox state is changed. First interval isn't
// blackboxed. Array should be sorted.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-setBlackboxedRanges
//
// parameters:
//
//	scriptID - Id of the script.
//	positions
func SetBlackboxedRanges(scriptID runtime.ScriptID, positions []*ScriptPosition) *SetBlackboxedRangesParams {
	return &SetBlackboxedRangesParams{
		ScriptID:  scriptID,
		Positions: positions,
	}
}

// Do executes Debugger.setBlackboxedRanges against the provided context.
func (p *SetBlackboxedRangesParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandSetBlackboxedRanges, p, nil)
}

// SetBreakpointParams sets JavaScript breakpoint at a given location.
type SetBreakpointParams struct {
	Location  *Location `json:"location"`                     // Location to set breakpoint in.
	Condition string    `json:"condition,omitempty,omitzero"` // Expression to use as a breakpoint condition. When specified, debugger will only stop on the breakpoint if this expression evaluates to true.
}

// SetBreakpoint sets JavaScript breakpoint at a given location.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-setBreakpoint
//
// parameters:
//
//	location - Location to set breakpoint in.
func SetBreakpoint(location *Location) *SetBreakpointParams {
	return &SetBreakpointParams{
		Location: location,
	}
}

// WithCondition expression to use as a breakpoint condition. When specified,
// debugger will only stop on the breakpoint if this expression evaluates to
// true.
func (p SetBreakpointParams) WithCondition(condition string) *SetBreakpointParams {
	p.Condition = condition
	return &p
}

// SetBreakpointReturns return values.
type SetBreakpointReturns struct {
	BreakpointID   BreakpointID `json:"breakpointId,omitempty,omitzero"`   // Id of the created breakpoint for further reference.
	ActualLocation *Location    `json:"actualLocation,omitempty,omitzero"` // Location this breakpoint resolved into.
}

// Do executes Debugger.setBreakpoint against the provided context.
//
// returns:
//
//	breakpointID - Id of the created breakpoint for further reference.
//	actualLocation - Location this breakpoint resolved into.
func (p *SetBreakpointParams) Do(ctx context.Context) (breakpointID BreakpointID, actualLocation *Location, err error) {
	// execute
	var res SetBreakpointReturns
	err = cdp.Execute(ctx, CommandSetBreakpoint, p, &res)
	if err != nil {
		return "", nil, err
	}

	return res.BreakpointID, res.ActualLocation, nil
}

// SetInstrumentationBreakpointParams sets instrumentation breakpoint.
type SetInstrumentationBreakpointParams struct {
	Instrumentation SetInstrumentationBreakpointInstrumentation `json:"instrumentation"` // Instrumentation name.
}

// SetInstrumentationBreakpoint sets instrumentation breakpoint.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-setInstrumentationBreakpoint
//
// parameters:
//
//	instrumentation - Instrumentation name.
func SetInstrumentationBreakpoint(instrumentation SetInstrumentationBreakpointInstrumentation) *SetInstrumentationBreakpointParams {
	return &SetInstrumentationBreakpointParams{
		Instrumentation: instrumentation,
	}
}

// SetInstrumentationBreakpointReturns return values.
type SetInstrumentationBreakpointReturns struct {
	BreakpointID BreakpointID `json:"breakpointId,omitempty,omitzero"` // Id of the created breakpoint for further reference.
}

// Do executes Debugger.setInstrumentationBreakpoint against the provided context.
//
// returns:
//
//	breakpointID - Id of the created breakpoint for further reference.
func (p *SetInstrumentationBreakpointParams) Do(ctx context.Context) (breakpointID BreakpointID, err error) {
	// execute
	var res SetInstrumentationBreakpointReturns
	err = cdp.Execute(ctx, CommandSetInstrumentationBreakpoint, p, &res)
	if err != nil {
		return "", err
	}

	return res.BreakpointID, nil
}

// SetBreakpointByURLParams sets JavaScript breakpoint at given location
// specified either by URL or URL regex. Once this command is issued, all
// existing parsed scripts will have breakpoints resolved and returned in
// locations property. Further matching script parsing will result in subsequent
// breakpointResolved events issued. This logical breakpoint will survive page
// reloads.
type SetBreakpointByURLParams struct {
	LineNumber   int64  `json:"lineNumber"`                      // Line number to set breakpoint at.
	URL          string `json:"url,omitempty,omitzero"`          // URL of the resources to set breakpoint on.
	URLRegex     string `json:"urlRegex,omitempty,omitzero"`     // Regex pattern for the URLs of the resources to set breakpoints on. Either url or urlRegex must be specified.
	ScriptHash   string `json:"scriptHash,omitempty,omitzero"`   // Script hash of the resources to set breakpoint on.
	ColumnNumber int64  `json:"columnNumber,omitempty,omitzero"` // Offset in the line to set breakpoint at.
	Condition    string `json:"condition,omitempty,omitzero"`    // Expression to use as a breakpoint condition. When specified, debugger will only stop on the breakpoint if this expression evaluates to true.
}

// SetBreakpointByURL sets JavaScript breakpoint at given location specified
// either by URL or URL regex. Once this command is issued, all existing parsed
// scripts will have breakpoints resolved and returned in locations property.
// Further matching script parsing will result in subsequent breakpointResolved
// events issued. This logical breakpoint will survive page reloads.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-setBreakpointByUrl
//
// parameters:
//
//	lineNumber - Line number to set breakpoint at.
func SetBreakpointByURL(lineNumber int64) *SetBreakpointByURLParams {
	return &SetBreakpointByURLParams{
		LineNumber: lineNumber,
	}
}

// WithURL URL of the resources to set breakpoint on.
func (p SetBreakpointByURLParams) WithURL(url string) *SetBreakpointByURLParams {
	p.URL = url
	return &p
}

// WithURLRegex regex pattern for the URLs of the resources to set
// breakpoints on. Either url or urlRegex must be specified.
func (p SetBreakpointByURLParams) WithURLRegex(urlRegex string) *SetBreakpointByURLParams {
	p.URLRegex = urlRegex
	return &p
}

// WithScriptHash script hash of the resources to set breakpoint on.
func (p SetBreakpointByURLParams) WithScriptHash(scriptHash string) *SetBreakpointByURLParams {
	p.ScriptHash = scriptHash
	return &p
}

// WithColumnNumber offset in the line to set breakpoint at.
func (p SetBreakpointByURLParams) WithColumnNumber(columnNumber int64) *SetBreakpointByURLParams {
	p.ColumnNumber = columnNumber
	return &p
}

// WithCondition expression to use as a breakpoint condition. When specified,
// debugger will only stop on the breakpoint if this expression evaluates to
// true.
func (p SetBreakpointByURLParams) WithCondition(condition string) *SetBreakpointByURLParams {
	p.Condition = condition
	return &p
}

// SetBreakpointByURLReturns return values.
type SetBreakpointByURLReturns struct {
	BreakpointID BreakpointID `json:"breakpointId,omitempty,omitzero"` // Id of the created breakpoint for further reference.
	Locations    []*Location  `json:"locations,omitempty,omitzero"`    // List of the locations this breakpoint resolved into upon addition.
}

// Do executes Debugger.setBreakpointByUrl against the provided context.
//
// returns:
//
//	breakpointID - Id of the created breakpoint for further reference.
//	locations - List of the locations this breakpoint resolved into upon addition.
func (p *SetBreakpointByURLParams) Do(ctx context.Context) (breakpointID BreakpointID, locations []*Location, err error) {
	// execute
	var res SetBreakpointByURLReturns
	err = cdp.Execute(ctx, CommandSetBreakpointByURL, p, &res)
	if err != nil {
		return "", nil, err
	}

	return res.BreakpointID, res.Locations, nil
}

// SetBreakpointOnFunctionCallParams sets JavaScript breakpoint before each
// call to the given function. If another function was created from the same
// source as a given one, calling it will also trigger the breakpoint.
type SetBreakpointOnFunctionCallParams struct {
	ObjectID  runtime.RemoteObjectID `json:"objectId"`                     // Function object id.
	Condition string                 `json:"condition,omitempty,omitzero"` // Expression to use as a breakpoint condition. When specified, debugger will stop on the breakpoint if this expression evaluates to true.
}

// SetBreakpointOnFunctionCall sets JavaScript breakpoint before each call to
// the given function. If another function was created from the same source as a
// given one, calling it will also trigger the breakpoint.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-setBreakpointOnFunctionCall
//
// parameters:
//
//	objectID - Function object id.
func SetBreakpointOnFunctionCall(objectID runtime.RemoteObjectID) *SetBreakpointOnFunctionCallParams {
	return &SetBreakpointOnFunctionCallParams{
		ObjectID: objectID,
	}
}

// WithCondition expression to use as a breakpoint condition. When specified,
// debugger will stop on the breakpoint if this expression evaluates to true.
func (p SetBreakpointOnFunctionCallParams) WithCondition(condition string) *SetBreakpointOnFunctionCallParams {
	p.Condition = condition
	return &p
}

// SetBreakpointOnFunctionCallReturns return values.
type SetBreakpointOnFunctionCallReturns struct {
	BreakpointID BreakpointID `json:"breakpointId,omitempty,omitzero"` // Id of the created breakpoint for further reference.
}

// Do executes Debugger.setBreakpointOnFunctionCall against the provided context.
//
// returns:
//
//	breakpointID - Id of the created breakpoint for further reference.
func (p *SetBreakpointOnFunctionCallParams) Do(ctx context.Context) (breakpointID BreakpointID, err error) {
	// execute
	var res SetBreakpointOnFunctionCallReturns
	err = cdp.Execute(ctx, CommandSetBreakpointOnFunctionCall, p, &res)
	if err != nil {
		return "", err
	}

	return res.BreakpointID, nil
}

// SetBreakpointsActiveParams activates / deactivates all breakpoints on the
// page.
type SetBreakpointsActiveParams struct {
	Active bool `json:"active"` // New value for breakpoints active state.
}

// SetBreakpointsActive activates / deactivates all breakpoints on the page.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-setBreakpointsActive
//
// parameters:
//
//	active - New value for breakpoints active state.
func SetBreakpointsActive(active bool) *SetBreakpointsActiveParams {
	return &SetBreakpointsActiveParams{
		Active: active,
	}
}

// Do executes Debugger.setBreakpointsActive against the provided context.
func (p *SetBreakpointsActiveParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandSetBreakpointsActive, p, nil)
}

// SetPauseOnExceptionsParams defines pause on exceptions state. Can be set
// to stop on all exceptions, uncaught exceptions, or caught exceptions, no
// exceptions. Initial pause on exceptions state is none.
type SetPauseOnExceptionsParams struct {
	State ExceptionsState `json:"state"` // Pause on exceptions mode.
}

// SetPauseOnExceptions defines pause on exceptions state. Can be set to stop
// on all exceptions, uncaught exceptions, or caught exceptions, no exceptions.
// Initial pause on exceptions state is none.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-setPauseOnExceptions
//
// parameters:
//
//	state - Pause on exceptions mode.
func SetPauseOnExceptions(state ExceptionsState) *SetPauseOnExceptionsParams {
	return &SetPauseOnExceptionsParams{
		State: state,
	}
}

// Do executes Debugger.setPauseOnExceptions against the provided context.
func (p *SetPauseOnExceptionsParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandSetPauseOnExceptions, p, nil)
}

// SetReturnValueParams changes return value in top frame. Available only at
// return break position.
type SetReturnValueParams struct {
	NewValue *runtime.CallArgument `json:"newValue"` // New return value.
}

// SetReturnValue changes return value in top frame. Available only at return
// break position.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-setReturnValue
//
// parameters:
//
//	newValue - New return value.
func SetReturnValue(newValue *runtime.CallArgument) *SetReturnValueParams {
	return &SetReturnValueParams{
		NewValue: newValue,
	}
}

// Do executes Debugger.setReturnValue against the provided context.
func (p *SetReturnValueParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandSetReturnValue, p, nil)
}

// SetScriptSourceParams edits JavaScript source live. In general, functions
// that are currently on the stack can not be edited with a single exception: If
// the edited function is the top-most stack frame and that is the only
// activation of that function on the stack. In this case the live edit will be
// successful and a Debugger.restartFrame for the top-most function is
// automatically triggered.
type SetScriptSourceParams struct {
	ScriptID             runtime.ScriptID `json:"scriptId"`             // Id of the script to edit.
	ScriptSource         string           `json:"scriptSource"`         // New content of the script.
	DryRun               bool             `json:"dryRun"`               // If true the change will not actually be applied. Dry run may be used to get result description without actually modifying the code.
	AllowTopFrameEditing bool             `json:"allowTopFrameEditing"` // If true, then scriptSource is allowed to change the function on top of the stack as long as the top-most stack frame is the only activation of that function.
}

// SetScriptSource edits JavaScript source live. In general, functions that
// are currently on the stack can not be edited with a single exception: If the
// edited function is the top-most stack frame and that is the only activation
// of that function on the stack. In this case the live edit will be successful
// and a Debugger.restartFrame for the top-most function is automatically
// triggered.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-setScriptSource
//
// parameters:
//
//	scriptID - Id of the script to edit.
//	scriptSource - New content of the script.
func SetScriptSource(scriptID runtime.ScriptID, scriptSource string) *SetScriptSourceParams {
	return &SetScriptSourceParams{
		ScriptID:             scriptID,
		ScriptSource:         scriptSource,
		DryRun:               false,
		AllowTopFrameEditing: false,
	}
}

// WithDryRun if true the change will not actually be applied. Dry run may be
// used to get result description without actually modifying the code.
func (p SetScriptSourceParams) WithDryRun(dryRun bool) *SetScriptSourceParams {
	p.DryRun = dryRun
	return &p
}

// WithAllowTopFrameEditing if true, then scriptSource is allowed to change
// the function on top of the stack as long as the top-most stack frame is the
// only activation of that function.
func (p SetScriptSourceParams) WithAllowTopFrameEditing(allowTopFrameEditing bool) *SetScriptSourceParams {
	p.AllowTopFrameEditing = allowTopFrameEditing
	return &p
}

// SetScriptSourceReturns return values.
type SetScriptSourceReturns struct {
	Status           SetScriptSourceStatus     `json:"status,omitempty,omitzero"`           // Whether the operation was successful or not. Only Ok denotes a successful live edit while the other enum variants denote why the live edit failed.
	ExceptionDetails *runtime.ExceptionDetails `json:"exceptionDetails,omitempty,omitzero"` // Exception details if any. Only present when status is CompileError.
}

// Do executes Debugger.setScriptSource against the provided context.
//
// returns:
//
//	status - Whether the operation was successful or not. Only Ok denotes a successful live edit while the other enum variants denote why the live edit failed.
//	exceptionDetails - Exception details if any. Only present when status is CompileError.
func (p *SetScriptSourceParams) Do(ctx context.Context) (status SetScriptSourceStatus, exceptionDetails *runtime.ExceptionDetails, err error) {
	// execute
	var res SetScriptSourceReturns
	err = cdp.Execute(ctx, CommandSetScriptSource, p, &res)
	if err != nil {
		return "", nil, err
	}

	return res.Status, res.ExceptionDetails, nil
}

// SetSkipAllPausesParams makes page not interrupt on any pauses (breakpoint,
// exception, dom exception etc).
type SetSkipAllPausesParams struct {
	Skip bool `json:"skip"` // New value for skip pauses state.
}

// SetSkipAllPauses makes page not interrupt on any pauses (breakpoint,
// exception, dom exception etc).
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-setSkipAllPauses
//
// parameters:
//
//	skip - New value for skip pauses state.
func SetSkipAllPauses(skip bool) *SetSkipAllPausesParams {
	return &SetSkipAllPausesParams{
		Skip: skip,
	}
}

// Do executes Debugger.setSkipAllPauses against the provided context.
func (p *SetSkipAllPausesParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandSetSkipAllPauses, p, nil)
}

// SetVariableValueParams changes value of variable in a callframe.
// Object-based scopes are not supported and must be mutated manually.
type SetVariableValueParams struct {
	ScopeNumber  int64                 `json:"scopeNumber"`  // 0-based number of scope as was listed in scope chain. Only 'local', 'closure' and 'catch' scope types are allowed. Other scopes could be manipulated manually.
	VariableName string                `json:"variableName"` // Variable name.
	NewValue     *runtime.CallArgument `json:"newValue"`     // New variable value.
	CallFrameID  CallFrameID           `json:"callFrameId"`  // Id of callframe that holds variable.
}

// SetVariableValue changes value of variable in a callframe. Object-based
// scopes are not supported and must be mutated manually.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-setVariableValue
//
// parameters:
//
//	scopeNumber - 0-based number of scope as was listed in scope chain. Only 'local', 'closure' and 'catch' scope types are allowed. Other scopes could be manipulated manually.
//	variableName - Variable name.
//	newValue - New variable value.
//	callFrameID - Id of callframe that holds variable.
func SetVariableValue(scopeNumber int64, variableName string, newValue *runtime.CallArgument, callFrameID CallFrameID) *SetVariableValueParams {
	return &SetVariableValueParams{
		ScopeNumber:  scopeNumber,
		VariableName: variableName,
		NewValue:     newValue,
		CallFrameID:  callFrameID,
	}
}

// Do executes Debugger.setVariableValue against the provided context.
func (p *SetVariableValueParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandSetVariableValue, p, nil)
}

// StepIntoParams steps into the function call.
type StepIntoParams struct {
	BreakOnAsyncCall bool             `json:"breakOnAsyncCall"`            // Debugger will pause on the execution of the first async task which was scheduled before next pause.
	SkipList         []*LocationRange `json:"skipList,omitempty,omitzero"` // The skipList specifies location ranges that should be skipped on step into.
}

// StepInto steps into the function call.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-stepInto
//
// parameters:
func StepInto() *StepIntoParams {
	return &StepIntoParams{
		BreakOnAsyncCall: false,
	}
}

// WithBreakOnAsyncCall debugger will pause on the execution of the first
// async task which was scheduled before next pause.
func (p StepIntoParams) WithBreakOnAsyncCall(breakOnAsyncCall bool) *StepIntoParams {
	p.BreakOnAsyncCall = breakOnAsyncCall
	return &p
}

// WithSkipList the skipList specifies location ranges that should be skipped
// on step into.
func (p StepIntoParams) WithSkipList(skipList []*LocationRange) *StepIntoParams {
	p.SkipList = skipList
	return &p
}

// Do executes Debugger.stepInto against the provided context.
func (p *StepIntoParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandStepInto, p, nil)
}

// StepOutParams steps out of the function call.
type StepOutParams struct{}

// StepOut steps out of the function call.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-stepOut
func StepOut() *StepOutParams {
	return &StepOutParams{}
}

// Do executes Debugger.stepOut against the provided context.
func (p *StepOutParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandStepOut, nil, nil)
}

// StepOverParams steps over the statement.
type StepOverParams struct {
	SkipList []*LocationRange `json:"skipList,omitempty,omitzero"` // The skipList specifies location ranges that should be skipped on step over.
}

// StepOver steps over the statement.
//
// See: https://chromedevtools.github.io/devtools-protocol/tot/Debugger#method-stepOver
//
// parameters:
func StepOver() *StepOverParams {
	return &StepOverParams{}
}

// WithSkipList the skipList specifies location ranges that should be skipped
// on step over.
func (p StepOverParams) WithSkipList(skipList []*LocationRange) *StepOverParams {
	p.SkipList = skipList
	return &p
}

// Do executes Debugger.stepOver against the provided context.
func (p *StepOverParams) Do(ctx context.Context) (err error) {
	return cdp.Execute(ctx, CommandStepOver, p, nil)
}

// Command names.
const (
	CommandContinueToLocation           = "Debugger.continueToLocation"
	CommandDisable                      = "Debugger.disable"
	CommandEnable                       = "Debugger.enable"
	CommandEvaluateOnCallFrame          = "Debugger.evaluateOnCallFrame"
	CommandGetPossibleBreakpoints       = "Debugger.getPossibleBreakpoints"
	CommandGetScriptSource              = "Debugger.getScriptSource"
	CommandDisassembleWasmModule        = "Debugger.disassembleWasmModule"
	CommandNextWasmDisassemblyChunk     = "Debugger.nextWasmDisassemblyChunk"
	CommandGetStackTrace                = "Debugger.getStackTrace"
	CommandPause                        = "Debugger.pause"
	CommandRemoveBreakpoint             = "Debugger.removeBreakpoint"
	CommandRestartFrame                 = "Debugger.restartFrame"
	CommandResume                       = "Debugger.resume"
	CommandSearchInContent              = "Debugger.searchInContent"
	CommandSetAsyncCallStackDepth       = "Debugger.setAsyncCallStackDepth"
	CommandSetBlackboxExecutionContexts = "Debugger.setBlackboxExecutionContexts"
	CommandSetBlackboxPatterns          = "Debugger.setBlackboxPatterns"
	CommandSetBlackboxedRanges          = "Debugger.setBlackboxedRanges"
	CommandSetBreakpoint                = "Debugger.setBreakpoint"
	CommandSetInstrumentationBreakpoint = "Debugger.setInstrumentationBreakpoint"
	CommandSetBreakpointByURL           = "Debugger.setBreakpointByUrl"
	CommandSetBreakpointOnFunctionCall  = "Debugger.setBreakpointOnFunctionCall"
	CommandSetBreakpointsActive         = "Debugger.setBreakpointsActive"
	CommandSetPauseOnExceptions         = "Debugger.setPauseOnExceptions"
	CommandSetReturnValue               = "Debugger.setReturnValue"
	CommandSetScriptSource              = "Debugger.setScriptSource"
	CommandSetSkipAllPauses             = "Debugger.setSkipAllPauses"
	CommandSetVariableValue             = "Debugger.setVariableValue"
	CommandStepInto                     = "Debugger.stepInto"
	CommandStepOut                      = "Debugger.stepOut"
	CommandStepOver                     = "Debugger.stepOver"
)
