package lttb

import (
	"fmt"
	"reflect"
	"sync"
	"testing"
	"unsafe"

	golttb "github.com/dgryski/go-lttb"
	"github.com/google/go-cmp/cmp"
)

func TestDownsample(t *testing.T) {
	t.Parallel()

	for _, threshold := range []int{0, len(points[0]), len(points[0]) + 1} {
		have, err := Downsample(len(points[0]), threshold, newIterator(points[0]))
		if err != nil {
			t.Fatalf("threshold-%d: got err: %v", threshold, err)
		}

		want := points[0]
		if diff := cmp.Diff(have, want, cmp.AllowUnexported(Point{})); diff != "" {
			t.Errorf("threshold-%d: %s", threshold, diff)
		}
	}

	for i := 1; i < 3; i++ {
		threshold := i
		_, err := Downsample(len(points[0]), threshold, newIterator(points[0]))
		if have, want := fmt.Sprint(err), "lttb: min threshold is 3"; have != want {
			t.Errorf("threshold-%d: have err: %v, want %v", threshold, have, want)
		}
	}

	var wg sync.WaitGroup
	for i, ps := range points {
		for threshold := 3; threshold < len(ps); threshold++ {
			wg.Add(1)
			i, ps, threshold := i, ps, threshold
			go func() {
				defer wg.Done()

				msg := func(fmtstr string, args ...interface{}) string {
					return fmt.Sprintf(
						"points=%d len=%d threshold=%d: "+fmtstr,
						append([]interface{}{i, len(ps), threshold}, args...)...,
					)
				}

				ours, err := Downsample(len(ps), threshold, newIterator(ps))
				if err != nil {
					t.Error(msg("error: %v", err))
				}

				if have, want := len(ours), threshold; have != want {
					t.Error(msg("len(samples) != threshold: have %d, want %d", have, want))
				}

				if have, want := ours[0], ps[0]; have != want {
					t.Error(msg("samples[0] != data[0]: have %v, want %v", have, want))
				}

				if have, want := ours[len(ours)-1], ps[len(ps)-1]; have != want {
					t.Error(msg("samples[-1] != data[-1]: have %v, want %v", have, want))
				}

				// Test LTTB algorithm's equivalence to dgrisky/go-lttb
				in := *(*[]golttb.Point[float64])(unsafe.Pointer(&ps)) // #skipcq: GSC-G103
				out := golttb.LTTB(in, threshold)
				theirs := *(*[]Point)(unsafe.Pointer(&out)) // #skipcq: GSC-G103

				if !reflect.DeepEqual(ours, theirs) {
					t.Error(msg(cmp.Diff(ours, theirs, cmp.AllowUnexported(Point{}))))
				}
			}()
		}
	}

	wg.Wait()
}

func BenchmarkLTTB(b *testing.B) {
	data := *(*[]golttb.Point[float64])(unsafe.Pointer(&points[0])) // #skipcq: GSC-G103
	b.Run("dgryski", func(b *testing.B) {
		for i := 0; i < b.N; i++ {
			golttb.LTTB(data, 1000)
		}
	})

	b.Run("tsenart", func(b *testing.B) {
		for i := 0; i < b.N; i++ {
			Downsample(len(data), 1000, newIterator(points[0]))
		}
	})
}

func newIterator(data []Point) Iter {
	return func(count int) ([]Point, error) {
		if count > len(data) {
			count = len(data)
		}
		ps := data[:count]
		data = data[count:]
		return ps, nil
	}
}

// From https://raw.githubusercontent.com/sveinn-steinarsson/flot-downsample/master/demo_data.js
var points = [][]Point{
	{
		{0, 29.357995947822218},
		{1, 29.40932479606209},
		{2, 29.28168582006162},
		{3, 30.409965579108867},
		{4, 30.7726859735917},
		{5, 30.839942247539028},
		{6, 30.760611642264667},
		{7, 31.203663004229718},
		{8, 31.38899603525572},
		{9, 30.890299916955737},
		{10, 30.467811944911556},
		{11, 30.596837868069542},
		{12, 30.59789593509767},
		{13, 30.19693062465079},
		{14, 29.89081330734553},
		{15, 29.54668002901058},
		{16, 29.54890739422219},
		{17, 30.53743760171474},
		{18, 30.74066032317061},
		{19, 30.3774450601516},
		{20, 30.095148889986568},
		{21, 30.057979182917986},
		{22, 30.364655421168525},
		{23, 30.293450053773604},
		{24, 30.14578230340987},
		{25, 30.277772879951996},
		{26, 30.3711931235659},
		{27, 30.355932660992572},
		{28, 29.994740831603046},
		{29, 29.938012885023657},
		{30, 29.557268760451187},
		{31, 29.16161297604625},
		{32, 29.6921163421055},
		{33, 30.246270628292226},
		{34, 30.210955154680928},
		{35, 29.382964668934058},
		{36, 29.52892023906641},
		{37, 29.719742065732202},
		{38, 30.013856636945924},
		{39, 29.818636169776926},
		{40, 27.931224826933345},
		{41, 28.103057873678374},
		{42, 28.276025902356782},
		{43, 28.193497487789774},
		{44, 28.22099171488288},
		{45, 28.06872431241809},
		{46, 27.239835885250365},
		{47, 26.606504358317906},
		{48, 27.220308136213916},
		{49, 25.80790898978574},
		{50, 26.060334838062595},
		{51, 25.89611036822564},
		{52, 25.824433242208308},
		{53, 25.89212193260212},
		{54, 27.28963669872412},
		{55, 27.208259984780806},
		{56, 27.298211480886604},
		{57, 27.056163307023077},
		{58, 26.434701869643924},
		{59, 26.21938188318228},
		{60, 26.402877759468772},
		{61, 26.326019357967294},
		{62, 27.592635104460253},
		{63, 27.597468405434316},
		{64, 27.533410081050647},
		{65, 27.30713964830731},
		{66, 27.170789712751404},
		{67, 27.610166301145746},
		{68, 27.619847332319885},
		{69, 27.55974343996647},
		{70, 28.168803492093716},
		{71, 28.284229862452584},
		{72, 28.292638086760352},
		{73, 28.523718178262335},
		{74, 28.52268591410558},
		{75, 28.576637206595993},
		{76, 27.763715461946813},
		{77, 27.48488075310065},
		{78, 27.304006460281794},
		{79, 26.902025619666333},
		{80, 26.690162079132378},
		{81, 26.52277286738866},
		{82, 26.09406789919051},
		{83, 26.029039984983562},
		{84, 26.145181108429618},
		{85, 24.75335912549342},
		{86, 24.787772395126176},
		{87, 24.966652093726385},
		{88, 25.33621421606084},
		{89, 25.29283081429076},
		{90, 25.427452306231707},
		{91, 25.34334450564347},
		{92, 25.971939771661045},
		{93, 25.907910638400836},
		{94, 25.971895864171003},
		{95, 26.448419417928317},
		{96, 26.461057396494514},
		{97, 26.332417666691025},
		{98, 26.17395667752311},
		{99, 25.318121919340044},
		{100, 26.092918888834987},
		{101, 26.080903284583307},
		{102, 26.65713935644778},
		{103, 26.90227335392346},
		{104, 27.14904791481757},
		{105, 27.45775156727039},
		{106, 28.476933177720472},
		{107, 27.899526884805073},
		{108, 27.274094274806096},
		{109, 27.007320452528134},
		{110, 27.767594298696466},
		{111, 28.669197835798027},
		{112, 28.78287095361503},
		{113, 28.744062941208348},
		{114, 29.266136201836588},
		{115, 29.29631346563548},
		{116, 29.30576721224685},
		{117, 28.92417256867908},
		{118, 29.172020952087326},
		{119, 29.36572118529946},
		{120, 29.407882394097168},
		{121, 29.24581675687127},
		{122, 30.374496989862006},
		{123, 29.75867484229172},
		{124, 29.724977349500893},
		{125, 29.766232845230032},
		{126, 29.755425025733203},
		{127, 29.891515813029994},
		{128, 29.224842590902917},
		{129, 29.19894861750696},
		{130, 28.877260649054524},
		{131, 29.100160763856657},
		{132, 29.11303194254891},
		{133, 29.471499370130353},
		{134, 29.725430707066725},
		{135, 29.875645922022283},
		{136, 29.32336837740634},
		{137, 29.659223458914788},
		{138, 29.916122460150415},
		{139, 29.947591268372232},
		{140, 29.912720660976237},
		{141, 30.030001305562784},
		{142, 30.01603062688162},
		{143, 30.206305874273855},
		{144, 29.654798308068155},
		{145, 29.51005186796388},
		{146, 29.479249298103124},
		{147, 29.50465455692649},
		{148, 28.668390496723326},
		{149, 29.70235193454973},
		{150, 29.651621644229916},
		{151, 29.514646394709878},
		{152, 29.506006233800203},
		{153, 29.85094476486748},
		{154, 29.815453456016304},
		{155, 29.942433471924208},
		{156, 29.690047803042003},
		{157, 29.081114874079432},
		{158, 29.064499954878283},
		{159, 29.17427761652929},
		{160, 28.48135454514647},
		{161, 28.43614347392514},
		{162, 28.615830690136196},
		{163, 27.244815449582045},
		{164, 28.030839937141593},
		{165, 28.09433740820739},
		{166, 28.134090510807432},
		{167, 28.333335545414524},
		{168, 28.187860385951467},
		{169, 28.275647207865163},
		{170, 28.110663780904385},
		{171, 27.941108395946074},
		{172, 27.155839334130572},
		{173, 27.486338387798014},
		{174, 27.822551655905976},
		{175, 27.863745721980674},
		{176, 27.51492123912736},
		{177, 27.83845450366903},
		{178, 27.143346163904276},
		{179, 26.470510229480666},
		{180, 27.030260699355054},
		{181, 26.530046883178517},
		{182, 26.061343714664623},
		{183, 26.451944204073293},
		{184, 26.430516873002972},
		{185, 27.54560649645601},
		{186, 27.607455732128237},
		{187, 27.3151823435893},
		{188, 27.015508353290546},
		{189, 27.564104270774138},
		{190, 27.401004407024764},
		{191, 27.158236015306873},
		{192, 27.15560082391509},
		{193, 27.052060260660955},
		{194, 27.170854273897024},
		{195, 27.764649442110628},
		{196, 28.37485762180034},
		{197, 28.693644963914128},
		{198, 28.639988945921225},
		{199, 28.55415913033328},
		{200, 28.227269873938933},
		{201, 28.17193587609251},
		{202, 28.326544342536973},
		{203, 27.64207374329132},
		{204, 28.28132114918632},
		{205, 28.87895845286964},
		{206, 28.188530355189847},
		{207, 27.543826104728634},
		{208, 27.518668557969406},
		{209, 27.501005443892083},
		{210, 27.4034023213343},
		{211, 27.81919230719886},
		{212, 28.51752545491846},
		{213, 28.45642871817993},
		{214, 28.281434136230057},
		{215, 28.955222450571647},
		{216, 28.817758413114476},
		{217, 28.512855374048605},
		{218, 28.53649013938214},
		{219, 28.037741647416063},
		{220, 28.758762367232816},
		{221, 28.411225526795217},
		{222, 28.081382658095393},
		{223, 28.018197966678386},
		{224, 27.669840575416934},
		{225, 27.729482622798493},
		{226, 27.44804950151258},
		{227, 27.057357503159633},
		{228, 27.02667664651584},
		{229, 26.99034079421472},
		{230, 27.624460374012415},
		{231, 26.6107079586933},
		{232, 26.586632740266342},
		{233, 26.50503068257422},
		{234, 26.528094947562206},
		{235, 27.42586110225183},
		{236, 27.58607997646959},
		{237, 27.119751888180552},
		{238, 27.08687612699868},
		{239, 26.50890594997075},
		{240, 27.31126847238326},
		{241, 27.241646652930587},
		{242, 27.206837899664286},
		{243, 27.806910729195003},
		{244, 27.943871889687294},
		{245, 27.682133734768627},
		{246, 27.717567280916626},
		{247, 28.50754775235408},
		{248, 28.522618646129153},
		{249, 28.119450170073687},
		{250, 28.142446798473102},
		{251, 27.493875384325104},
		{252, 27.232866008532874},
		{253, 26.815819408391537},
		{254, 26.66806605894335},
		{255, 26.832795316319906},
		{256, 26.6709072973403},
		{257, 26.805339614467922},
		{258, 25.87709141573906},
		{259, 25.78663848060838},
		{260, 26.292688529709856},
		{261, 26.13540568260593},
		{262, 26.950679537489023},
		{263, 26.74360016328177},
		{264, 25.880978465490344},
		{265, 26.625349316487267},
		{266, 27.726596358113618},
		{267, 27.678660195691705},
		{268, 27.712915567796777},
		{269, 27.721698306913026},
		{270, 27.905267233295028},
		{271, 27.957187159455156},
		{272, 27.858194094400584},
		{273, 28.081932932040928},
		{274, 27.735044748607958},
		{275, 27.032793497721116},
		{276, 27.507939311361685},
		{277, 27.574645954142067},
		{278, 27.574051095225684},
		{279, 27.759761781417595},
		{280, 27.365795276665352},
		{281, 26.617667215269847},
		{282, 26.798097033417232},
		{283, 26.195989054524656},
		{284, 26.976938134909243},
		{285, 26.89198848346594},
		{286, 27.147250351373707},
		{287, 27.43948527543756},
		{288, 28.0008367927751},
		{289, 28.231697069321097},
		{290, 28.7967810312443},
		{291, 29.512759843423506},
		{292, 29.41555351035952},
		{293, 30.70537467842703},
		{294, 30.332009655179892},
		{295, 30.368538690398843},
		{296, 30.73440265654596},
		{297, 30.967799052657455},
		{298, 31.22392323854322},
		{299, 31.122470949743033},
		{300, 31.140432351354303},
		{301, 31.13729855237942},
		{302, 30.716789453632746},
		{303, 31.055606178854195},
		{304, 31.025310612429436},
		{305, 30.746317663776093},
		{306, 31.16989662941391},
		{307, 29.623226075091324},
		{308, 29.11707721446215},
		{309, 29.671462419401518},
		{310, 29.32656872224774},
		{311, 29.360034344664133},
		{312, 30.193073601940906},
		{313, 30.051282853224887},
		{314, 30.094809176026143},
		{315, 30.62616938793058},
		{316, 31.24738345775188},
		{317, 31.288951935975373},
		{318, 30.793680071961138},
		{319, 30.855230722079575},
		{320, 30.172789122099953},
		{321, 30.16588226960243},
		{322, 30.02896795596743},
		{323, 28.710256908027688},
		{324, 28.788445957400086},
		{325, 29.14084131034792},
		{326, 29.154793392113767},
		{327, 29.509956518244064},
		{328, 29.919766479376083},
		{329, 30.809790278228583},
		{330, 31.51994434827819},
		{331, 31.14360887134284},
		{332, 30.960009728867895},
		{333, 31.007757592227232},
		{334, 29.912529162927882},
		{335, 30.430549132609936},
		{336, 30.286270410087386},
		{337, 30.29853067098949},
		{338, 30.28484502358319},
		{339, 30.2649148646112},
		{340, 30.269814084876984},
		{341, 29.48743777656781},
		{342, 29.25324229055582},
		{343, 29.3224085984498},
		{344, 28.845749148292715},
		{345, 28.25734414947841},
		{346, 28.22244460496951},
		{347, 28.212727797755832},
		{348, 28.099709252761244},
		{349, 28.157686542165766},
		{350, 28.15110192590233},
		{351, 28.242118337655835},
		{352, 29.53367482995351},
		{353, 29.532585019641683},
		{354, 29.943653842668887},
		{355, 28.64740678713178},
		{356, 28.627878421859094},
		{357, 28.85948017439502},
		{358, 29.421837208187565},
		{359, 29.213674689162733},
		{360, 29.588524659441383},
		{361, 29.871736651370572},
		{362, 29.97536581410366},
		{363, 29.090848126781193},
		{364, 29.07367624649577},
		{365, 28.936182688349078},
		{366, 28.478980562612755},
		{367, 27.701450632319055},
		{368, 27.70314463302395},
		{369, 27.690633468393905},
		{370, 27.712418470005847},
		{371, 28.04360951096915},
		{372, 27.899319369636174},
		{373, 27.67953057101954},
		{374, 27.33713099741098},
		{375, 27.715275282261704},
		{376, 27.09457980740444},
		{377, 26.564904718909652},
		{378, 26.851805720484524},
		{379, 27.17971899490677},
		{380, 27.203035420946204},
		{381, 26.98833921789535},
		{382, 27.3376969508508},
		{383, 27.36673199620505},
		{384, 27.22367712016426},
		{385, 27.295774446517406},
		{386, 27.24426813335004},
		{387, 27.804345559349205},
		{388, 27.81107241888124},
		{389, 28.007141902081127},
		{390, 28.196204812447014},
		{391, 27.570724094739713},
		{392, 28.353947742891283},
		{393, 28.264285808134655},
		{394, 28.001544074943496},
		{395, 28.073693599174273},
		{396, 28.13948170648531},
		{397, 27.7246917685699},
		{398, 27.372056917732422},
		{399, 26.76664288679182},
		{400, 26.726530330089947},
		{401, 26.73659456496834},
		{402, 26.35282363879538},
		{403, 26.953655414803745},
		{404, 27.037728660238816},
		{405, 26.391122852260175},
		{406, 26.45637305967239},
		{407, 25.78486318168484},
		{408, 25.92078676697798},
		{409, 26.4970571420541},
		{410, 25.86809752383717},
		{411, 26.534413662015773},
		{412, 27.226357547185447},
		{413, 26.71252553894152},
		{414, 26.435002451423312},
		{415, 26.921455461481504},
		{416, 27.06045686220195},
		{417, 27.09418382303274},
		{418, 28.011098852919908},
		{419, 27.896959438752912},
		{420, 27.914042725695705},
		{421, 27.547368437091734},
		{422, 26.710013475304418},
		{423, 26.67332778875601},
		{424, 26.624659424882857},
		{425, 26.57172379809563},
		{426, 26.846697174621756},
		{427, 25.8834697613218},
		{428, 26.0400273638686},
		{429, 25.922770964698366},
		{430, 26.550977646141284},
		{431, 26.7196171595444},
		{432, 26.57693368973477},
		{433, 26.754937488902705},
		{434, 26.68923906981248},
		{435, 27.578054073124644},
		{436, 28.491944955522996},
		{437, 29.462067517762534},
		{438, 29.049663449716007},
		{439, 29.29125398211083},
		{440, 29.716399938469106},
		{441, 29.697795912329543},
		{442, 28.842426341018918},
		{443, 27.995794268014496},
		{444, 27.568707173494055},
		{445, 27.572194900322412},
		{446, 28.94876289646488},
		{447, 28.827587778805206},
		{448, 28.20122725800809},
		{449, 28.982543824618322},
		{450, 28.98764510982713},
		{451, 28.476638374330257},
		{452, 27.8971845311884},
		{453, 28.357599004387694},
		{454, 29.550067169716076},
		{455, 29.059986913307316},
		{456, 29.44698201092504},
		{457, 29.444893222605785},
		{458, 30.2987283695555},
		{459, 30.490753195685443},
		{460, 30.3712818585665},
		{461, 31.03092246398632},
		{462, 31.1306416649066},
		{463, 31.204020755490298},
		{464, 31.302761670091414},
		{465, 31.287884426895985},
		{466, 30.49414678626753},
		{467, 30.9842218570926},
		{468, 30.88247586217915},
		{469, 31.65627890749247},
		{470, 31.300373482363312},
		{471, 31.009412915052188},
		{472, 31.135319385771062},
		{473, 31.142947189076295},
		{474, 31.095825728106682},
		{475, 31.402733294236572},
		{476, 31.348161546083656},
		{477, 31.600766880442425},
		{478, 31.51732023898528},
		{479, 31.218212979786024},
		{480, 31.431310499487758},
		{481, 31.536799819023997},
		{482, 31.590576295671646},
		{483, 31.49429616322668},
		{484, 32.1816892011556},
		{485, 32.61239327037908},
		{486, 32.69902576477615},
		{487, 32.89146487332367},
		{488, 33.87564290993047},
		{489, 33.89045560640269},
		{490, 33.8295470579556},
		{491, 33.83070162926511},
		{492, 33.57726370546161},
		{493, 33.80516778960143},
		{494, 34.26212065396207},
		{495, 34.4134739195819},
		{496, 34.72554255082181},
		{497, 34.7082104992009},
		{498, 34.60279005425693},
		{499, 34.231204914439},
		{500, 34.099389551023904},
		{501, 34.22803109888905},
		{502, 34.610384365507166},
		{503, 34.902573683307146},
		{504, 35.0720672663373},
		{505, 35.02095234082352},
		{506, 35.21029830385259},
		{507, 36.39939489896903},
		{508, 36.49626843650683},
		{509, 36.91759867136517},
		{510, 36.62252163480901},
		{511, 36.637052547917165},
		{512, 37.07289132044249},
		{513, 37.943749079996564},
		{514, 37.31678988480502},
		{515, 37.10207023840552},
		{516, 37.50103188910752},
		{517, 36.711556296699854},
		{518, 36.69985201593515},
		{519, 35.88305508619134},
		{520, 35.819177348929244},
		{521, 35.238662967688704},
		{522, 35.204473848884604},
		{523, 35.73171925218891},
		{524, 37.10654099191614},
		{525, 36.97832117698606},
		{526, 37.88944422270623},
		{527, 37.566765945668905},
		{528, 36.952108419930035},
		{529, 37.16172877345256},
		{530, 36.79432449958657},
		{531, 36.83537718548058},
		{532, 37.241795032269486},
		{533, 36.98172720447497},
		{534, 37.12848852369968},
		{535, 37.144201171899056},
		{536, 37.291084012425635},
		{537, 37.20361280452033},
		{538, 37.14673041600966},
		{539, 36.56539045560211},
		{540, 36.46052858989325},
		{541, 36.577035838031236},
		{542, 36.413434939980604},
		{543, 36.830541819855966},
		{544, 36.708601164617306},
		{545, 36.754028252130254},
		{546, 36.86880281045058},
		{547, 36.20522653559122},
		{548, 36.19016853939169},
		{549, 35.60660078018533},
		{550, 35.58380480655832},
		{551, 36.15574897354542},
		{552, 35.96603587272573},
		{553, 36.01846266557714},
		{554, 36.1246059100681},
		{555, 35.776226062342324},
		{556, 35.53756438274776},
		{557, 36.09285691162272},
		{558, 36.024821241563764},
		{559, 36.07242575850486},
		{560, 36.902047436744915},
		{561, 36.783371618493575},
		{562, 36.97312505169229},
		{563, 38.33987731208528},
		{564, 38.12211125979531},
		{565, 37.867344065570286},
		{566, 37.91294220403035},
		{567, 37.90421561454514},
		{568, 37.52932323225818},
		{569, 38.09065950561944},
		{570, 38.3481668904528},
		{571, 38.38853417076884},
		{572, 38.82875488776441},
		{573, 39.30100206552399},
		{574, 39.22872668781665},
		{575, 39.26377992972712},
		{576, 39.11286987148877},
		{577, 38.55996270262097},
		{578, 39.53759984840164},
		{579, 39.434972831723044},
		{580, 39.58456959406775},
		{581, 40.02469156333131},
		{582, 39.83799518416734},
		{583, 41.066310308184526},
		{584, 41.249674158569356},
		{585, 41.84405324758257},
		{586, 42.44197850928179},
		{587, 42.44435586698777},
		{588, 42.51450858652831},
		{589, 42.497079884180856},
		{590, 41.86201338457368},
		{591, 41.97430910917171},
		{592, 42.38604395890263},
		{593, 42.77701512105045},
		{594, 42.917712727399795},
		{595, 41.92733574585986},
		{596, 41.98001270512809},
		{597, 42.21709441156598},
		{598, 41.73101118068132},
		{599, 41.511566903501254},
		{600, 41.151843749654674},
		{601, 40.936735208747976},
		{602, 41.07670022261416},
		{603, 41.00310883262705},
		{604, 41.175657930592564},
		{605, 39.65057441921113},
		{606, 40.59197307871199},
		{607, 40.19328008308628},
		{608, 39.96642122821799},
		{609, 39.980554983282815},
		{610, 39.198802516126015},
		{611, 40.28881956306739},
		{612, 40.835001630650424},
		{613, 40.96840019049191},
		{614, 41.47803284199829},
		{615, 42.31117692587696},
		{616, 42.56501744522036},
		{617, 40.75017426985965},
		{618, 40.54474737773533},
		{619, 40.32299207284596},
		{620, 40.59233541049153},
		{621, 39.799369403233875},
		{622, 39.90740256156506},
		{623, 39.84656589770128},
		{624, 40.51863738080248},
		{625, 40.36927897222197},
		{626, 39.577466020927666},
		{627, 40.24059624379228},
		{628, 39.888654784809695},
		{629, 39.06459566315487},
		{630, 39.21824625899819},
		{631, 39.707949621228416},
		{632, 39.6862655989382},
		{633, 39.232670266668386},
		{634, 38.72226788949448},
		{635, 38.90143970392931},
		{636, 38.58114412526509},
		{637, 37.47764070721714},
		{638, 38.14287868141234},
		{639, 37.95489640397961},
		{640, 37.978479310316544},
		{641, 38.08870157682401},
		{642, 37.86650279398312},
		{643, 37.83158400391423},
		{644, 37.661339861002425},
		{645, 37.77028468310174},
		{646, 38.19873503550377},
		{647, 37.84181306063926},
		{648, 36.15953352691078},
		{649, 36.15954745676219},
		{650, 36.70039654101666},
		{651, 36.747404928048674},
		{652, 36.45440618086992},
		{653, 36.50375344553345},
		{654, 36.12499971175383},
		{655, 36.25540639161984},
		{656, 36.14128341022631},
		{657, 36.13550126300937},
		{658, 36.47636290489859},
		{659, 36.52934581511136},
		{660, 36.180684191314654},
		{661, 36.1345774586021},
		{662, 35.88734753480093},
		{663, 35.910114094303935},
		{664, 37.494624894014834},
		{665, 37.388322481490356},
		{666, 37.51949607161456},
		{667, 37.42672416426224},
		{668, 37.607431895150505},
		{669, 38.1169043451165},
		{670, 38.198701944392226},
		{671, 37.672228180889036},
		{672, 37.573597996143555},
		{673, 37.61475549631909},
		{674, 37.704351969731974},
		{675, 36.526503465131775},
		{676, 37.14283711760895},
		{677, 37.042216595367044},
		{678, 37.19943347657908},
		{679, 37.030753837557455},
		{680, 37.29832109446188},
		{681, 37.34909191845969},
		{682, 37.396667764315495},
		{683, 37.71513970238973},
		{684, 37.035963822400284},
		{685, 37.09132966834279},
		{686, 36.671363246669195},
		{687, 36.725271547538746},
		{688, 36.732765304609515},
		{689, 37.049756926857675},
		{690, 37.100425916489115},
		{691, 37.80849557841213},
		{692, 38.00370168917147},
		{693, 38.65064301128746},
		{694, 38.16137008804746},
		{695, 38.16628690738749},
		{696, 38.509868983636345},
		{697, 38.51329094834429},
		{698, 38.63409846835321},
		{699, 38.33551139578646},
		{700, 37.86178126970245},
		{701, 37.20625176875522},
		{702, 37.13864013016608},
		{703, 36.598956314019325},
		{704, 36.73225170870823},
		{705, 36.645909925459954},
		{706, 36.931378033672665},
		{707, 37.15159698187774},
		{708, 37.20499450218726},
		{709, 37.312534314076736},
		{710, 38.234122213566266},
		{711, 38.23482019038691},
		{712, 39.50821685803287},
		{713, 39.3687472311753},
		{714, 40.43514365409954},
		{715, 40.44682389295688},
		{716, 39.81067959969004},
		{717, 39.4239320658717},
		{718, 39.54620136509768},
		{719, 39.70807547297041},
		{720, 39.66237013592385},
		{721, 39.72023820319751},
		{722, 40.02829005394066},
		{723, 39.9923106822397},
		{724, 39.88229973919716},
		{725, 39.577778632304906},
		{726, 39.42713453659046},
		{727, 39.43664948339558},
		{728, 39.90314312163109},
		{729, 40.00741574107013},
		{730, 40.7177664595581},
		{731, 40.72045099279074},
		{732, 38.3054003174184},
		{733, 37.63394119070095},
		{734, 37.190128597342806},
		{735, 37.63802392396217},
		{736, 37.09659077679385},
		{737, 37.08430082665866},
		{738, 37.031926029911546},
		{739, 36.92762266554352},
		{740, 36.768283960686276},
		{741, 36.80986118421022},
		{742, 37.00023584840072},
		{743, 37.37757227329164},
		{744, 37.65237108287256},
		{745, 37.65431119266612},
		{746, 37.78202911987313},
		{747, 37.26755576516154},
		{748, 37.253878362362926},
		{749, 37.25181149353707},
		{750, 38.046853987139436},
		{751, 38.52088550994239},
		{752, 38.50060736503108},
		{753, 39.721168829902766},
		{754, 39.714977806533284},
		{755, 41.2027128451164},
		{756, 41.271353835056736},
		{757, 41.42756394972893},
		{758, 41.20791889996719},
		{759, 41.51542890247571},
		{760, 41.533271125006856},
		{761, 41.57431095381387},
		{762, 42.43297879649776},
		{763, 42.697486616671064},
		{764, 42.943817792210965},
		{765, 43.00326844639773},
		{766, 43.04122373477804},
		{767, 42.72385932125831},
		{768, 42.91563938083615},
		{769, 43.09789890970447},
		{770, 43.21939147903337},
		{771, 42.593500985806116},
		{772, 42.552074422506266},
		{773, 41.96625047640962},
		{774, 42.43760021996765},
		{775, 42.15036608724706},
		{776, 42.01801170238841},
		{777, 42.108251206065354},
		{778, 42.13720797898185},
		{779, 41.734510693754125},
		{780, 41.98861197273539},
		{781, 41.855890804590906},
		{782, 41.83653956517087},
		{783, 41.84729249879647},
		{784, 41.34451591903922},
		{785, 41.198937855641844},
		{786, 40.9934126996872},
		{787, 40.33193886547389},
		{788, 40.42177093298795},
		{789, 40.56423832118387},
		{790, 40.56930078212437},
		{791, 40.44001792153413},
		{792, 39.36239492015202},
		{793, 39.2299512438198},
		{794, 39.98930528782557},
		{795, 39.78753261124546},
		{796, 39.7771218730521},
		{797, 40.08078557739916},
		{798, 39.9026881045777},
		{799, 39.846598822934695},
		{800, 39.819165988548605},
		{801, 39.779536346223246},
		{802, 39.417184051357914},
		{803, 39.04248228046798},
		{804, 39.04467471746209},
		{805, 39.19976668706493},
		{806, 38.62608414650186},
		{807, 38.56950606795273},
		{808, 38.11427821927686},
		{809, 38.003564811967195},
		{810, 38.9186775339263},
		{811, 39.32280711240079},
		{812, 39.37066040646152},
		{813, 39.35771550162152},
		{814, 39.82593927687213},
		{815, 39.41511912781052},
		{816, 39.632089125382805},
		{817, 39.7013716770543},
		{818, 39.71871324205181},
		{819, 39.3691037472924},
		{820, 39.4630528375144},
		{821, 39.463557590573124},
		{822, 39.84149848240294},
		{823, 39.731112944493475},
		{824, 39.46553273230596},
		{825, 38.983001317774516},
		{826, 39.111806633972954},
		{827, 39.11241986582127},
		{828, 39.060815486719235},
		{829, 38.62106612409818},
		{830, 38.45726527644363},
		{831, 38.46243114268653},
		{832, 37.81089603641852},
		{833, 37.06489698994548},
		{834, 35.65801215219413},
		{835, 35.598044467413054},
		{836, 35.51269552874315},
		{837, 35.26479970268285},
		{838, 36.55531335895278},
		{839, 36.707855369730034},
		{840, 36.5489032234404},
		{841, 36.3485272462068},
		{842, 36.365162732119266},
		{843, 36.175883792606705},
		{844, 36.16643352096541},
		{845, 36.721302046157795},
		{846, 36.738751647960406},
		{847, 36.499637035175965},
		{848, 36.461882187600125},
		{849, 36.474313661197286},
		{850, 36.138203284641186},
		{851, 36.47458889999187},
		{852, 36.9122685068765},
		{853, 36.84193719836966},
		{854, 36.016869149964144},
		{855, 35.908626743361204},
		{856, 35.853057192524155},
		{857, 35.49400703601318},
		{858, 35.23328368215224},
		{859, 35.43381027728788},
		{860, 35.441912396566295},
		{861, 36.11980562386126},
		{862, 35.987081872648865},
		{863, 35.354742046128464},
		{864, 35.11437170180045},
		{865, 35.10075744627144},
		{866, 35.33061979174065},
		{867, 35.26826901332004},
		{868, 34.76420766370852},
		{869, 34.36825504722903},
		{870, 33.95310090031108},
		{871, 34.76854906618263},
		{872, 34.96941220214851},
		{873, 34.89082477498721},
		{874, 34.930226673548425},
		{875, 34.818344629526955},
		{876, 33.46752591731415},
		{877, 32.4461932776035},
		{878, 32.63093881621515},
		{879, 32.571549274273316},
		{880, 31.70273743751098},
		{881, 31.560570134656707},
		{882, 30.11896242237412},
		{883, 29.976753491813355},
		{884, 30.151470049420453},
		{885, 29.968667652811302},
		{886, 29.914108861900175},
		{887, 29.315297425726456},
		{888, 29.465096537128055},
		{889, 30.38412313034993},
		{890, 30.529005936656326},
		{891, 30.377111661676715},
		{892, 30.690112657456975},
		{893, 31.19665096563887},
		{894, 30.23180166165202},
		{895, 30.214575260830543},
		{896, 30.61887791868095},
		{897, 31.69516453152183},
		{898, 31.724657127738052},
		{899, 31.3260343161325},
		{900, 31.448576476567858},
		{901, 31.384197051367188},
		{902, 31.42382604514501},
		{903, 31.77933470325809},
		{904, 31.54527210384609},
		{905, 31.662859065404803},
		{906, 31.952233276070345},
		{907, 31.710429406503827},
		{908, 31.69910928793987},
		{909, 31.26054806750292},
		{910, 31.252913740901814},
		{911, 31.069356216279907},
		{912, 31.450376052945543},
		{913, 31.446602308578885},
		{914, 32.88848724097737},
		{915, 32.97405474992589},
		{916, 32.95989543002121},
		{917, 33.50228160406862},
		{918, 33.57376811625785},
		{919, 33.49455873288756},
		{920, 32.739675795181945},
		{921, 32.69683391724602},
		{922, 33.81401934792278},
		{923, 33.68504119872676},
		{924, 33.08473835369315},
		{925, 32.71783936639953},
		{926, 33.576629314879405},
		{927, 33.496669377208725},
		{928, 32.32323980202774},
		{929, 32.19540083945545},
		{930, 31.681562294745188},
		{931, 31.21936553988134},
		{932, 32.02350301138198},
		{933, 31.99391696064781},
		{934, 31.222294641434715},
		{935, 31.49393327555392},
		{936, 31.60562877897029},
		{937, 31.719747501964054},
		{938, 31.631337228109494},
		{939, 31.876599195053196},
		{940, 31.839975677295737},
		{941, 31.57998061624259},
		{942, 32.247156373569744},
		{943, 33.13453225404717},
		{944, 33.57148226581986},
		{945, 33.67881606804257},
		{946, 34.27627744671606},
		{947, 34.2667236377486},
		{948, 35.0394972266412},
		{949, 35.06221556029825},
		{950, 35.317831610972426},
		{951, 35.47650956265658},
		{952, 35.92884652798027},
		{953, 36.40171114341679},
		{954, 35.651733034914436},
		{955, 36.021038886103796},
		{956, 36.194620959224366},
		{957, 36.14405991221338},
		{958, 36.02271118328351},
		{959, 36.01617792703485},
		{960, 36.718644368420634},
		{961, 36.658954226071046},
		{962, 36.58675202460727},
		{963, 35.89078679017791},
		{964, 35.94749384377098},
		{965, 35.92838317262612},
		{966, 34.32490064912566},
		{967, 33.68329350874763},
		{968, 33.72780060374807},
		{969, 33.521931400031036},
		{970, 33.48546378484414},
		{971, 33.729671379093105},
		{972, 32.483138764110244},
		{973, 32.70061143845538},
		{974, 31.882058598338592},
		{975, 31.85978103716163},
		{976, 31.360355110091252},
		{977, 31.338506662447838},
		{978, 31.4975196000238},
		{979, 31.636516566035457},
		{980, 31.76928554942759},
		{981, 31.596565826225252},
		{982, 31.24504603522091},
		{983, 30.97685607999551},
		{984, 30.776149614114455},
		{985, 30.369022511891757},
		{986, 30.65823464304316},
		{987, 30.536961423755283},
		{988, 30.543171096017304},
		{989, 29.709033348016842},
		{990, 29.787775153315135},
		{991, 30.020098011459627},
		{992, 30.161570003700696},
		{993, 30.110594653721783},
		{994, 30.269521765071406},
		{995, 30.887702552351005},
		{996, 31.092501292990715},
		{997, 30.96235060201595},
		{998, 31.371052725132344},
		{999, 31.652461552974707},
		{1000, 31.925382030352395},
		{1001, 32.09718279718192},
		{1002, 32.404705995647134},
		{1003, 32.40399854679425},
		{1004, 32.105619052543034},
		{1005, 32.123902211392696},
		{1006, 32.26894930916644},
		{1007, 32.06571185804976},
		{1008, 32.27138533885846},
		{1009, 32.459468377391936},
		{1010, 32.45637104295923},
		{1011, 33.97355416481095},
		{1012, 34.073372443262755},
		{1013, 34.555509854599876},
		{1014, 34.001992433795586},
		{1015, 34.028775375039515},
		{1016, 33.15444517016032},
		{1017, 33.25999551152861},
		{1018, 33.56889630842039},
		{1019, 33.836008073592865},
		{1020, 34.6557254577998},
		{1021, 34.16915786529838},
		{1022, 33.91355886515464},
		{1023, 33.53814944935212},
		{1024, 33.73093246034526},
		{1025, 33.779257221704746},
		{1026, 33.55901656076105},
		{1027, 33.522411543634334},
		{1028, 33.308377022855574},
		{1029, 33.46872487000558},
		{1030, 33.398428332372106},
		{1031, 32.78130678002823},
		{1032, 32.569113758452666},
		{1033, 31.58571790720343},
		{1034, 32.16172427938117},
		{1035, 31.661979621179476},
		{1036, 31.632175065844496},
		{1037, 31.789546741031184},
		{1038, 32.26513372369237},
		{1039, 32.18304328830448},
		{1040, 32.71362639662142},
		{1041, 31.299268596529224},
		{1042, 27.792537058477052},
		{1043, 31.385259077931753},
		{1044, 30.591065858607116},
		{1045, 31.272338876555935},
		{1046, 31.673417165689862},
		{1047, 31.185485416617453},
		{1048, 31.29116101415942},
		{1049, 31.420045699024275},
		{1050, 32.72027856671056},
		{1051, 32.70728464282712},
		{1052, 32.84488754596194},
		{1053, 33.19971370723099},
		{1054, 33.00087402494973},
		{1055, 32.99077340899255},
		{1056, 32.83232129107784},
		{1057, 32.30498262311541},
		{1058, 32.30946094376116},
		{1059, 32.940891339151776},
		{1060, 32.68854192681414},
		{1061, 32.73363409330329},
		{1062, 32.79589758581314},
		{1063, 32.289791532294295},
		{1064, 32.68604720099993},
		{1065, 33.33078243465505},
		{1066, 33.54826648297002},
		{1067, 33.40865566016796},
		{1068, 33.41296471315078},
		{1069, 32.77182145012062},
		{1070, 32.33224727266914},
		{1071, 32.27203157051601},
		{1072, 32.26109905266389},
		{1073, 32.921499096795074},
		{1074, 33.38395175893338},
		{1075, 33.78768855190261},
		{1076, 33.442367223026764},
		{1077, 33.237913854994794},
		{1078, 31.147232293821737},
		{1079, 31.146482126046756},
		{1080, 31.71150780117646},
		{1081, 31.814296312484636},
		{1082, 30.955859429080363},
		{1083, 30.987259649137073},
		{1084, 30.89065910781576},
		{1085, 31.16579586461409},
		{1086, 31.118733218136573},
		{1087, 30.533273297600925},
		{1088, 29.252834071159857},
		{1089, 29.685743434805644},
		{1090, 29.705920523322913},
		{1091, 29.85069842502032},
		{1092, 30.388305038615574},
		{1093, 30.161758590806624},
		{1094, 30.779652282941164},
		{1095, 30.91190072880913},
		{1096, 31.56528710281942},
		{1097, 31.768065818580393},
		{1098, 31.8230334699979},
		{1099, 32.1658225590106},
		{1100, 32.18402247418236},
		{1101, 33.487267406897004},
		{1102, 33.51902089155666},
		{1103, 33.39833431034105},
		{1104, 32.50713483623776},
		{1105, 32.66530621931014},
		{1106, 32.78927126650209},
		{1107, 32.650964726546256},
		{1108, 32.70706148431291},
		{1109, 32.916943844199004},
		{1110, 32.49004088753038},
		{1111, 32.945619481604744},
		{1112, 33.03101393088018},
		{1113, 33.01159487630862},
		{1114, 33.17414749212253},
		{1115, 34.21219988578758},
		{1116, 34.00043238845383},
		{1117, 33.61923485510282},
		{1118, 34.275948039809364},
		{1119, 33.87944072377637},
		{1120, 33.57361872056542},
		{1121, 33.60777850243776},
		{1122, 33.95519829789812},
		{1123, 34.101866130091636},
		{1124, 35.08344176854378},
		{1125, 35.290045657998256},
		{1126, 34.99789523160777},
		{1127, 35.11162512342385},
		{1128, 35.150203027075506},
		{1129, 34.9547553234651},
		{1130, 35.68267560816188},
		{1131, 35.94530333733083},
		{1132, 36.40684420002165},
		{1133, 36.90897436886569},
		{1134, 36.68925056413046},
		{1135, 36.77241779692453},
		{1136, 36.7768373962142},
		{1137, 36.615559524302526},
		{1138, 36.15108571210806},
		{1139, 36.31887166999472},
		{1140, 36.25359696272827},
		{1141, 36.04048635034041},
		{1142, 36.00631287066912},
		{1143, 36.459483786091326},
		{1144, 36.523711914237204},
		{1145, 35.76502964124057},
		{1146, 36.18603634547918},
		{1147, 36.143227345894964},
		{1148, 35.42846972819206},
		{1149, 34.22551552384586},
		{1150, 35.085356710729094},
		{1151, 34.978408345953554},
		{1152, 35.02162912627897},
		{1153, 34.6749544956021},
		{1154, 34.5294083257531},
		{1155, 33.989361791250865},
		{1156, 33.90560409554212},
		{1157, 34.22819180059112},
		{1158, 35.09979866464593},
		{1159, 35.459597394271825},
		{1160, 35.028413870661275},
		{1161, 35.07365163864229},
		{1162, 35.008455674695846},
		{1163, 34.87027987153975},
		{1164, 32.73740982908738},
		{1165, 35.86288920810587},
		{1166, 36.38448730256452},
		{1167, 35.69198792205293},
		{1168, 35.7287802068078},
		{1169, 35.67047911791235},
		{1170, 35.65931562676154},
		{1171, 35.63860255804954},
		{1172, 36.28377914027185},
		{1173, 36.31724433582806},
		{1174, 36.42873315721097},
		{1175, 36.55293797881833},
		{1176, 36.68858364718202},
		{1177, 36.68542383419321},
		{1178, 36.59630640724288},
		{1179, 36.46919978323624},
		{1180, 35.97612499492244},
		{1181, 35.97470028204401},
		{1182, 36.88186754506275},
		{1183, 36.89180762556425},
		{1184, 37.398624973193115},
		{1185, 37.6485574658129},
		{1186, 37.59944793075436},
		{1187, 37.52123044476694},
		{1188, 37.545545238118},
		{1189, 37.63568398432097},
		{1190, 37.60117292673316},
		{1191, 37.76737303107856},
		{1192, 37.796708933792935},
		{1193, 38.44788884052119},
		{1194, 38.433526334130406},
		{1195, 38.53261361058563},
		{1196, 38.20625193191746},
		{1197, 38.216892858955696},
		{1198, 38.10624817074387},
		{1199, 38.31498368822061},
		{1200, 38.28171670233931},
		{1201, 38.03504598890333},
		{1202, 37.469381149070585},
		{1203, 37.645843865994834},
		{1204, 37.996863546861626},
		{1205, 37.39010863231819},
		{1206, 37.916484620097826},
		{1207, 37.9792008265341},
		{1208, 37.56402382212345},
		{1209, 37.64236828858301},
		{1210, 38.20048604682538},
		{1211, 38.1135776386156},
		{1212, 38.06926736333271},
		{1213, 37.85109980183144},
		{1214, 37.8667193253515},
		{1215, 37.79731989239691},
		{1216, 37.79569290509785},
		{1217, 37.507680665850124},
		{1218, 37.86538998455543},
		{1219, 37.97413406343451},
		{1220, 38.153189871670634},
		{1221, 37.705945919911386},
		{1222, 38.003018638410715},
		{1223, 37.97804975659373},
		{1224, 37.86005819640923},
		{1225, 37.74622347438676},
		{1226, 37.24273013769576},
		{1227, 37.471847978186396},
		{1228, 36.574577925568015},
		{1229, 36.898112403986175},
		{1230, 37.93262825928585},
		{1231, 37.98350758652906},
		{1232, 37.3327630080204},
		{1233, 37.87253628352105},
		{1234, 37.03465628886385},
		{1235, 37.3628779234913},
		{1236, 37.257920892232235},
		{1237, 37.355020597223806},
		{1238, 36.42612381389829},
		{1239, 35.71849960760021},
		{1240, 35.569737001180215},
		{1241, 35.639328778372764},
		{1242, 36.67505414760637},
		{1243, 36.529877025906195},
		{1244, 36.17805747298349},
		{1245, 36.20730061152778},
		{1246, 36.7654190000416},
		{1247, 36.066963570873966},
		{1248, 34.80315462413555},
		{1249, 34.79204497446974},
		{1250, 34.96282416930701},
		{1251, 35.29525328571966},
		{1252, 35.39276674967225},
		{1253, 35.17789113339765},
		{1254, 35.1734239698592},
		{1255, 36.27186268761327},
		{1256, 36.12969032844646},
		{1257, 35.56797468172719},
		{1258, 35.413952983479305},
		{1259, 35.473554398381665},
		{1260, 34.959780400150386},
		{1261, 34.705612847987084},
		{1262, 35.122081505219015},
		{1263, 35.54647234411569},
		{1264, 35.56585165590874},
		{1265, 34.07163330273662},
		{1266, 34.061764136955},
		{1267, 34.98718482740182},
		{1268, 35.194478377248124},
		{1269, 34.814779163097306},
		{1270, 35.63339066284622},
		{1271, 36.248018614293784},
		{1272, 35.89931722172987},
		{1273, 36.02707673294714},
		{1274, 35.47928825105021},
		{1275, 35.339226659095814},
		{1276, 36.04365822060287},
		{1277, 36.12975065194946},
		{1278, 35.657328019340454},
		{1279, 35.92456974548614},
		{1280, 36.330656914157935},
		{1281, 36.192088120039486},
		{1282, 35.48734879533039},
		{1283, 35.20967589696835},
		{1284, 35.16528283177774},
		{1285, 35.05450151673027},
		{1286, 34.083660774124404},
		{1287, 33.772468853879964},
		{1288, 33.29191351706884},
		{1289, 33.40417208662001},
		{1290, 32.607679572698984},
		{1291, 32.60800666862871},
		{1292, 32.6060770265256},
		{1293, 32.50260313437235},
		{1294, 32.66708354095242},
		{1295, 32.66682027747426},
		{1296, 33.0078376785496},
		{1297, 32.79448909601},
		{1298, 33.33055451566498},
		{1299, 33.80720872380922},
		{1300, 34.03662018610426},
		{1301, 33.471561770605334},
		{1302, 33.464542278200675},
		{1303, 33.5135192502925},
		{1304, 33.522105679079196},
		{1305, 33.486732091763336},
		{1306, 34.102696294452976},
		{1307, 34.28410402104561},
		{1308, 34.50821094970082},
		{1309, 34.2800121277366},
		{1310, 34.07699583444763},
		{1311, 33.29032898991077},
		{1312, 33.447416548419994},
		{1313, 33.07070289905697},
		{1314, 32.76441433179846},
		{1315, 32.35364095869243},
		{1316, 32.19453136744108},
		{1317, 32.001227569765504},
		{1318, 32.1893553685659},
		{1319, 32.17192961357054},
		{1320, 32.05156175088731},
		{1321, 32.01319521995405},
		{1322, 31.361540797597904},
		{1323, 31.358737048716332},
		{1324, 31.4094163556713},
		{1325, 31.385674150907523},
		{1326, 30.397821572277874},
		{1327, 30.430993318308072},
		{1328, 30.559151734617863},
		{1329, 30.709052945784077},
		{1330, 30.762480629193092},
		{1331, 30.428823245671445},
		{1332, 30.549369360601233},
		{1333, 30.401524153538364},
		{1334, 30.862514784963658},
		{1335, 30.543179751483258},
		{1336, 30.546876105280212},
		{1337, 30.8118556608746},
		{1338, 30.73308277586123},
		{1339, 31.09586849152352},
		{1340, 30.443543543528172},
		{1341, 30.574438626388865},
		{1342, 30.433112514559557},
		{1343, 29.60468065056554},
		{1344, 29.518027759834045},
		{1345, 29.530716732475156},
		{1346, 29.258574444161773},
		{1347, 28.96213974116116},
		{1348, 30.094589411608464},
		{1349, 30.027880530088794},
		{1350, 30.00173657659883},
		{1351, 30.7570211333231},
		{1352, 30.905455874138642},
		{1353, 30.95776014140416},
		{1354, 31.25924961163502},
		{1355, 31.28781614068638},
		{1356, 31.259224092967646},
		{1357, 31.471490532081248},
		{1358, 30.66993696457886},
		{1359, 29.51601931381402},
		{1360, 29.658359527955362},
		{1361, 29.498776144228817},
		{1362, 29.0952884352307},
		{1363, 27.583844233928247},
		{1364, 27.104268459905846},
		{1365, 25.95144210780829},
		{1366, 25.823464503416666},
		{1367, 25.72514162857761},
		{1368, 25.654123815674225},
		{1369, 26.051979147188657},
		{1370, 25.841597348616787},
		{1371, 25.03088000259754},
		{1372, 24.40430526113091},
		{1373, 24.39450861715055},
		{1374, 24.868258110961357},
		{1375, 24.87891945990089},
		{1376, 24.826829203959203},
		{1377, 24.723127466527284},
		{1378, 25.189101108151846},
		{1379, 24.999892026613335},
		{1380, 25.403952465657824},
		{1381, 25.395818912419763},
		{1382, 25.361488701260853},
		{1383, 25.168837708367573},
		{1384, 25.822974064286573},
		{1385, 25.184633998151693},
		{1386, 25.673392593447062},
		{1387, 25.493115319468433},
		{1388, 25.616885562952366},
		{1389, 26.26185502334556},
		{1390, 26.63671721716633},
		{1391, 27.41659614028529},
		{1392, 27.180001027499763},
		{1393, 27.31191586358311},
		{1394, 27.41516507193562},
		{1395, 27.621542573182392},
		{1396, 28.333329103903207},
		{1397, 28.490364050694232},
		{1398, 28.282737229661876},
		{1399, 28.487368411327616},
		{1400, 27.619991904229085},
		{1401, 27.775228746290942},
		{1402, 27.75784788858771},
		{1403, 26.388008503362745},
		{1404, 26.390446759456594},
		{1405, 26.804841863728445},
		{1406, 26.645161583486924},
		{1407, 25.978560150820044},
		{1408, 25.928009125329705},
		{1409, 25.928237301882948},
		{1410, 25.750677096758384},
		{1411, 25.3386270250864},
		{1412, 25.140774560316675},
		{1413, 24.34074260607449},
		{1414, 24.754961425981048},
		{1415, 24.730194413283783},
		{1416, 24.920450795877183},
		{1417, 25.024801997124456},
		{1418, 24.6651943693905},
		{1419, 25.194960339720506},
		{1420, 25.34229435510204},
		{1421, 25.305380825552025},
		{1422, 25.412841036914795},
		{1423, 25.505668659821385},
		{1424, 25.930434241278466},
		{1425, 25.958267633145237},
		{1426, 26.031451433938873},
		{1427, 26.586990196306772},
		{1428, 27.292096678546},
		{1429, 28.12071488383503},
		{1430, 28.02832144500101},
		{1431, 28.58386148422691},
		{1432, 28.79404266374261},
		{1433, 29.738295917981866},
		{1434, 29.720007389392677},
		{1435, 29.63633778940114},
		{1436, 29.68841036823205},
		{1437, 29.575033122960512},
		{1438, 29.539149897548384},
		{1439, 29.80686249903287},
		{1440, 30.862182502538953},
		{1441, 30.42493141449586},
		{1442, 30.427703473898646},
		{1443, 30.502760352966796},
		{1444, 30.537509584235174},
		{1445, 30.206448675310977},
		{1446, 30.230172108384416},
		{1447, 30.164364772985248},
		{1448, 29.417259389108583},
		{1449, 29.280495127154293},
		{1450, 29.331776246449813},
		{1451, 28.98662862990845},
		{1452, 29.752510306571484},
		{1453, 29.70066010243845},
		{1454, 29.357272975382973},
		{1455, 29.25016590767666},
		{1456, 29.355245117563307},
		{1457, 29.533981898101043},
		{1458, 29.569017902727573},
		{1459, 29.611976373457114},
		{1460, 29.731805290138638},
		{1461, 29.757994213197602},
		{1462, 30.0901622202274},
		{1463, 31.279393484956277},
		{1464, 31.228915684424805},
		{1465, 31.39987446802975},
		{1466, 30.81523244839042},
		{1467, 31.19319811379786},
		{1468, 31.21869066135987},
		{1469, 31.378064726660238},
		{1470, 31.36503448737635},
		{1471, 30.890723656435668},
		{1472, 31.278271018448063},
		{1473, 31.27351266944355},
		{1474, 30.906433524045895},
		{1475, 31.094199381637562},
		{1476, 32.22446329349785},
		{1477, 31.755676495678124},
		{1478, 31.863329580086756},
		{1479, 34.1769485077606},
		{1480, 33.67109006952487},
		{1481, 34.032560599893785},
		{1482, 35.17463593455005},
		{1483, 34.68680491085987},
		{1484, 35.34323186904023},
		{1485, 36.06846197316916},
		{1486, 35.75474567916718},
		{1487, 34.95879681397107},
		{1488, 34.91513272089497},
		{1489, 34.27429545432343},
		{1490, 34.47677587538645},
		{1491, 34.8837191993053},
		{1492, 34.888682229883216},
		{1493, 34.6615444728743},
		{1494, 34.691697874937915},
		{1495, 34.673137205747096},
		{1496, 34.619588001105676},
		{1497, 34.67924802646811},
		{1498, 34.711467226688804},
		{1499, 34.378496298766606},
		{1500, 34.01906505065623},
		{1501, 34.1921149855785},
		{1502, 34.645939695035565},
		{1503, 34.55325778595661},
		{1504, 34.837013970080854},
		{1505, 34.3201819146262},
		{1506, 34.21433963893517},
		{1507, 34.209807913410046},
		{1508, 34.29503290261106},
		{1509, 34.372533557964495},
		{1510, 34.29249674028179},
		{1511, 34.1793952178106},
		{1512, 34.23027204162766},
		{1513, 34.30024655177506},
		{1514, 34.29076290991813},
		{1515, 34.872527881393644},
		{1516, 34.88183138613434},
		{1517, 34.99793065640857},
		{1518, 35.116718830483286},
		{1519, 35.11046533465197},
		{1520, 36.014438556584366},
		{1521, 36.02360905142489},
		{1522, 35.91482043384545},
		{1523, 35.809274512011065},
		{1524, 35.59619773000511},
		{1525, 35.44019543122139},
		{1526, 35.64144780604629},
		{1527, 35.58080982359661},
		{1528, 35.452624932745614},
		{1529, 35.467238834711935},
		{1530, 35.041073967062886},
		{1531, 34.27022167037798},
		{1532, 33.77867219668047},
		{1533, 33.675446148880425},
		{1534, 34.29649579312852},
		{1535, 33.80063094230085},
		{1536, 33.83706673096555},
		{1537, 34.19813699318839},
		{1538, 34.67898341154741},
		{1539, 34.61565170828406},
		{1540, 34.88435222539337},
		{1541, 34.74803849982474},
		{1542, 34.88836986733776},
		{1543, 34.316664346866176},
		{1544, 35.1429379016867},
		{1545, 35.236518386423754},
		{1546, 35.480022507695814},
		{1547, 35.51129848892317},
		{1548, 35.55074228557621},
		{1549, 35.47965462275921},
		{1550, 35.1614123321548},
		{1551, 34.649935916461956},
		{1552, 33.831408529297455},
		{1553, 33.68659149830347},
		{1554, 34.76348070802675},
		{1555, 34.75436175642519},
		{1556, 34.670519236256325},
		{1557, 34.40743801250581},
		{1558, 35.03821600965653},
		{1559, 35.074059881857586},
		{1560, 34.66313179952048},
		{1561, 34.05122520128333},
		{1562, 34.369261269504776},
		{1563, 34.306862410511414},
		{1564, 34.35513437341248},
		{1565, 34.42016699549108},
		{1566, 34.32411882750936},
		{1567, 35.10444315366103},
		{1568, 35.21667415434252},
		{1569, 34.56235394065422},
		{1570, 34.99569538327316},
		{1571, 35.051462529787685},
		{1572, 35.65901988130965},
		{1573, 35.30919470315533},
		{1574, 35.43427648307305},
		{1575, 35.26105873637407},
		{1576, 34.984210066219426},
		{1577, 34.974915529232504},
		{1578, 34.64250053081252},
		{1579, 34.855051142104344},
		{1580, 34.61935447447808},
		{1581, 33.30095950674964},
		{1582, 33.52159175029287},
		{1583, 33.30245847520868},
		{1584, 33.22017010546928},
		{1585, 33.302509858225584},
		{1586, 33.37904181575153},
		{1587, 33.25298440074963},
		{1588, 33.25372890621039},
		{1589, 33.55633290212615},
		{1590, 33.564240306946566},
		{1591, 33.77554531353173},
		{1592, 33.77747427631988},
		{1593, 33.758375775192526},
		{1594, 34.36417258641864},
		{1595, 34.31644882079321},
		{1596, 33.55182639869085},
		{1597, 33.56219047110169},
		{1598, 33.44110230199712},
		{1599, 33.37853776463871},
		{1600, 32.03520633056414},
		{1601, 32.091659309762015},
		{1602, 32.3728274207352},
		{1603, 32.32296532348558},
		{1604, 32.248876392188286},
		{1605, 32.2486900311155},
		{1606, 32.30827587355294},
		{1607, 31.55600150552031},
		{1608, 32.275031428073206},
		{1609, 32.27879842764017},
		{1610, 32.12685506952653},
		{1611, 32.04232751622579},
		{1612, 32.3717511617595},
		{1613, 32.90499426727991},
		{1614, 31.34160341506861},
		{1615, 32.79935984939758},
		{1616, 33.3133749284244},
		{1617, 33.330066949919654},
		{1618, 33.6696207424472},
		{1619, 33.44727416894953},
		{1620, 32.937035354917406},
		{1621, 32.42098193976603},
		{1622, 32.40919020921572},
		{1623, 32.40926615430823},
		{1624, 31.91939496950345},
		{1625, 31.745665996112034},
		{1626, 31.762245569926346},
		{1627, 31.76510278674478},
		{1628, 32.18895805255132},
		{1629, 32.17884231058142},
		{1630, 31.634810462741328},
		{1631, 31.569967535436145},
		{1632, 32.29578832893024},
		{1633, 32.930732352651475},
		{1634, 33.03934988366926},
		{1635, 33.21635409643862},
		{1636, 33.261326384884754},
		{1637, 33.39655980314689},
		{1638, 35.11945078307172},
		{1639, 35.076572468626274},
		{1640, 34.95856618844892},
		{1641, 34.85019550097897},
		{1642, 34.70787112679453},
		{1643, 34.64396731467868},
		{1644, 34.03523254807719},
		{1645, 34.599545249438016},
		{1646, 34.911136115744284},
		{1647, 35.12923123083873},
		{1648, 34.19771898956377},
		{1649, 34.081163717338875},
		{1650, 33.952021163589286},
		{1651, 34.03909563960654},
		{1652, 33.826004075423334},
		{1653, 33.69501183562262},
		{1654, 33.54101912632574},
		{1655, 33.534398838396605},
		{1656, 33.551479085175345},
		{1657, 34.17014679283176},
		{1658, 33.68298225679782},
		{1659, 33.914956135823154},
		{1660, 33.85350696847641},
		{1661, 33.5930418617416},
		{1662, 33.698585740003914},
		{1663, 33.53164709646962},
		{1664, 32.942749249597995},
		{1665, 32.78927293361512},
		{1666, 32.805622975726415},
		{1667, 32.678834731539396},
		{1668, 32.49005865138826},
		{1669, 32.45598674269622},
		{1670, 32.38305018611963},
		{1671, 32.39849660167694},
		{1672, 33.26082532734355},
		{1673, 33.32014170105489},
		{1674, 33.73004210039798},
		{1675, 33.724298776862106},
		{1676, 33.68423539254731},
		{1677, 33.563159682253556},
		{1678, 33.46610598060115},
		{1679, 33.46576401188156},
		{1680, 33.6082093792797},
		{1681, 34.06829829976045},
		{1682, 33.78773476628623},
		{1683, 33.847057378024175},
		{1684, 35.250271418221374},
		{1685, 35.09767253763393},
		{1686, 35.074861457363895},
		{1687, 35.40140497372611},
		{1688, 36.31807881704327},
		{1689, 37.93153971576717},
		{1690, 37.96917757654617},
		{1691, 38.22198787034048},
		{1692, 38.23678368537367},
		{1693, 38.20599590804662},
		{1694, 38.68369317469104},
		{1695, 38.79215511609461},
		{1696, 38.802109502845106},
		{1697, 38.753495704229714},
		{1698, 38.96421874631208},
		{1699, 38.90821233250077},
		{1700, 39.16182425119748},
		{1701, 38.853831057337715},
		{1702, 38.63611099293661},
		{1703, 38.24474852988975},
		{1704, 38.38469009174058},
		{1705, 38.32872094403988},
		{1706, 38.153737338484326},
		{1707, 38.264923332806745},
		{1708, 38.280563758044565},
		{1709, 38.18750466726223},
		{1710, 38.56983066481024},
		{1711, 39.505085657555355},
		{1712, 39.69872974322229},
		{1713, 40.39204144577866},
		{1714, 40.48553785053598},
		{1715, 41.3269897150344},
		{1716, 41.609581527655735},
		{1717, 41.65892696057186},
		{1718, 41.837313217148804},
		{1719, 41.8289668854985},
		{1720, 42.15881862704277},
		{1721, 42.73710053588433},
		{1722, 42.348456207492845},
		{1723, 42.096157824123665},
		{1724, 41.92253556951924},
		{1725, 41.917137854036646},
		{1726, 41.97608613143687},
		{1727, 41.72574431293022},
		{1728, 41.24861627279633},
		{1729, 41.87472943187815},
		{1730, 41.87568170648001},
		{1731, 42.17909911414902},
		{1732, 42.049177661936255},
		{1733, 41.97474828244962},
		{1734, 41.356541003497355},
		{1735, 41.22195929777421},
		{1736, 41.194030410051106},
		{1737, 41.392479793475935},
		{1738, 41.34843546848941},
		{1739, 41.34402528705148},
		{1740, 42.07661962525981},
		{1741, 41.52647896910617},
		{1742, 41.44561097861686},
		{1743, 40.958470191837534},
		{1744, 41.037829903821645},
		{1745, 41.07205195864329},
		{1746, 41.46906637533652},
		{1747, 41.49343224957735},
		{1748, 40.610182804195105},
		{1749, 41.017801824810185},
		{1750, 40.79059366319508},
		{1751, 40.70441966049171},
		{1752, 40.70503981535662},
		{1753, 39.93683574905368},
		{1754, 39.752239396736506},
		{1755, 39.7645522699645},
		{1756, 39.55588740392734},
		{1757, 39.74727156272358},
		{1758, 39.9737782466988},
		{1759, 40.39693575020464},
		{1760, 40.216674372755165},
		{1761, 40.25136763580497},
		{1762, 39.38556407545778},
		{1763, 39.26119592497736},
		{1764, 39.16195286452751},
		{1765, 38.86948923485995},
		{1766, 39.61600974311452},
		{1767, 39.39522112770508},
		{1768, 39.514089866690526},
		{1769, 39.45799756858736},
		{1770, 38.90570527579156},
		{1771, 38.80478381534035},
		{1772, 38.83322167157176},
		{1773, 39.64297039815119},
		{1774, 40.33223301684341},
		{1775, 40.41694728495109},
		{1776, 40.06020383799442},
		{1777, 40.07361881200165},
		{1778, 39.77916448556436},
		{1779, 39.96893684870016},
		{1780, 40.57894081959218},
		{1781, 39.89797496618613},
		{1782, 39.35261558123242},
		{1783, 39.36008989987693},
		{1784, 38.0512977550727},
		{1785, 38.264000581925124},
		{1786, 38.26169623365976},
		{1787, 39.10252385765842},
		{1788, 39.27088966509622},
		{1789, 39.15216453676914},
		{1790, 39.41049327702329},
		{1791, 39.39907719439436},
		{1792, 38.82328669157878},
		{1793, 39.35804609496024},
		{1794, 39.52870877903105},
		{1795, 40.562208982859616},
		{1796, 41.015089776340226},
		{1797, 40.87083252184389},
		{1798, 40.53994999420047},
		{1799, 41.02575733506004},
		{1800, 41.17619373256442},
		{1801, 41.19685023564847},
		{1802, 40.98189573065346},
		{1803, 41.159763360041595},
		{1804, 41.59421552557119},
		{1805, 41.20806640088285},
		{1806, 40.414699831836835},
		{1807, 39.92415966374783},
		{1808, 39.62606546104271},
		{1809, 39.857180263975444},
		{1810, 39.59080107893363},
		{1811, 39.6974862276464},
		{1812, 39.442229052918876},
		{1813, 39.30199180817845},
		{1814, 39.48025720867229},
		{1815, 39.5287187420122},
		{1816, 38.853563440015805},
		{1817, 38.77749075357144},
		{1818, 39.12960232658636},
		{1819, 39.07838534119988},
		{1820, 39.149292277699324},
		{1821, 39.00774814154133},
		{1822, 38.63589451271645},
		{1823, 38.95501568270912},
		{1824, 38.4982817921926},
		{1825, 39.10088500152322},
		{1826, 39.59621160868451},
		{1827, 39.45833354062081},
		{1828, 39.59936421761144},
		{1829, 39.57053924615646},
		{1830, 39.55660528183884},
		{1831, 39.71569674116191},
		{1832, 39.71328630121623},
		{1833, 39.5500745849456},
		{1834, 39.52081122302288},
		{1835, 38.74032920127892},
		{1836, 39.19864530117539},
		{1837, 39.06947571924525},
		{1838, 39.34422885689023},
		{1839, 39.6353362225693},
		{1840, 40.71459696346432},
		{1841, 40.66496952468566},
		{1842, 41.006778236777826},
		{1843, 40.98904213311685},
		{1844, 40.108733704463795},
		{1845, 39.422159369592656},
		{1846, 39.23842350070322},
		{1847, 37.531170184849564},
		{1848, 38.05353522108415},
		{1849, 38.31111320664359},
		{1850, 38.64900610422957},
		{1851, 38.686484249347565},
		{1852, 38.68434197825875},
		{1853, 38.72332458907287},
		{1854, 38.81119764084882},
		{1855, 38.46493433585336},
		{1856, 38.51556317974943},
		{1857, 38.764010535750494},
		{1858, 38.55294929767745},
		{1859, 38.078366113942565},
		{1860, 38.42988961361437},
		{1861, 39.047014068329304},
		{1862, 39.532635843081685},
		{1863, 39.10571993423328},
		{1864, 39.60131420080908},
		{1865, 38.81680138357229},
		{1866, 39.25082176080607},
		{1867, 38.15677220409511},
		{1868, 38.62555210254061},
		{1869, 38.47132788365266},
		{1870, 38.487714681626166},
		{1871, 38.39888830728324},
		{1872, 37.54975105530615},
		{1873, 36.620496899833185},
		{1874, 36.76592939311881},
		{1875, 36.77417927197788},
		{1876, 36.78457313190018},
		{1877, 36.726556157401774},
		{1878, 37.19484339680942},
		{1879, 37.12075085414386},
		{1880, 36.977258419880144},
		{1881, 36.91078025788851},
		{1882, 37.02130072401423},
		{1883, 36.55535121811925},
		{1884, 35.989467257305485},
		{1885, 35.963032268137205},
		{1886, 36.010752822606925},
		{1887, 36.02787704722448},
		{1888, 36.49326570398064},
		{1889, 36.23934542841747},
		{1890, 35.87776015328743},
		{1891, 35.09797528518658},
		{1892, 35.077065290325194},
		{1893, 35.311414001953885},
		{1894, 34.57305509209034},
		{1895, 34.6842074805997},
		{1896, 34.80183031401632},
		{1897, 35.47101264307737},
		{1898, 35.17048394585489},
		{1899, 35.089175368832215},
		{1900, 35.22509160122155},
		{1901, 35.01261679236839},
		{1902, 35.20621635881338},
		{1903, 35.37294264170206},
		{1904, 34.459027714667855},
		{1905, 35.092154761591416},
		{1906, 34.8517351148361},
		{1907, 34.86550272062247},
		{1908, 35.125600870401854},
		{1909, 34.83446288221645},
		{1910, 34.78242112077965},
		{1911, 34.31203602602105},
		{1912, 34.279540168694986},
		{1913, 35.1096803630126},
		{1914, 34.514233849300744},
		{1915, 34.53539867779291},
		{1916, 34.279414052434674},
		{1917, 34.36455267465703},
		{1918, 34.10174900420402},
		{1919, 34.41175163800032},
		{1920, 34.291999205288036},
		{1921, 34.19594013042914},
		{1922, 34.18838364564101},
		{1923, 36.07392483120914},
		{1924, 36.05098431555323},
		{1925, 36.76095964181586},
		{1926, 37.82863560027054},
		{1927, 37.52980493319014},
		{1928, 37.53527503569141},
		{1929, 37.512668716830554},
		{1930, 37.11106227609642},
		{1931, 37.349978643969806},
		{1932, 37.20979207398837},
		{1933, 37.29059943130983},
		{1934, 37.12675552575493},
		{1935, 37.34021255585861},
		{1936, 37.33863802694112},
		{1937, 36.93793225844994},
		{1938, 36.96147855185145},
		{1939, 37.079544775050316},
		{1940, 36.96800493367608},
		{1941, 36.62130515314802},
		{1942, 36.64654224799271},
		{1943, 36.564393168165026},
		{1944, 36.76798510268848},
		{1945, 36.34484500467934},
		{1946, 35.77622429269998},
		{1947, 36.28861461359311},
		{1948, 35.62308019530623},
		{1949, 35.44131748549641},
		{1950, 34.85392909768805},
		{1951, 34.55133668610086},
		{1952, 35.092025906373635},
		{1953, 34.768122961192404},
		{1954, 35.02372314142307},
		{1955, 35.11384033534372},
		{1956, 35.09859916781124},
		{1957, 35.63073140705392},
		{1958, 35.47556248383545},
		{1959, 34.87826460124942},
		{1960, 34.87959003890974},
		{1961, 34.025365090862095},
		{1962, 33.804380004654575},
		{1963, 32.78020946544929},
		{1964, 32.75918431846117},
		{1965, 32.75851314921747},
		{1966, 32.93198375479733},
		{1967, 35.25828074460553},
		{1968, 34.28240693566641},
		{1969, 34.339864432373425},
		{1970, 34.12397376413126},
		{1971, 34.79011341612875},
		{1972, 34.64686103755323},
		{1973, 34.599313028973235},
		{1974, 34.53242934134927},
		{1975, 34.606548457455425},
		{1976, 34.83272344102554},
		{1977, 34.49913157098365},
		{1978, 34.70437557256804},
		{1979, 35.19565588926615},
		{1980, 35.185492244262484},
		{1981, 33.550787666460536},
		{1982, 34.02484580154227},
		{1983, 34.15044793215261},
		{1984, 34.543313702949256},
		{1985, 34.49383893491851},
		{1986, 34.69468439935647},
		{1987, 34.36140916945972},
		{1988, 33.199535764838636},
		{1989, 33.54089185371181},
		{1990, 33.25543526295492},
		{1991, 33.126250139798344},
		{1992, 32.1006804695537},
		{1993, 32.11793625750763},
		{1994, 31.547141310894098},
		{1995, 31.358848538734513},
		{1996, 30.96324192071878},
		{1997, 31.081433577858583},
		{1998, 31.004611604459708},
		{1999, 30.86284767543519},
		{2000, 30.105433307161203},
		{2001, 29.32024170286219},
		{2002, 29.319658236582814},
		{2003, 29.758958255855212},
		{2004, 30.117663632341145},
		{2005, 29.548890824528907},
		{2006, 29.35044585015188},
		{2007, 29.38762759040267},
		{2008, 29.391950686507506},
		{2009, 30.353380249710128},
		{2010, 29.46899984490753},
		{2011, 29.33586115695675},
		{2012, 29.36114381531963},
		{2013, 29.256022938524392},
		{2014, 29.207874497314503},
		{2015, 29.266115045795534},
		{2016, 29.11037246533442},
		{2017, 29.98833975350206},
		{2018, 29.827356428113166},
		{2019, 29.196369679790173},
		{2020, 29.09133736287457},
		{2021, 29.35398305475309},
		{2022, 29.54604416792868},
		{2023, 28.36216318117799},
		{2024, 28.172545798424313},
		{2025, 29.167896346989487},
		{2026, 29.134747648857985},
		{2027, 28.0133642954549},
		{2028, 27.97694686027503},
		{2029, 28.374194072568063},
		{2030, 28.37095153088029},
		{2031, 28.388216858848796},
		{2032, 27.992329533869835},
		{2033, 27.367870148995426},
		{2034, 27.251995482961384},
		{2035, 27.746729670664767},
		{2036, 27.679330290354688},
		{2037, 28.028729073068334},
		{2038, 28.028705243794963},
		{2039, 28.350786367009555},
		{2040, 28.352026432670012},
		{2041, 28.225867429505602},
		{2042, 28.211139910084785},
		{2043, 28.126532263542863},
		{2044, 28.0111605756859},
		{2045, 28.037986830307158},
		{2046, 27.771048205304385},
		{2047, 28.101282898505325},
		{2048, 27.956754355558264},
		{2049, 28.898451768426284},
		{2050, 28.33803560429799},
		{2051, 28.51219696371789},
		{2052, 28.687032331939648},
		{2053, 28.519256261795114},
		{2054, 28.534411822889545},
		{2055, 28.475878463300383},
		{2056, 28.425823720753847},
		{2057, 28.531458363352172},
		{2058, 28.371986321151727},
		{2059, 27.281010674258564},
		{2060, 27.285423846392753},
		{2061, 26.982970399775176},
		{2062, 26.779917891194174},
		{2063, 27.334018434853117},
		{2064, 27.261844645356348},
		{2065, 27.08643958372844},
		{2066, 27.399456700893154},
		{2067, 27.35064731231653},
		{2068, 27.27964623953228},
		{2069, 27.594268915970815},
		{2070, 27.529530803142432},
		{2071, 27.57032488737316},
		{2072, 27.4481123592095},
		{2073, 26.840190037017013},
		{2074, 25.848284039508844},
		{2075, 27.842568040851305},
		{2076, 28.053732543316965},
		{2077, 28.626766043264496},
		{2078, 28.147220846160913},
		{2079, 28.116929624170805},
		{2080, 28.006707726319842},
		{2081, 28.30104291864807},
		{2082, 28.33894272086177},
		{2083, 28.44881078446907},
		{2084, 28.407665090419357},
		{2085, 29.120266868128827},
		{2086, 29.3317588176526},
		{2087, 29.665040474972592},
		{2088, 29.513552545038365},
		{2089, 29.692597123031344},
		{2090, 28.981378296214995},
		{2091, 29.09551924330645},
		{2092, 29.103036068928503},
		{2093, 29.546315663547908},
		{2094, 29.957633494290146},
		{2095, 29.904489752089027},
		{2096, 29.905243264397946},
		{2097, 29.046115152161175},
		{2098, 29.044952829872138},
		{2099, 29.266240445467794},
		{2100, 29.59245009354912},
		{2101, 29.711595098413557},
		{2102, 30.01909364831503},
		{2103, 30.02844709237877},
		{2104, 29.95602684268363},
		{2105, 29.966649831798698},
		{2106, 29.870132199150568},
		{2107, 30.054469107153587},
		{2108, 30.00047327238945},
		{2109, 29.56478041606059},
		{2110, 29.90146475387948},
		{2111, 29.622907648647697},
		{2112, 30.309630232664347},
		{2113, 29.964474309001496},
		{2114, 29.49100419801057},
		{2115, 29.961421989896223},
		{2116, 30.346134915473066},
		{2117, 30.604426349400814},
		{2118, 31.112159919665434},
		{2119, 31.422724201504558},
		{2120, 31.585365400843976},
		{2121, 31.996984724582912},
		{2122, 32.01819220164321},
		{2123, 29.835521461379106},
		{2124, 30.096046165850005},
		{2125, 30.103782504040677},
		{2126, 30.053631787942646},
		{2127, 29.906750519100672},
		{2128, 29.906138940448397},
		{2129, 29.852649788445774},
		{2130, 30.207615016068207},
		{2131, 30.820165055355577},
		{2132, 30.66806843164003},
		{2133, 29.933937853455785},
		{2134, 29.836930249713454},
		{2135, 29.643987595361455},
		{2136, 30.19074784312923},
		{2137, 30.22542736675569},
		{2138, 29.853735660522815},
		{2139, 29.84557151500528},
		{2140, 29.839193019325773},
		{2141, 31.156302724258683},
		{2142, 31.25546101123808},
		{2143, 31.108453918938817},
		{2144, 30.666860514055692},
		{2145, 30.645333186456657},
		{2146, 30.170151295689198},
		{2147, 30.295207809019463},
		{2148, 30.8518824742632},
		{2149, 31.113576998851187},
		{2150, 31.31604954619179},
		{2151, 31.312585749752536},
		{2152, 30.864704949670024},
		{2153, 30.89452281143898},
		{2154, 31.23342385585233},
		{2155, 31.404784721717572},
		{2156, 31.18805814717659},
		{2157, 31.837154644201334},
		{2158, 31.806849694846097},
		{2159, 31.730917911346975},
		{2160, 31.640559706015285},
		{2161, 31.04070924433706},
		{2162, 29.808188744723488},
		{2163, 30.56052952306829},
		{2164, 30.606670887563382},
		{2165, 30.40372997396476},
		{2166, 30.720199473446222},
		{2167, 30.76154769857017},
		{2168, 30.970925562360136},
		{2169, 31.019650519858754},
		{2170, 31.229380140618062},
		{2171, 31.388864514313752},
		{2172, 31.354301610850698},
		{2173, 30.99276274081928},
		{2174, 31.915179561018494},
		{2175, 31.918576205835393},
		{2176, 33.159103841393794},
		{2177, 33.29787809852651},
		{2178, 33.01252169346593},
		{2179, 33.02073471727393},
		{2180, 32.99606635802466},
		{2181, 33.00728410289347},
		{2182, 34.05607738376311},
		{2183, 33.15858774223605},
		{2184, 33.43289159046996},
		{2185, 33.4955768254167},
		{2186, 33.49850941243009},
		{2187, 33.6849166584474},
		{2188, 33.94841513146962},
		{2189, 33.30304766805812},
		{2190, 33.22108394356688},
		{2191, 32.69882923499378},
		{2192, 32.427937313925405},
		{2193, 32.18858592278298},
		{2194, 32.32103253993895},
		{2195, 32.71485240989556},
		{2196, 33.33520209595963},
		{2197, 33.263196044170755},
		{2198, 32.710035724800086},
		{2199, 33.15410338967067},
		{2200, 33.49162431986145},
		{2201, 33.710578425275074},
		{2202, 33.32543853826808},
		{2203, 33.25527764293191},
		{2204, 33.46839755330703},
		{2205, 33.683203814839125},
		{2206, 33.73810431818219},
		{2207, 33.26839137241745},
		{2208, 32.870015366666216},
		{2209, 33.00655626613422},
		{2210, 33.461231149892406},
		{2211, 33.89423423998636},
		{2212, 33.50309147431082},
		{2213, 33.579287468108},
		{2214, 33.66791505848393},
		{2215, 34.38967435369897},
		{2216, 36.28181224897229},
		{2217, 36.88357351027984},
		{2218, 36.84878280662571},
		{2219, 37.47217008391058},
		{2220, 37.84817431115949},
		{2221, 37.90417263033801},
		{2222, 37.83987872444627},
		{2223, 37.32997096688406},
		{2224, 37.4853701637283},
		{2225, 38.101980955554055},
		{2226, 37.634879540380766},
		{2227, 37.458475294951995},
		{2228, 37.24084436632412},
		{2229, 38.45249609002218},
		{2230, 37.85699416217567},
		{2231, 37.35638525020401},
		{2232, 37.34244352498811},
		{2233, 37.61816197522845},
		{2234, 37.16649173849021},
		{2235, 37.47322953068236},
		{2236, 37.2333430674181},
		{2237, 37.14977263499141},
		{2238, 36.93966508491671},
		{2239, 37.069212698488194},
		{2240, 37.13031285412952},
		{2241, 37.12637993171949},
		{2242, 36.91777324959473},
		{2243, 36.89130918063377},
		{2244, 36.83287503871216},
		{2245, 37.20202075141188},
		{2246, 36.83524257279788},
		{2247, 36.830944738890715},
		{2248, 36.98000618572528},
		{2249, 37.42482479721468},
		{2250, 37.63036405693426},
		{2251, 36.667366298537495},
		{2252, 36.752893008045845},
		{2253, 36.353792852732994},
		{2254, 36.68479700023098},
		{2255, 36.69141503905663},
		{2256, 36.87350479821814},
		{2257, 36.165233755112176},
		{2258, 35.733553198879285},
		{2259, 35.48805636280678},
		{2260, 35.620206059835176},
		{2261, 35.633174461262},
		{2262, 35.8437425347841},
		{2263, 35.9725442605288},
		{2264, 35.30481451302577},
		{2265, 34.799856216789664},
		{2266, 35.20793503084299},
		{2267, 35.15095956450417},
		{2268, 35.52063000497536},
		{2269, 35.7847373835048},
		{2270, 35.4149729701676},
		{2271, 35.20790810237125},
		{2272, 35.015963290895314},
		{2273, 35.152141159882405},
		{2274, 34.7677733076443},
		{2275, 34.48805775061282},
		{2276, 34.48754941675096},
		{2277, 34.31241590281594},
		{2278, 34.39912341428924},
		{2279, 34.214813920779555},
		{2280, 34.28073309875244},
		{2281, 33.48991097769715},
		{2282, 33.48007351832354},
		{2283, 34.358913453235225},
		{2284, 34.81847331063591},
		{2285, 35.01545093767003},
		{2286, 35.522962489940845},
		{2287, 35.4137445712856},
		{2288, 36.11012685056512},
		{2289, 36.22211408792958},
		{2290, 36.18967456479244},
		{2291, 36.19988717533946},
		{2292, 36.82497371024283},
		{2293, 36.828687017747605},
		{2294, 36.81449738807559},
		{2295, 36.83820772921837},
		{2296, 37.24345441577025},
		{2297, 36.92597317758977},
		{2298, 35.8576345692627},
		{2299, 35.88555065526424},
		{2300, 36.06990172210221},
		{2301, 35.94117611192266},
		{2302, 35.917302525035176},
		{2303, 35.67840840680457},
		{2304, 36.24622015952215},
		{2305, 36.3620909907333},
		{2306, 36.530845130699554},
		{2307, 37.10057430482785},
		{2308, 36.99538603909066},
		{2309, 37.317285421916054},
		{2310, 37.023853418725835},
		{2311, 37.331141223084856},
		{2312, 36.8562932336869},
		{2313, 36.872868345401656},
		{2314, 37.29544282161283},
		{2315, 37.162364534385944},
		{2316, 37.12578408822864},
		{2317, 37.27181141630765},
		{2318, 37.4312789350087},
		{2319, 37.13237760566537},
		{2320, 37.11211557589226},
		{2321, 36.8242524461256},
		{2322, 36.99871196220295},
		{2323, 36.53900862036208},
		{2324, 37.02861438655951},
		{2325, 36.87864302249147},
		{2326, 37.35471016767267},
		{2327, 37.48320749003392},
		{2328, 37.90669667178226},
		{2329, 38.226516715835075},
		{2330, 38.22517385982157},
		{2331, 38.517597628277066},
		{2332, 39.13171392966848},
		{2333, 38.43519575012611},
		{2334, 39.019538645173135},
		{2335, 38.96816384019192},
		{2336, 39.18980442092051},
		{2337, 39.20626086856055},
		{2338, 38.91508217017819},
		{2339, 39.768877736103605},
		{2340, 39.77576200648055},
		{2341, 39.70340066789778},
		{2342, 39.28514875368393},
		{2343, 39.45020952653042},
		{2344, 39.39155936608262},
		{2345, 38.98575368659195},
		{2346, 39.07457749590117},
		{2347, 39.315569014125366},
		{2348, 39.534223631257824},
		{2349, 39.46496086158911},
		{2350, 39.68920036760856},
		{2351, 39.76991133023266},
		{2352, 39.535528882389414},
		{2353, 38.78464337191674},
		{2354, 38.84802165134902},
		{2355, 39.13969405342098},
		{2356, 39.57233319093269},
		{2357, 39.4775906546037},
		{2358, 39.22345113789871},
		{2359, 38.57809299449735},
		{2360, 38.57645107268238},
		{2361, 38.972137312472306},
		{2362, 38.48045973837729},
		{2363, 38.13356592124918},
		{2364, 37.58820508109955},
		{2365, 37.201698739301705},
		{2366, 36.94671668201051},
		{2367, 37.2977080973896},
		{2368, 37.56900473168416},
		{2369, 38.107005730339885},
		{2370, 37.66230861685099},
		{2371, 37.91179580848597},
		{2372, 38.021982005474676},
		{2373, 37.980026494228056},
		{2374, 37.936145999397155},
		{2375, 38.02211453233284},
		{2376, 38.184818957491586},
		{2377, 38.001562339346805},
		{2378, 38.17115924132589},
		{2379, 38.02933185572005},
		{2380, 39.26672886459662},
		{2381, 39.438886818730374},
		{2382, 39.632666785719366},
		{2383, 39.69431724624071},
		{2384, 38.45229246819694},
		{2385, 38.92044292382193},
		{2386, 38.71930950151329},
		{2387, 38.743236180041634},
		{2388, 38.983429309450706},
		{2389, 38.97420423718335},
		{2390, 38.7215640748561},
		{2391, 37.29737183692541},
		{2392, 37.45124430298096},
		{2393, 37.528370436924256},
		{2394, 37.95222937643749},
		{2395, 37.83598429862198},
		{2396, 37.497742223483016},
		{2397, 37.445480102504746},
		{2398, 37.30971600962666},
		{2399, 37.35293634455916},
		{2400, 37.53645804280457},
		{2401, 37.39990233842864},
		{2402, 37.35307876697158},
		{2403, 37.20884959153676},
		{2404, 37.37721955667315},
		{2405, 38.163601842424356},
		{2406, 38.1744583572895},
		{2407, 38.22526886325096},
		{2408, 37.74537108958047},
		{2409, 37.695977476731315},
		{2410, 37.132399280518335},
		{2411, 37.58951197469921},
		{2412, 37.595014414031965},
		{2413, 37.8352174960965},
		{2414, 38.19482676835253},
		{2415, 38.81615407880503},
		{2416, 38.96954505622985},
		{2417, 38.51014481284331},
		{2418, 38.70949789629867},
		{2419, 38.8330673644232},
		{2420, 38.79852936780312},
		{2421, 38.36989947803435},
		{2422, 38.38299952696908},
		{2423, 38.37911293511124},
		{2424, 38.30523674843859},
		{2425, 38.2780523618985},
		{2426, 38.11238736059342},
		{2427, 38.146761901654045},
		{2428, 38.16577426042987},
		{2429, 37.21503059523459},
		{2430, 37.20194861904684},
		{2431, 37.93629061523524},
		{2432, 37.73740404804743},
		{2433, 38.19428643465362},
		{2434, 38.665082590845316},
		{2435, 38.73654599191024},
		{2436, 38.91621864442491},
		{2437, 38.91583279385087},
		{2438, 39.070111723777586},
		{2439, 39.020903222278925},
		{2440, 39.11691716267563},
		{2441, 38.48057387525625},
		{2442, 38.53568696118676},
		{2443, 38.643673021215534},
		{2444, 38.260311740941205},
		{2445, 38.22563157129045},
		{2446, 38.02954799036356},
		{2447, 38.37221447795219},
		{2448, 38.39689602415647},
		{2449, 38.39841000702669},
		{2450, 38.42466352375716},
		{2451, 38.54182197292112},
		{2452, 37.97168851025008},
		{2453, 37.98072923120449},
		{2454, 38.500048729444565},
		{2455, 38.51204601961472},
		{2456, 38.57119577782335},
		{2457, 38.66797170791196},
		{2458, 38.42048390776265},
		{2459, 38.37374323690084},
		{2460, 37.32968095967905},
		{2461, 37.08263012999697},
		{2462, 37.50308137969606},
		{2463, 37.74228959632586},
		{2464, 37.74069036775092},
		{2465, 37.17849832026422},
		{2466, 36.96662265441138},
		{2467, 36.94747312813148},
		{2468, 37.478003899130535},
		{2469, 36.78538811979787},
		{2470, 36.72166338143237},
		{2471, 36.05942648742004},
		{2472, 35.87090337493837},
		{2473, 35.86422187859753},
		{2474, 36.19415669239082},
		{2475, 36.4304121266589},
		{2476, 37.3789805517048},
		{2477, 37.50777745489909},
		{2478, 37.1957829652079},
		{2479, 37.17977294040911},
		{2480, 38.216074640194954},
		{2481, 39.76987261793349},
		{2482, 39.710123436790155},
		{2483, 39.46441746053927},
		{2484, 39.887390948196604},
		{2485, 40.47940528392495},
		{2486, 40.96173004505429},
		{2487, 40.80366573878367},
		{2488, 41.54175623171586},
		{2489, 41.9020390419709},
		{2490, 42.33439627394617},
		{2491, 43.42510969518392},
		{2492, 43.3467495490359},
		{2493, 43.55870952188062},
		{2494, 43.687712555042445},
		{2495, 43.5164043044117},
		{2496, 43.50241651823093},
		{2497, 44.29101702903774},
		{2498, 43.557445214897555},
		{2499, 43.93564846197703},
		{2500, 43.443137649447785},
		{2501, 42.20225850019654},
		{2502, 41.79435089000459},
		{2503, 41.76196350512668},
		{2504, 42.303354344610476},
		{2505, 43.10084057387829},
		{2506, 43.178994474468176},
		{2507, 42.428481094831376},
		{2508, 42.547354773308356},
		{2509, 42.76222854703309},
		{2510, 41.82251108158902},
		{2511, 39.95979151189301},
		{2512, 39.95821075474937},
		{2513, 40.79681508224348},
		{2514, 40.81708050706749},
		{2515, 40.86098377024993},
		{2516, 41.37066167344864},
		{2517, 40.811835396416136},
		{2518, 41.75210935179267},
		{2519, 42.081642066666376},
		{2520, 42.08290785008212},
		{2521, 42.24300735871055},
		{2522, 43.469996426408535},
		{2523, 43.07953421381988},
		{2524, 42.75861204389449},
		{2525, 42.68285436365714},
		{2526, 42.70557759253858},
		{2527, 42.48853467614813},
		{2528, 42.57942769581852},
		{2529, 41.641644251781955},
		{2530, 41.995153297318495},
		{2531, 41.948491392461136},
		{2532, 42.10736843531852},
		{2533, 42.09104450737047},
		{2534, 41.844893380773094},
		{2535, 41.604667841842605},
		{2536, 41.664479889713284},
		{2537, 41.721596977396956},
		{2538, 41.03970005650016},
		{2539, 41.02554512620144},
		{2540, 40.87794875769463},
		{2541, 39.90420781213615},
		{2542, 39.917556293894414},
		{2543, 39.854472672630656},
		{2544, 40.07480007115982},
		{2545, 39.67254038329676},
		{2546, 39.75546306998178},
		{2547, 39.738281539976},
		{2548, 40.68871115360799},
		{2549, 40.634406366307545},
		{2550, 40.825610497929816},
		{2551, 40.56134401134031},
		{2552, 40.49316949904079},
		{2553, 41.411980447849764},
		{2554, 41.45708514923727},
		{2555, 41.62699785722495},
		{2556, 40.73847974293596},
		{2557, 40.9036273444639},
		{2558, 40.835583475220645},
		{2559, 40.692726642278075},
		{2560, 40.02156508795992},
		{2561, 39.59727702576626},
		{2562, 39.85915504290939},
		{2563, 40.25776254670131},
		{2564, 38.98058165621274},
		{2565, 39.19626524671679},
		{2566, 40.27851337856},
		{2567, 40.42198078582052},
		{2568, 40.84324235005817},
		{2569, 41.43936550130537},
		{2570, 41.01778584521347},
		{2571, 41.71741296575474},
		{2572, 41.80764781109384},
		{2573, 41.86906945022645},
		{2574, 41.261486612033735},
		{2575, 41.05330428242297},
		{2576, 40.858767555527805},
		{2577, 41.20327089795783},
		{2578, 41.72211067822387},
		{2579, 42.24939510039889},
		{2580, 43.02032269889916},
		{2581, 43.169601435406285},
		{2582, 43.03240557408647},
		{2583, 44.058494616358495},
		{2584, 42.69632168879649},
		{2585, 42.70233698941648},
		{2586, 42.694697227758425},
		{2587, 42.88764129390405},
		{2588, 43.08913827262613},
		{2589, 42.89734340644753},
		{2590, 42.541548108008676},
		{2591, 42.810258239438845},
		{2592, 41.86604756233666},
		{2593, 41.62721508609488},
		{2594, 41.42397058232155},
		{2595, 41.226915898008365},
		{2596, 41.362278318282556},
		{2597, 40.55335045136749},
		{2598, 40.53227628820762},
		{2599, 40.55804572554107},
		{2600, 40.59654847444725},
		{2601, 40.600759547237764},
		{2602, 39.886740241510296},
		{2603, 39.74441936291896},
		{2604, 39.4574717226145},
		{2605, 40.060274508040756},
		{2606, 40.13087881643223},
		{2607, 40.034347420207766},
		{2608, 39.99959402815168},
		{2609, 39.96436408170099},
		{2610, 40.05829543634176},
		{2611, 39.36433405019379},
		{2612, 39.18728728622009},
		{2613, 39.24495519885113},
		{2614, 39.07757095227659},
		{2615, 39.274474362783934},
		{2616, 39.10484953636473},
		{2617, 38.82145181383544},
		{2618, 39.14891944323103},
		{2619, 38.38965006924832},
		{2620, 38.39353088180409},
		{2621, 37.5842486836372},
		{2622, 37.50572067949534},
		{2623, 37.47652007277792},
		{2624, 37.460762948164124},
		{2625, 37.70360102326645},
		{2626, 37.9284949964923},
		{2627, 37.94965323046383},
		{2628, 38.7335220955414},
		{2629, 38.261029750354496},
		{2630, 38.32794292196558},
		{2631, 38.15225856809112},
		{2632, 37.90928164592873},
		{2633, 38.23964767428957},
		{2634, 38.52123444438283},
		{2635, 38.42104882766221},
		{2636, 38.96022542956203},
		{2637, 39.24100815688652},
		{2638, 39.08793479950826},
		{2639, 39.08884782494273},
		{2640, 39.55798120001941},
		{2641, 39.105610334728816},
		{2642, 39.007630691250704},
		{2643, 38.20582502075655},
		{2644, 37.543315900244956},
		{2645, 37.57555493691154},
		{2646, 37.28692975360062},
		{2647, 36.23295703793576},
		{2648, 36.18918955965909},
		{2649, 36.08967399169209},
		{2650, 35.959764744648616},
		{2651, 36.19197227091526},
		{2652, 36.660749297052476},
		{2653, 36.64494239211856},
		{2654, 36.42558293570474},
		{2655, 35.6375906092785},
		{2656, 35.70674524011707},
		{2657, 35.57068617114742},
		{2658, 35.373679501491125},
		{2659, 34.672906766786355},
		{2660, 34.63171730160524},
		{2661, 34.67893812694516},
		{2662, 34.654847702252816},
		{2663, 34.65402092594314},
		{2664, 34.771606902340174},
		{2665, 34.88770410184326},
		{2666, 35.02613347813431},
		{2667, 34.46478240438498},
		{2668, 34.68568272903622},
		{2669, 34.44312396898851},
		{2670, 34.21234874898513},
		{2671, 34.22641653591384},
		{2672, 34.128223876429104},
		{2673, 34.11242598726876},
		{2674, 33.96946719921404},
		{2675, 33.96323823944133},
		{2676, 34.10711370082857},
		{2677, 33.61921918842873},
		{2678, 33.86921234418534},
		{2679, 34.38300717848694},
		{2680, 34.47612413434144},
		{2681, 33.878358516222164},
		{2682, 33.95450317313319},
		{2683, 34.15914443090066},
		{2684, 33.8706108159827},
		{2685, 34.38200284883451},
		{2686, 34.38499453227831},
		{2687, 34.540332423206024},
		{2688, 34.180253970681264},
		{2689, 34.23917159483758},
		{2690, 33.3409032838207},
		{2691, 33.31758465582364},
		{2692, 33.77426222148222},
		{2693, 33.98469402333139},
		{2694, 35.18897714992098},
		{2695, 35.27834052976643},
		{2696, 35.30947212883306},
		{2697, 34.697407680416966},
		{2698, 35.480822427408455},
		{2699, 35.56467660433882},
		{2700, 35.53420547926852},
		{2701, 35.541541884085056},
		{2702, 35.587731051837416},
		{2703, 35.20649466374446},
		{2704, 34.23840909461753},
		{2705, 34.03444342209502},
		{2706, 33.765366528347904},
		{2707, 33.83889062651724},
		{2708, 33.58019639477255},
		{2709, 33.404308515206424},
		{2710, 32.5794052072048},
		{2711, 32.585104546723436},
		{2712, 32.266279628194816},
		{2713, 31.763850208735782},
		{2714, 31.82468466746643},
		{2715, 31.693375409028246},
		{2716, 31.713589796403312},
		{2717, 31.89442805523013},
		{2718, 30.539601797884956},
		{2719, 30.419562355741576},
		{2720, 30.570511736331543},
		{2721, 30.436298375688793},
		{2722, 30.4150717914012},
		{2723, 30.65434061186432},
		{2724, 30.55584352842997},
		{2725, 30.666073288068162},
		{2726, 30.631386637749323},
		{2727, 30.253653739813487},
		{2728, 29.709659914781653},
		{2729, 28.896044518517407},
		{2730, 28.61563883266229},
		{2731, 28.70191534200063},
		{2732, 29.201426696903418},
		{2733, 29.55609328115069},
		{2734, 28.656892877727273},
		{2735, 29.07942162200021},
		{2736, 29.26501321868672},
		{2737, 29.646954975157882},
		{2738, 29.334935013945213},
		{2739, 29.050159467980038},
		{2740, 29.0197508395469},
		{2741, 28.774518059870104},
		{2742, 29.406230838159324},
		{2743, 29.384319812035454},
		{2744, 28.16252911704915},
		{2745, 28.370968057426175},
		{2746, 28.47565669331631},
		{2747, 28.557859294540695},
		{2748, 28.61247793755877},
		{2749, 28.290652073805152},
		{2750, 27.951340656361044},
		{2751, 28.423165549960824},
		{2752, 28.44548879989753},
		{2753, 28.43557149451693},
		{2754, 28.041081598237184},
		{2755, 27.037025918511283},
		{2756, 27.037046717739166},
		{2757, 26.76662330440143},
		{2758, 26.699917649004856},
		{2759, 26.889805976598808},
		{2760, 26.801047959108622},
		{2761, 26.760134602011636},
		{2762, 27.18343703683992},
		{2763, 27.16859736505303},
		{2764, 27.04599114188329},
		{2765, 27.07137489048196},
		{2766, 27.998508184180817},
		{2767, 27.90008295282876},
		{2768, 27.64403546500875},
		{2769, 27.526129921325765},
		{2770, 27.17477572419831},
		{2771, 27.341255122694978},
		{2772, 27.281292839268044},
		{2773, 27.457194113628123},
		{2774, 27.125831819180696},
		{2775, 27.153488900596923},
		{2776, 26.40548276755705},
		{2777, 26.20664419363637},
		{2778, 25.447443897366075},
		{2779, 25.436213063192547},
		{2780, 25.80691677092328},
		{2781, 25.911262332850754},
		{2782, 25.827467186922345},
		{2783, 25.93386934563249},
		{2784, 25.87141867343481},
		{2785, 26.127903579867812},
		{2786, 26.11594064081708},
		{2787, 25.968383286170432},
		{2788, 26.12819163023331},
		{2789, 26.528383252253256},
		{2790, 26.63041226276307},
		{2791, 26.12308929789773},
		{2792, 25.824556557378628},
		{2793, 25.705278327514673},
		{2794, 25.533179388113364},
		{2795, 25.111955575936644},
		{2796, 25.19306942608654},
		{2797, 25.315610017978926},
		{2798, 25.54647388963821},
		{2799, 25.487771689167698},
		{2800, 25.81216153045184},
		{2801, 25.763987775825093},
		{2802, 25.902526292298354},
		{2803, 25.968428482836103},
		{2804, 26.41323130119539},
		{2805, 26.49882702247749},
		{2806, 26.49213018635884},
		{2807, 27.54190493767045},
		{2808, 26.535580599916567},
		{2809, 26.57432320629819},
		{2810, 26.90159184032591},
		{2811, 26.587265607613453},
		{2812, 26.59291668712681},
		{2813, 26.423373614090597},
		{2814, 27.866860910585345},
		{2815, 28.207557672760228},
		{2816, 28.496172416512515},
		{2817, 28.42429902292854},
		{2818, 28.524648108317813},
		{2819, 28.611122350682358},
		{2820, 27.732969193443907},
		{2821, 27.399970276602343},
		{2822, 27.905344738574847},
		{2823, 27.942078810801064},
		{2824, 27.804282414772008},
		{2825, 27.8069558632371},
		{2826, 27.832235140508022},
		{2827, 27.37610449016281},
		{2828, 27.448768550966395},
		{2829, 27.772837570338186},
		{2830, 28.168201851600365},
		{2831, 28.149299964281663},
		{2832, 28.131056291061356},
		{2833, 27.900258674853603},
		{2834, 27.900129431796973},
		{2835, 27.703804331410204},
		{2836, 27.233402978544067},
		{2837, 28.62270120196561},
		{2838, 28.753984952852},
		{2839, 27.849450371851088},
		{2840, 28.607099713321475},
		{2841, 28.581922133054277},
		{2842, 28.44598231620244},
		{2843, 28.502126322789735},
		{2844, 28.33255356108348},
		{2845, 28.285305697352577},
		{2846, 28.09977585227804},
		{2847, 27.757945405078186},
		{2848, 26.392688537225578},
		{2849, 26.425165673292835},
		{2850, 26.545835303452666},
		{2851, 26.439538464328816},
		{2852, 26.269936477378405},
		{2853, 26.512109899359487},
		{2854, 27.512278345593856},
		{2855, 28.082715243376764},
		{2856, 28.41596094710675},
		{2857, 27.45862971920284},
		{2858, 27.39257771938314},
		{2859, 27.36345333435615},
		{2860, 27.47151777939197},
		{2861, 27.950894827790314},
		{2862, 27.60764930763731},
		{2863, 27.60813226132976},
		{2864, 27.372749995542335},
		{2865, 27.521145751594496},
		{2866, 27.940176424747328},
		{2867, 28.35191115083003},
		{2868, 27.83432934764762},
		{2869, 29.15786344470242},
		{2870, 29.79498828662717},
		{2871, 29.568955183772818},
		{2872, 29.517280037492394},
		{2873, 29.63961776908481},
		{2874, 30.662190826973372},
		{2875, 31.71938876522025},
		{2876, 31.826598098832324},
		{2877, 31.826543856914203},
		{2878, 31.789643116898706},
		{2879, 31.92180528432921},
		{2880, 31.80959471460479},
		{2881, 31.681403018636786},
		{2882, 31.191957159319717},
		{2883, 30.912044142938722},
		{2884, 30.34671826324825},
		{2885, 29.98006193864975},
		{2886, 29.472355419791427},
		{2887, 29.479494293595344},
		{2888, 29.752640347399335},
		{2889, 30.09718091452955},
		{2890, 30.431387560542564},
		{2891, 29.496695832798583},
		{2892, 29.246495986640035},
		{2893, 29.222427974948758},
		{2894, 29.124858728303685},
		{2895, 28.965028596965585},
		{2896, 29.08714179149652},
		{2897, 30.4283543168946},
		{2898, 31.215229365152215},
		{2899, 31.150712888260642},
		{2900, 30.857590831985288},
		{2901, 31.460128657051456},
		{2902, 31.159427314891474},
		{2903, 31.165617956156684},
		{2904, 31.299914014945326},
		{2905, 31.11690963033505},
		{2906, 30.967174302858048},
		{2907, 30.995885840876113},
		{2908, 31.29636960058828},
		{2909, 31.434858618252605},
		{2910, 31.416675748910425},
		{2911, 30.205448705736853},
		{2912, 29.93565034197684},
		{2913, 30.343706590216623},
		{2914, 30.21957109990217},
		{2915, 30.85282680317679},
		{2916, 30.968361747768547},
		{2917, 30.009066802972328},
		{2918, 29.932154181020326},
		{2919, 30.093782671075324},
		{2920, 30.404844874889584},
		{2921, 30.331546653465736},
		{2922, 30.147767848823545},
		{2923, 29.709117723930547},
		{2924, 29.3659833798085},
		{2925, 28.862774209616173},
		{2926, 28.51448224611682},
		{2927, 28.76793928928293},
		{2928, 28.673496746750068},
		{2929, 28.472443770823936},
		{2930, 29.022303006465368},
		{2931, 28.00637473939249},
		{2932, 28.031008433611056},
		{2933, 27.729183209158307},
		{2934, 27.9366018038988},
		{2935, 27.640659462246116},
		{2936, 27.171697419833},
		{2937, 26.408787788436918},
		{2938, 26.28204138551267},
		{2939, 25.932006217741982},
		{2940, 26.169627229499696},
		{2941, 25.822279798439705},
		{2942, 26.35586961220703},
		{2943, 26.18132397957769},
		{2944, 26.044539631613738},
		{2945, 25.974095714129387},
		{2946, 27.05091837318809},
		{2947, 26.974693513444837},
		{2948, 27.072069424664924},
		{2949, 26.25090442405271},
		{2950, 26.73596169363717},
		{2951, 25.905671210399422},
		{2952, 25.691113922860477},
		{2953, 26.299532712205195},
		{2954, 24.96955683778935},
		{2955, 24.888869198579197},
		{2956, 24.601211234973405},
		{2957, 24.034114487943505},
		{2958, 24.024743202429732},
		{2959, 25.019008794538035},
		{2960, 24.432773221953887},
		{2961, 24.544330149848985},
		{2962, 23.85023866590447},
		{2963, 24.036408763069247},
		{2964, 24.08876550655795},
		{2965, 24.05639813443458},
		{2966, 23.71814837085052},
		{2967, 24.703335417148605},
		{2968, 24.77916378099878},
		{2969, 24.839641043240448},
		{2970, 24.853870491674087},
		{2971, 25.840255119538664},
		{2972, 25.818729777629176},
		{2973, 25.73662194677672},
		{2974, 26.312249087826},
		{2975, 26.081048566805045},
		{2976, 26.039636620532686},
		{2977, 26.03568726819535},
		{2978, 26.063007917453888},
		{2979, 26.074627484746603},
		{2980, 26.040651463264155},
		{2981, 26.172289495364286},
		{2982, 26.62269151997738},
		{2983, 26.521078122843306},
		{2984, 26.48143361417722},
		{2985, 26.43597116107479},
		{2986, 26.479253919810066},
		{2987, 26.05243489612348},
		{2988, 26.065403680162337},
		{2989, 26.070748034587236},
		{2990, 26.054077155633216},
		{2991, 25.652831846415367},
		{2992, 25.876662312969824},
		{2993, 26.316574636642432},
		{2994, 26.27465443874206},
		{2995, 26.499843498562846},
		{2996, 26.427205343025584},
		{2997, 26.798259276629466},
		{2998, 26.901808522372313},
		{2999, 26.76079388053373},
		{3000, 26.95186823373628},
		{3001, 27.034524957712406},
		{3002, 26.87614970836558},
		{3003, 26.088631779268834},
		{3004, 26.258487166059158},
		{3005, 26.263116669541553},
		{3006, 26.556642147234385},
		{3007, 27.1359861481772},
		{3008, 27.05636721935326},
		{3009, 27.782632975155},
		{3010, 28.239461231129635},
		{3011, 28.171321453013107},
		{3012, 27.388829894601574},
		{3013, 27.765206394186475},
		{3014, 27.665523688066237},
		{3015, 27.652201388513607},
		{3016, 27.76330159496823},
		{3017, 27.652745228165806},
		{3018, 29.16493980313136},
		{3019, 29.760186326718667},
		{3020, 29.524598241887276},
		{3021, 29.871951567195385},
		{3022, 29.972022031694312},
		{3023, 29.991328977556655},
		{3024, 30.51114039017488},
		{3025, 30.358833052283288},
		{3026, 31.113437416596867},
		{3027, 30.73034455732333},
		{3028, 32.411088299325755},
		{3029, 32.448779387658455},
		{3030, 32.40261565557047},
		{3031, 33.32318930682262},
		{3032, 33.53986533177098},
		{3033, 34.14238204328724},
		{3034, 34.324204651156585},
		{3035, 34.55394232184136},
		{3036, 33.80917147487097},
		{3037, 33.87637554473631},
		{3038, 33.52627570280564},
		{3039, 32.710090552383846},
		{3040, 32.87558231122469},
		{3041, 32.83723732102603},
		{3042, 32.68286627184585},
		{3043, 32.645629659702166},
		{3044, 32.485577854400574},
		{3045, 32.54464973603174},
		{3046, 32.72824138287855},
		{3047, 32.569049941408885},
		{3048, 32.05613996926093},
		{3049, 31.418578354753414},
		{3050, 31.974802932294068},
		{3051, 31.517876027087098},
		{3052, 31.478771719965934},
		{3053, 32.33321726337759},
		{3054, 32.15907422841291},
		{3055, 32.39386557285096},
		{3056, 32.92288104967725},
		{3057, 32.58205659155766},
		{3058, 33.02979168623353},
		{3059, 33.104577943842884},
		{3060, 32.7508051872267},
		{3061, 30.759481316657414},
		{3062, 29.93908316497376},
		{3063, 30.168995523464204},
		{3064, 30.234201200937164},
		{3065, 29.60257745821407},
		{3066, 29.42426454491802},
		{3067, 29.780785754603404},
		{3068, 29.758857379324553},
		{3069, 30.339341677481386},
		{3070, 30.17509659780736},
		{3071, 30.39121521041002},
		{3072, 30.39268093037097},
		{3073, 30.119809979067465},
		{3074, 29.755321684850596},
		{3075, 30.254534421486376},
		{3076, 30.27128056523592},
		{3077, 30.269129730144403},
		{3078, 30.268406185451322},
		{3079, 30.808812815476546},
		{3080, 30.541034763482905},
		{3081, 30.616071356417166},
		{3082, 30.49034629358917},
		{3083, 30.327451475881006},
		{3084, 30.20172837588546},
		{3085, 30.552615556325403},
		{3086, 31.018620896589276},
		{3087, 31.016826954239793},
		{3088, 31.03257471371818},
		{3089, 30.76388835634399},
		{3090, 30.750773848565544},
		{3091, 28.90734498274122},
		{3092, 28.88512555430076},
		{3093, 29.235602659773928},
		{3094, 29.227088109152646},
		{3095, 29.265927078146902},
		{3096, 29.63113527441051},
		{3097, 30.325843427198762},
		{3098, 30.366277557982425},
		{3099, 30.81125656901041},
		{3100, 29.607530596142997},
		{3101, 30.157333515983648},
		{3102, 30.18646934065617},
		{3103, 30.27379415294111},
		{3104, 30.289365730425246},
		{3105, 30.283540247033166},
		{3106, 31.32710063553683},
		{3107, 30.380921899830913},
		{3108, 30.22467645076004},
		{3109, 30.23362298145734},
		{3110, 30.61875215657774},
		{3111, 30.98837784635033},
		{3112, 31.03260739138466},
		{3113, 30.976968690215546},
		{3114, 31.79541274581999},
		{3115, 32.30837337558492},
		{3116, 30.958484720733267},
		{3117, 31.248323567786898},
		{3118, 31.17080295379335},
		{3119, 31.490285494334042},
		{3120, 31.28299461589361},
		{3121, 31.12746538392657},
		{3122, 30.304134609467486},
		{3123, 30.482783363335685},
		{3124, 30.800000711544357},
		{3125, 30.923176972067218},
		{3126, 30.92985171567796},
		{3127, 31.237871497288655},
		{3128, 30.082238264022603},
		{3129, 30.6089077906634},
		{3130, 31.235939986705212},
		{3131, 31.14323375771582},
		{3132, 31.32918643869485},
		{3133, 31.47312641489513},
		{3134, 31.466063254090106},
		{3135, 31.425832509717182},
		{3136, 32.1719990071092},
		{3137, 32.41592307689316},
		{3138, 32.676500312625016},
		{3139, 33.35536331207698},
		{3140, 33.65570232797892},
		{3141, 33.04862310178812},
		{3142, 33.03181458653315},
		{3143, 32.67952645100101},
		{3144, 32.58771553180158},
		{3145, 32.60526161774588},
		{3146, 32.59144411017273},
		{3147, 33.328724603980014},
		{3148, 32.82736116656727},
		{3149, 32.585413549083036},
		{3150, 31.909366787173354},
		{3151, 31.890172235321025},
		{3152, 32.01030930891559},
		{3153, 31.876026787120207},
		{3154, 31.177003426608458},
		{3155, 31.269152245617928},
		{3156, 31.321299373154623},
		{3157, 31.185431050212017},
		{3158, 31.33308403249633},
		{3159, 31.396718431876668},
		{3160, 30.967056835014755},
		{3161, 31.937036635712108},
		{3162, 31.782999539504566},
		{3163, 31.463231624729538},
		{3164, 31.462330205674178},
		{3165, 32.38149340842041},
		{3166, 31.490726766628722},
		{3167, 31.180181319988332},
		{3168, 31.285452919047007},
		{3169, 32.61725672571165},
		{3170, 32.54476600813624},
		{3171, 32.867677415690224},
		{3172, 32.851230576948424},
		{3173, 32.447875933693695},
		{3174, 32.327497251158114},
		{3175, 32.54030609356738},
		{3176, 33.219908261056695},
		{3177, 33.05354548450645},
		{3178, 32.91346861269424},
		{3179, 32.972877044342646},
		{3180, 32.93786082284114},
		{3181, 33.05108392047543},
		{3182, 33.89537603616726},
		{3183, 33.82184220941501},
		{3184, 33.83547506546621},
		{3185, 33.46981809068626},
		{3186, 33.48014088578789},
		{3187, 33.99535008164156},
		{3188, 34.134687943016004},
		{3189, 34.116357932617554},
		{3190, 34.3777412140907},
		{3191, 34.47433796931576},
		{3192, 35.329063289425584},
		{3193, 36.389648583072216},
		{3194, 37.152845814499535},
		{3195, 37.04698767782292},
		{3196, 36.924986340055234},
		{3197, 36.72395148805289},
		{3198, 37.099937243992414},
		{3199, 37.03586955519982},
		{3200, 36.85659515898313},
		{3201, 37.68701715509432},
		{3202, 36.94563230667493},
		{3203, 36.9801530400879},
		{3204, 37.124204726164955},
		{3205, 37.21769294584},
		{3206, 37.36244560706185},
		{3207, 37.177958656048105},
		{3208, 37.17234726327357},
		{3209, 37.2334453881957},
		{3210, 37.443103290909},
		{3211, 37.47601907379215},
		{3212, 37.492358954797},
		{3213, 37.75965833545013},
		{3214, 37.44614357662946},
		{3215, 37.36038063096078},
		{3216, 37.05950813066276},
		{3217, 38.18103211360699},
		{3218, 36.53683047392274},
		{3219, 36.83403875076923},
		{3220, 36.799322175892655},
		{3221, 35.76343601069714},
		{3222, 35.732925176029},
		{3223, 36.030449190392034},
		{3224, 35.53535958118256},
		{3225, 35.46682168640866},
		{3226, 35.73775862190484},
		{3227, 35.61261356902764},
		{3228, 35.60401961780849},
		{3229, 35.7950184685732},
		{3230, 35.680731912648454},
		{3231, 35.91734464456234},
		{3232, 36.215345080856956},
		{3233, 36.169986316646735},
		{3234, 36.19636620447901},
		{3235, 35.68416087517605},
		{3236, 34.782382760280335},
		{3237, 34.682205263960896},
		{3238, 34.6915019903415},
		{3239, 34.01850844249432},
		{3240, 34.49164616505544},
		{3241, 34.81821831651344},
		{3242, 34.17367204635451},
		{3243, 33.50257123928646},
		{3244, 33.74605123682653},
		{3245, 33.207146127232235},
		{3246, 33.350131113450125},
		{3247, 33.31108845186046},
		{3248, 33.25208316113067},
		{3249, 32.96860920898072},
		{3250, 32.81656955503311},
		{3251, 32.596084703753874},
		{3252, 32.72681616233501},
		{3253, 32.989201391855794},
		{3254, 32.720933976649086},
		{3255, 32.76607998687764},
		{3256, 32.971731400496566},
		{3257, 32.978292043059476},
		{3258, 32.96571499076951},
		{3259, 33.54075000081999},
		{3260, 33.428210870837745},
		{3261, 33.37480509177156},
		{3262, 32.04401143262248},
		{3263, 31.758176393318287},
		{3264, 31.640748236625246},
		{3265, 31.64024961699623},
		{3266, 32.12120327486007},
		{3267, 32.09409145385511},
		{3268, 31.82108292829529},
		{3269, 31.767305808090345},
		{3270, 31.61485887968538},
		{3271, 31.25747957686745},
		{3272, 30.343098748976058},
		{3273, 30.310065993327697},
		{3274, 30.08147466217843},
		{3275, 29.06160851755138},
		{3276, 29.649133669716544},
		{3277, 29.479576201649866},
		{3278, 29.53734181807828},
		{3279, 29.875185928183274},
		{3280, 30.026430749893212},
		{3281, 29.070477635815827},
		{3282, 30.080721132734187},
		{3283, 30.213354081612774},
		{3284, 30.584419177806883},
		{3285, 30.554197032740483},
		{3286, 30.743451866698816},
		{3287, 32.72273350370443},
		{3288, 32.55596605650281},
		{3289, 32.58240930965781},
		{3290, 32.42498977194752},
		{3291, 32.61029965109377},
		{3292, 32.9308941823172},
		{3293, 32.89483722129089},
		{3294, 32.85881557046856},
		{3295, 32.726785325793635},
		{3296, 33.04118883100211},
		{3297, 33.83571180198559},
		{3298, 33.647394009337454},
		{3299, 33.520540287573574},
		{3300, 33.162168528911174},
		{3301, 33.508306024446206},
		{3302, 33.525507479986295},
		{3303, 33.52207711917462},
		{3304, 33.627821807306546},
		{3305, 33.198515456466744},
		{3306, 33.42607172054768},
		{3307, 33.07791501061832},
		{3308, 33.107977635263275},
		{3309, 32.842151272849485},
		{3310, 33.05292527827272},
		{3311, 32.957185834270064},
		{3312, 33.06230697948977},
		{3313, 33.61764522661923},
		{3314, 34.3994082345058},
		{3315, 34.29897488531151},
		{3316, 34.21313709031122},
		{3317, 34.15640542042003},
		{3318, 34.189075319669456},
		{3319, 34.065995348334695},
		{3320, 33.99963167652209},
		{3321, 33.824584145650626},
		{3322, 32.62903776472264},
		{3323, 32.07197931490158},
		{3324, 31.212041654609635},
		{3325, 31.35529289602581},
		{3326, 31.795752220664465},
		{3327, 32.055153913003146},
		{3328, 32.120788126976784},
		{3329, 32.11630360272935},
		{3330, 32.137853597557225},
		{3331, 32.09903615115943},
		{3332, 32.30648496829282},
		{3333, 32.07508894046564},
		{3334, 32.234779372557384},
		{3335, 31.75162714209973},
		{3336, 31.175617555355522},
		{3337, 31.132722428185286},
		{3338, 30.740013555537057},
		{3339, 30.80440681916967},
		{3340, 30.389128544023528},
		{3341, 30.923705876216804},
		{3342, 30.570681686489785},
		{3343, 30.56960911154055},
		{3344, 30.572573228321385},
		{3345, 30.56283258873671},
		{3346, 30.75768607862666},
		{3347, 30.70512333660316},
		{3348, 30.73660038831468},
		{3349, 31.14637617533282},
		{3350, 31.20431061055002},
		{3351, 31.20123522063627},
		{3352, 30.97866560996787},
		{3353, 31.09897337898735},
		{3354, 31.592251360854707},
		{3355, 31.658003377873115},
		{3356, 31.749358291626148},
		{3357, 31.615018574293643},
		{3358, 32.13348019004665},
		{3359, 32.26754114906748},
		{3360, 32.114108645990356},
		{3361, 32.09349668953626},
		{3362, 32.12743992074132},
		{3363, 33.26855777434632},
		{3364, 33.20672720628778},
		{3365, 33.222054502547586},
		{3366, 33.276565557113116},
		{3367, 33.24616375772876},
		{3368, 33.22546957275849},
		{3369, 33.197145715564346},
		{3370, 32.89226183861777},
		{3371, 32.63753378800023},
		{3372, 32.85392625336492},
		{3373, 32.654738825317295},
		{3374, 33.52629206488907},
		{3375, 33.551906915417874},
		{3376, 33.013300850717286},
		{3377, 32.45224067089088},
		{3378, 33.536730399855294},
		{3379, 33.382025520697695},
		{3380, 33.65246947890217},
		{3381, 33.729081781626235},
		{3382, 33.68914809456685},
		{3383, 34.27142675843858},
		{3384, 34.3384613970915},
		{3385, 33.041230980966134},
		{3386, 33.01273553419829},
		{3387, 33.81018910800749},
		{3388, 34.24585759532097},
		{3389, 34.61769546449783},
		{3390, 34.83617615053483},
		{3391, 35.197775455649},
		{3392, 35.41194481617064},
		{3393, 36.97802352361716},
		{3394, 36.955361245072574},
		{3395, 36.46776848088892},
		{3396, 36.439365640673294},
		{3397, 37.01957434647059},
		{3398, 36.90031551517573},
		{3399, 36.34320496911004},
		{3400, 36.27540466884174},
		{3401, 35.499445889025864},
		{3402, 35.438767477098544},
		{3403, 35.23819008394313},
		{3404, 35.19377350745659},
		{3405, 35.244867145440864},
		{3406, 34.786303985903714},
		{3407, 34.68233052858934},
		{3408, 34.68606294217492},
		{3409, 35.15865659415544},
		{3410, 36.152459222278075},
		{3411, 36.09225370958755},
		{3412, 36.43348782533814},
		{3413, 37.17740333669563},
		{3414, 37.09959143159908},
		{3415, 37.19283547903976},
		{3416, 37.067139992874246},
		{3417, 37.12896533687519},
		{3418, 37.374869441323746},
		{3419, 37.001095013732005},
		{3420, 35.8685171365076},
		{3421, 35.7790596667844},
		{3422, 35.767099214183794},
		{3423, 35.66846294670802},
		{3424, 36.04001746550384},
		{3425, 36.0480945234249},
		{3426, 36.02197200446126},
		{3427, 35.60312899342081},
		{3428, 35.61743213922281},
		{3429, 35.481336812650724},
		{3430, 35.6265808419118},
		{3431, 35.58736423372758},
		{3432, 34.924601234309385},
		{3433, 34.90979132407057},
		{3434, 35.12660838689912},
		{3435, 35.27414359176062},
		{3436, 34.4572696783567},
		{3437, 34.855188368960206},
		{3438, 34.7694402963776},
		{3439, 34.65799602194358},
		{3440, 34.63813926025997},
		{3441, 34.877437006370286},
		{3442, 34.15314684377819},
		{3443, 34.255400076640996},
		{3444, 34.396642283465276},
		{3445, 33.719365607039315},
		{3446, 33.641859553881055},
		{3447, 33.24778570063584},
		{3448, 33.03425968268789},
		{3449, 31.735215667970092},
		{3450, 32.52074978898301},
		{3451, 32.51001723815836},
		{3452, 31.988903384703473},
		{3453, 32.00635868003232},
		{3454, 33.15034114204495},
		{3455, 33.150708817243185},
		{3456, 33.160443645038676},
		{3457, 33.25145744811404},
		{3458, 31.855456989765536},
		{3459, 31.540691769358283},
		{3460, 31.22731634782854},
		{3461, 30.109943743087637},
		{3462, 30.081678413940462},
		{3463, 30.071462139339655},
		{3464, 31.811852478724177},
		{3465, 31.7817438259946},
		{3466, 31.65842780226553},
		{3467, 31.645232457369303},
		{3468, 31.577808697440936},
		{3469, 31.965622730682476},
		{3470, 31.55851189895187},
		{3471, 31.55400579073177},
		{3472, 31.615309544387085},
		{3473, 30.577964817545922},
		{3474, 31.380957115505957},
		{3475, 31.65967766234796},
		{3476, 31.650632080520086},
		{3477, 31.583511885018947},
		{3478, 31.46866237334157},
		{3479, 31.51539173204228},
		{3480, 31.683674297218584},
		{3481, 31.843475655353362},
		{3482, 31.833474481849517},
		{3483, 31.83374497628001},
		{3484, 31.297115220943716},
		{3485, 31.84548063776127},
		{3486, 30.76165139386353},
		{3487, 30.779380693616407},
		{3488, 30.560174724155342},
		{3489, 31.442644349165654},
		{3490, 31.23456150289414},
		{3491, 31.47065724426264},
		{3492, 31.04388200251078},
		{3493, 31.887381742205424},
		{3494, 31.022715733669997},
		{3495, 31.132724334176533},
		{3496, 31.254131102633508},
		{3497, 29.47070713463634},
		{3498, 30.102350147175464},
		{3499, 30.80347150519261},
		{3500, 31.10662961863165},
		{3501, 31.069676016670755},
		{3502, 31.26759264402205},
		{3503, 31.469315701418683},
		{3504, 31.42829010862853},
		{3505, 31.20304707788888},
		{3506, 30.880459790358785},
		{3507, 31.732791810278144},
		{3508, 31.666863184217398},
		{3509, 31.0372666719452},
		{3510, 30.95557997606788},
		{3511, 31.980223292918765},
		{3512, 32.012017886826364},
		{3513, 31.69007433474044},
		{3514, 31.768581447723278},
		{3515, 32.15852386773846},
		{3516, 31.853809576176754},
		{3517, 31.849132994909127},
		{3518, 32.43271679812625},
		{3519, 31.639885035846955},
		{3520, 31.634677292557605},
		{3521, 31.65931351662495},
		{3522, 31.64099204228493},
		{3523, 31.10736248404926},
		{3524, 30.95348491889584},
		{3525, 30.39386488435902},
		{3526, 30.082639749439995},
		{3527, 31.189586327936507},
		{3528, 31.43839740369831},
		{3529, 31.313018471682966},
		{3530, 31.24190268841608},
		{3531, 31.373635460402006},
		{3532, 31.26635675448505},
		{3533, 30.98293712220932},
		{3534, 31.431319445797822},
		{3535, 30.662258590716984},
		{3536, 31.555087070316418},
		{3537, 32.98014017195333},
		{3538, 32.718573086874514},
		{3539, 32.69374365847009},
		{3540, 32.334428458462085},
		{3541, 32.79981705652948},
		{3542, 32.54809889650045},
		{3543, 32.25529291766223},
		{3544, 31.785544957781138},
		{3545, 31.689101326443183},
		{3546, 31.617220851427742},
		{3547, 31.549033274787455},
		{3548, 32.92223632877409},
		{3549, 32.8330770506699},
		{3550, 33.014591854223255},
		{3551, 33.36379119283432},
		{3552, 33.055540509396565},
		{3553, 32.46582242202945},
		{3554, 32.09927951615547},
		{3555, 32.68733881264074},
		{3556, 33.86320988116308},
		{3557, 33.94494214738513},
		{3558, 33.529412010032885},
		{3559, 33.4888869543669},
		{3560, 33.38035950901418},
		{3561, 33.31012171482195},
		{3562, 33.48535073196302},
		{3563, 33.597392261160415},
		{3564, 33.4512300687611},
		{3565, 32.81838924040587},
		{3566, 33.41685969176084},
		{3567, 33.44158744976233},
		{3568, 31.864631180692385},
		{3569, 31.69972472198296},
		{3570, 31.62978284463028},
		{3571, 33.02109212331147},
		{3572, 33.012962884618574},
		{3573, 32.541542372624065},
		{3574, 31.491346373021337},
		{3575, 31.615350692550003},
		{3576, 31.804732803514614},
		{3577, 31.785970967334098},
		{3578, 31.302173527640477},
		{3579, 31.066768598626613},
		{3580, 30.83902872742016},
		{3581, 30.647189213101132},
		{3582, 30.679283971247305},
		{3583, 30.635220918876687},
		{3584, 30.474194541456203},
		{3585, 30.323171221808224},
		{3586, 30.231880521112476},
		{3587, 30.14165312650818},
		{3588, 30.587932180323367},
		{3589, 30.515797815362596},
		{3590, 30.697580128792886},
		{3591, 30.5092767324158},
		{3592, 30.22933286653618},
		{3593, 30.399883720364244},
		{3594, 30.563655322165122},
		{3595, 30.916284482811637},
		{3596, 31.266903691899042},
		{3597, 31.277953104128557},
		{3598, 31.248343781043335},
		{3599, 31.665546713305297},
		{3600, 31.780736348640186},
		{3601, 32.560705734243456},
		{3602, 32.58949125031766},
		{3603, 32.648379121925466},
		{3604, 32.648994209302636},
		{3605, 32.89262541394924},
		{3606, 32.386706534468246},
		{3607, 33.10476073358268},
		{3608, 32.127562371704016},
		{3609, 32.08021758550007},
		{3610, 32.01828734139976},
		{3611, 32.149616630644026},
		{3612, 32.24939276030207},
		{3613, 31.908269478103314},
		{3614, 31.75980113913783},
		{3615, 31.599177196466123},
		{3616, 30.82883453465096},
		{3617, 30.805848182195113},
		{3618, 30.67015361507686},
		{3619, 30.82551357191438},
		{3620, 31.231900275482825},
		{3621, 31.39570326592559},
		{3622, 31.220478731815824},
		{3623, 31.191057259483284},
		{3624, 31.71256738228641},
		{3625, 31.767353330007293},
		{3626, 31.7421965553774},
		{3627, 32.169320034067475},
		{3628, 32.83045534082564},
		{3629, 33.09082984672529},
		{3630, 33.091418882239445},
		{3631, 33.534698294315376},
		{3632, 34.98860015194777},
		{3633, 34.887785100600965},
		{3634, 34.52612439270598},
		{3635, 35.222998429400846},
		{3636, 35.32696073852328},
		{3637, 33.91614263793164},
		{3638, 33.955178987781},
		{3639, 32.98821123005988},
		{3640, 31.54612425654933},
		{3641, 30.47179194130409},
		{3642, 29.121801087250198},
		{3643, 28.98075756334753},
		{3644, 29.21120341558452},
		{3645, 28.94324522035289},
		{3646, 30.61040164869518},
		{3647, 30.938644800763065},
		{3648, 31.88964085600857},
		{3649, 31.104255236657654},
		{3650, 30.80739952109552},
		{3651, 30.87061979495017},
		{3652, 30.50628535976767},
		{3653, 30.526826937324103},
		{3654, 30.39733248162421},
		{3655, 30.396253710063263},
		{3656, 30.500732657862326},
		{3657, 30.464941464184626},
		{3658, 30.397674823493038},
		{3659, 31.559828803294607},
		{3660, 30.4775995110338},
		{3661, 30.605373430398316},
		{3662, 30.959781079195544},
		{3663, 30.686376335617936},
		{3664, 30.45394463567413},
		{3665, 30.836569443595376},
		{3666, 30.41004287663308},
		{3667, 30.76197883114409},
		{3668, 30.17037689073509},
		{3669, 30.057913986094512},
		{3670, 30.087583283388085},
		{3671, 30.21878224005264},
		{3672, 30.246448076959325},
		{3673, 30.58839113549538},
		{3674, 31.47426930437642},
		{3675, 31.010529526010682},
		{3676, 31.303422051514435},
		{3677, 31.003100220279816},
		{3678, 30.98326209088546},
		{3679, 30.69183779794675},
		{3680, 31.223367625034683},
		{3681, 30.789280075755688},
		{3682, 30.19829216553304},
		{3683, 30.363595369442493},
		{3684, 30.27935553254929},
		{3685, 30.1261971935347},
		{3686, 30.79863985302651},
		{3687, 31.047595403218057},
		{3688, 30.90228673518786},
		{3689, 30.543388017289054},
		{3690, 30.704356144092372},
		{3691, 30.63973229439528},
		{3692, 30.68310619503808},
		{3693, 30.603549759307885},
		{3694, 31.374934823463768},
		{3695, 31.389460746452443},
		{3696, 31.904273713484496},
		{3697, 31.885063882176844},
		{3698, 31.820524013817312},
		{3699, 31.82770668437377},
		{3700, 31.62129391692174},
		{3701, 33.01039885910495},
		{3702, 32.810936220934124},
		{3703, 33.11606920366353},
		{3704, 33.45236130557991},
		{3705, 34.35768750096602},
		{3706, 33.480461372240505},
		{3707, 33.73411429026118},
		{3708, 33.89279065017372},
		{3709, 34.23881544215997},
		{3710, 34.935789977307365},
		{3711, 34.77481188799792},
		{3712, 34.73666168190101},
		{3713, 34.049290860266446},
		{3714, 34.3821020789841},
		{3715, 34.6030641653466},
		{3716, 34.69594103398987},
		{3717, 34.17980982257179},
		{3718, 34.16518765028321},
		{3719, 34.23763587885022},
		{3720, 34.601596281681694},
		{3721, 35.4175582873499},
		{3722, 35.59749071347688},
		{3723, 34.32925786634385},
		{3724, 34.51637645217182},
		{3725, 34.525950151725944},
		{3726, 35.998069795400646},
		{3727, 35.963551274884786},
		{3728, 36.036689984025955},
		{3729, 37.24781768842208},
		{3730, 37.277514336598095},
		{3731, 37.30863902603817},
		{3732, 36.79217135191674},
		{3733, 36.79064349318625},
		{3734, 36.001098652284306},
		{3735, 35.86481428501773},
		{3736, 35.94802206870304},
		{3737, 37.14027447150489},
		{3738, 36.258549896818224},
		{3739, 36.802348037612525},
		{3740, 36.89335332794575},
		{3741, 37.908095460582},
		{3742, 37.71387537919735},
		{3743, 37.25060233029356},
		{3744, 36.911971103005506},
		{3745, 36.621626787271495},
		{3746, 36.78598563390066},
		{3747, 36.78867687247616},
		{3748, 36.846211552881314},
		{3749, 36.68451178474116},
		{3750, 36.30880218835804},
		{3751, 36.74490372547481},
		{3752, 36.93776738041527},
		{3753, 36.92794573529495},
		{3754, 36.88702244570728},
		{3755, 37.032752352208334},
		{3756, 37.09031587985092},
		{3757, 37.25861213788812},
		{3758, 37.0208483030904},
		{3759, 37.44627314073198},
		{3760, 36.60326002825027},
		{3761, 36.5349242191796},
		{3762, 36.18844767465498},
		{3763, 36.58284312296601},
		{3764, 36.501734860814345},
		{3765, 36.047516333562704},
		{3766, 35.32968089985233},
		{3767, 36.28983500653037},
		{3768, 35.9811194049413},
		{3769, 36.10451903716811},
		{3770, 36.34687133526118},
		{3771, 36.31609256803086},
		{3772, 37.63818828180108},
		{3773, 38.43083167710727},
		{3774, 38.457277972233115},
		{3775, 38.50074370521922},
		{3776, 38.57905462745658},
		{3777, 37.94556968927861},
		{3778, 38.250174481197824},
		{3779, 38.87016646551586},
		{3780, 38.96356411591443},
		{3781, 38.81436757446562},
		{3782, 38.70813805142539},
		{3783, 38.36257955910033},
		{3784, 38.334587300767666},
		{3785, 38.573334533834945},
		{3786, 38.706937363024785},
		{3787, 38.456878484269325},
		{3788, 38.59652287562196},
		{3789, 38.87303482456266},
		{3790, 38.26707578973226},
		{3791, 38.08710058110894},
		{3792, 38.107611873182606},
		{3793, 37.87348574106414},
		{3794, 38.467343581667315},
		{3795, 38.52284971229728},
		{3796, 38.80030552742671},
		{3797, 39.31444077052365},
		{3798, 39.36714339199697},
		{3799, 39.265510040584786},
		{3800, 39.35757249633458},
		{3801, 39.23029087940918},
		{3802, 40.236476990129525},
		{3803, 40.40447274896678},
		{3804, 39.90050059379676},
		{3805, 39.82968538918735},
		{3806, 39.568796564237815},
		{3807, 39.66844747677395},
		{3808, 39.48947001553621},
		{3809, 39.07804466925928},
		{3810, 39.08468133372294},
		{3811, 39.409077167178886},
		{3812, 39.59388813521032},
		{3813, 39.61292554954707},
		{3814, 39.53006069194008},
		{3815, 39.43424867493228},
		{3816, 39.364325046233276},
		{3817, 39.24878540639196},
		{3818, 38.91030638030182},
		{3819, 39.19021407549976},
		{3820, 39.52465873164222},
		{3821, 39.30579054542478},
		{3822, 39.066334049300906},
		{3823, 38.445919893706396},
		{3824, 38.82331512394734},
		{3825, 38.722751358387455},
		{3826, 39.18918989725993},
		{3827, 39.3325423377068},
		{3828, 40.07164954275041},
		{3829, 40.07074505067318},
		{3830, 40.27305857026511},
		{3831, 40.29841893573262},
		{3832, 40.35632841195591},
		{3833, 41.25954776973257},
		{3834, 41.08661644599172},
		{3835, 40.930860440282444},
		{3836, 40.67035270986339},
		{3837, 40.59686023107034},
		{3838, 40.966306832072135},
		{3839, 41.73206011126974},
		{3840, 41.83313880921912},
		{3841, 42.710027568498774},
		{3842, 43.09298354963694},
		{3843, 43.97481631223127},
		{3844, 43.06984802315324},
		{3845, 42.88682049659426},
		{3846, 42.70391340792306},
		{3847, 42.67733443704977},
		{3848, 42.312517547282376},
		{3849, 42.68248973003122},
		{3850, 41.89851060373508},
		{3851, 42.499365091434946},
		{3852, 42.04285692039637},
		{3853, 42.36335659371158},
		{3854, 43.00859290667485},
		{3855, 42.64595368210728},
		{3856, 42.84344419146314},
		{3857, 42.58554402244061},
		{3858, 42.94977322974173},
		{3859, 42.970561962196946},
		{3860, 43.09387131088587},
		{3861, 43.5526747563676},
		{3862, 43.59839466763858},
		{3863, 43.491401190645476},
		{3864, 43.890847555316626},
		{3865, 43.90554980755929},
		{3866, 43.29419748320046},
		{3867, 43.19271817040813},
		{3868, 43.15798450802592},
		{3869, 44.936354699949135},
		{3870, 44.925896564304466},
		{3871, 45.141313850266805},
		{3872, 45.15805553382705},
		{3873, 45.175146274315395},
		{3874, 44.602873917897206},
		{3875, 43.98425210064129},
		{3876, 44.21112444021221},
		{3877, 44.31207946892926},
		{3878, 44.32915256192521},
		{3879, 44.54003640246446},
		{3880, 44.43388783676019},
		{3881, 44.377913951958845},
		{3882, 44.08660155118512},
		{3883, 44.084088844428344},
		{3884, 44.678908282742874},
		{3885, 44.69090507621899},
		{3886, 44.66399823516045},
		{3887, 44.70003406608551},
		{3888, 44.64217161352199},
		{3889, 44.598427607594914},
		{3890, 44.64183638547881},
		{3891, 45.313385126126406},
		{3892, 45.213056881203656},
		{3893, 45.83751311620533},
		{3894, 45.97798477844367},
		{3895, 46.69554397624657},
		{3896, 46.780476914271844},
		{3897, 46.42161452367823},
		{3898, 46.485265361946965},
		{3899, 46.463419227463945},
		{3900, 46.40619518750473},
		{3901, 46.06654509998825},
		{3902, 46.57689530438109},
		{3903, 46.23083598006043},
		{3904, 46.574270685203786},
		{3905, 46.78507045779222},
		{3906, 46.762325322870765},
		{3907, 47.013171115464665},
		{3908, 47.35684909712627},
		{3909, 47.46573772776851},
		{3910, 47.79168684198062},
		{3911, 46.864811059928705},
		{3912, 46.19102604217612},
		{3913, 45.57512279475959},
		{3914, 45.27697004686091},
		{3915, 44.43233105401431},
		{3916, 44.48913258511433},
		{3917, 44.05739070330766},
		{3918, 43.43326838667846},
		{3919, 43.18101584512046},
		{3920, 43.12661207742961},
		{3921, 43.20235581216143},
		{3922, 43.21124585054493},
		{3923, 43.54158862241011},
		{3924, 43.50571782863192},
		{3925, 43.711364180631804},
		{3926, 43.18168675879529},
		{3927, 42.89854097019134},
		{3928, 42.910583730646124},
		{3929, 43.18510405314944},
		{3930, 42.639952988135626},
		{3931, 42.797465701366804},
		{3932, 43.44286812172023},
		{3933, 43.19594242280566},
		{3934, 43.423279844975504},
		{3935, 43.637554324106404},
		{3936, 43.56664792791941},
		{3937, 44.47718122890482},
		{3938, 42.71893554253818},
		{3939, 42.68619011219234},
		{3940, 42.764744246822005},
		{3941, 42.368141211578596},
		{3942, 42.72177242428145},
		{3943, 42.58290514752061},
		{3944, 42.500109121456575},
		{3945, 43.06917028887632},
		{3946, 43.1390878438651},
		{3947, 43.67566420376484},
		{3948, 43.32879166406192},
		{3949, 43.12282803410909},
		{3950, 43.53504871953674},
		{3951, 43.11341790856445},
		{3952, 43.09460900193214},
		{3953, 42.975278393392486},
		{3954, 43.02716626297875},
		{3955, 43.28841760595471},
		{3956, 43.15013781306844},
		{3957, 43.05005871018039},
		{3958, 42.643291525484216},
		{3959, 41.40330273303392},
		{3960, 41.190483201302776},
		{3961, 41.18851255115125},
		{3962, 41.139227053508684},
		{3963, 41.262957426835335},
		{3964, 41.56188105204588},
		{3965, 41.29178612732812},
		{3966, 42.1903083759515},
		{3967, 43.24502540234582},
		{3968, 43.53636550586511},
		{3969, 43.2653460646287},
		{3970, 43.08283306703269},
		{3971, 42.4092975371573},
		{3972, 42.339747488590305},
		{3973, 42.17938430330037},
		{3974, 42.9302231459865},
		{3975, 44.14104375702491},
		{3976, 44.32654181837024},
		{3977, 44.85103730540091},
		{3978, 43.686022699020185},
		{3979, 43.52045377054748},
		{3980, 44.066949791075196},
		{3981, 43.54943797407643},
		{3982, 43.55681098163001},
		{3983, 43.62378293786004},
		{3984, 43.27719869525927},
		{3985, 43.520080202226026},
		{3986, 43.48833320696304},
		{3987, 43.528692784115066},
		{3988, 43.39572733345793},
		{3989, 43.66669909833283},
		{3990, 43.627624602729895},
		{3991, 44.99085408453511},
		{3992, 44.957885207190635},
		{3993, 44.75608239056898},
		{3994, 44.79205318911418},
		{3995, 45.071567286436604},
		{3996, 45.0773216187218},
		{3997, 45.20628655580068},
		{3998, 45.042340799331626},
		{3999, 44.40459780963831},
		{4000, 44.84911779417697},
		{4001, 44.84679216126694},
		{4002, 44.9926860827092},
		{4003, 45.332476785661704},
		{4004, 45.32599258516697},
		{4005, 44.97196102902279},
		{4006, 44.975532895597816},
		{4007, 45.03190754404627},
		{4008, 43.66039558703255},
		{4009, 43.82989258131152},
		{4010, 44.06126134232322},
		{4011, 44.30529606181042},
		{4012, 44.44658393531378},
		{4013, 43.738910733073006},
		{4014, 44.212806733919},
		{4015, 44.10630077043948},
		{4016, 43.99563053457421},
		{4017, 44.095566651377375},
		{4018, 43.83450825940757},
		{4019, 43.84846576986777},
		{4020, 42.71505037386405},
		{4021, 43.00488908696087},
		{4022, 42.4995621932694},
		{4023, 42.28588980382473},
		{4024, 42.304954106759766},
		{4025, 41.580902668274064},
		{4026, 42.195928955837026},
		{4027, 42.32891452704911},
		{4028, 42.394421598632725},
		{4029, 42.88514428175226},
		{4030, 42.88411017501612},
		{4031, 42.44465234171097},
		{4032, 42.423161360690465},
		{4033, 42.06024809501421},
		{4034, 41.46983767233033},
		{4035, 41.450540092235606},
		{4036, 41.551824782132655},
		{4037, 41.497581890358525},
		{4038, 41.74337678928221},
		{4039, 42.01458088760071},
		{4040, 42.000161946828975},
		{4041, 42.09809076103792},
		{4042, 40.40121722063858},
		{4043, 40.405347534544276},
		{4044, 41.45596015966376},
		{4045, 41.34524024668175},
		{4046, 41.48790124860581},
		{4047, 40.85226478482957},
		{4048, 41.72644422704275},
		{4049, 42.28054826801833},
		{4050, 42.470889350803844},
		{4051, 42.58707283732388},
		{4052, 43.376033472640444},
		{4053, 43.273463592859194},
		{4054, 42.91653543267234},
		{4055, 43.68837327519839},
		{4056, 44.35550220099082},
		{4057, 44.303862119305684},
		{4058, 44.268994539070036},
		{4059, 44.12291558250871},
		{4060, 44.15997387053544},
		{4061, 43.847891173179995},
		{4062, 43.88605800805093},
		{4063, 43.76622485923872},
		{4064, 44.08049094209474},
		{4065, 43.52840705131174},
		{4066, 43.47588617544269},
		{4067, 43.42547149366908},
		{4068, 43.620164305725254},
		{4069, 43.62469613986601},
		{4070, 43.35722721012197},
		{4071, 43.11523304678633},
		{4072, 43.21270244719499},
		{4073, 42.799008504053795},
		{4074, 42.579441010470354},
		{4075, 43.616467334850654},
		{4076, 44.21857842681485},
		{4077, 44.096520800225065},
		{4078, 44.076122818906754},
		{4079, 43.882145702606714},
		{4080, 44.025173271395175},
		{4081, 44.08675228361723},
		{4082, 44.08280580012843},
		{4083, 44.089090764695634},
		{4084, 44.63027712418217},
		{4085, 44.64257806553344},
		{4086, 44.63488930186029},
		{4087, 43.49185769105879},
		{4088, 43.81922760361489},
		{4089, 44.55770909655596},
		{4090, 44.495768552475916},
		{4091, 44.70529568081817},
		{4092, 44.608267360939585},
		{4093, 44.32192140208638},
		{4094, 44.32913149339295},
		{4095, 43.92083958045917},
		{4096, 43.93629699159078},
		{4097, 44.42006096186749},
		{4098, 44.436560493979954},
		{4099, 45.03381928301654},
		{4100, 44.89092252034728},
		{4101, 45.228836061954944},
		{4102, 44.96315616449893},
		{4103, 45.317252160358144},
		{4104, 45.3699223441705},
		{4105, 44.99809335119828},
		{4106, 45.31350010825937},
		{4107, 45.28642508634763},
		{4108, 45.23753191697613},
		{4109, 45.55004475954644},
		{4110, 46.058774601853955},
		{4111, 45.83557796236126},
		{4112, 45.67893880941216},
		{4113, 45.58586429125116},
		{4114, 45.483336441906616},
		{4115, 44.95324979965775},
		{4116, 44.766511487040916},
		{4117, 44.863928712529},
		{4118, 45.521105859507294},
		{4119, 45.597789271022776},
		{4120, 45.548218639658785},
		{4121, 45.5671964393491},
		{4122, 45.66840481231619},
		{4123, 45.421280141591325},
		{4124, 45.42131956848279},
		{4125, 46.48565429648017},
		{4126, 46.691978861656345},
		{4127, 46.7203310903226},
		{4128, 47.83269632839022},
		{4129, 47.818072148512215},
		{4130, 47.904859885869236},
		{4131, 47.7041273974865},
		{4132, 47.69720727111671},
		{4133, 48.00314297086582},
		{4134, 46.56039022043427},
		{4135, 44.90413846099235},
		{4136, 45.52514774325245},
		{4137, 46.058621957431804},
		{4138, 46.12588682589276},
		{4139, 46.58227592497253},
		{4140, 46.17055554294809},
		{4141, 46.80784968189301},
		{4142, 46.49508518393976},
		{4143, 46.500309270278464},
		{4144, 46.414835172420304},
		{4145, 46.304075582190514},
		{4146, 46.22819939150845},
		{4147, 45.85490163160914},
		{4148, 46.16449836318747},
		{4149, 46.36607159615689},
		{4150, 45.33765696135226},
		{4151, 45.51784949111958},
		{4152, 45.27464232646123},
		{4153, 45.64354971534378},
		{4154, 45.57676214832657},
		{4155, 45.56569880057714},
		{4156, 45.645824829272414},
		{4157, 45.6060530616635},
		{4158, 45.78376908108006},
		{4159, 45.92037288384215},
		{4160, 45.93018712100647},
		{4161, 45.04435615370153},
		{4162, 43.97591673735583},
		{4163, 43.60804290415924},
		{4164, 43.25559660341626},
		{4165, 43.30560502979643},
		{4166, 43.32162929680021},
		{4167, 43.220797929589125},
		{4168, 43.88958904031423},
		{4169, 43.92613927037596},
		{4170, 43.28570468545945},
		{4171, 43.33758546950846},
		{4172, 43.38200781871171},
		{4173, 43.37439151387566},
		{4174, 43.62114145159183},
		{4175, 42.583155075251156},
		{4176, 41.79257202974383},
		{4177, 41.5301578370286},
		{4178, 40.72846655122725},
		{4179, 40.9074303438794},
		{4180, 41.486828841681415},
		{4181, 41.43693955839994},
		{4182, 41.25469470037508},
		{4183, 41.3152306690082},
		{4184, 42.05756123320117},
		{4185, 41.976765116312805},
		{4186, 42.55329890940964},
		{4187, 42.32763877115726},
		{4188, 42.331621409998704},
		{4189, 41.678572270110294},
		{4190, 41.78167622000538},
		{4191, 41.57841592927911},
		{4192, 41.41582587790593},
		{4193, 40.829011801535664},
		{4194, 42.57985084363515},
		{4195, 42.4375122885433},
		{4196, 42.32575825185156},
		{4197, 42.800600639022605},
		{4198, 42.07267000939594},
		{4199, 41.991890779837725},
		{4200, 42.01125834670128},
		{4201, 42.01783872316503},
		{4202, 41.99633327331731},
		{4203, 42.2264225459579},
		{4204, 42.41931446868698},
		{4205, 42.5104434476498},
		{4206, 42.51820041440961},
		{4207, 43.10274662201198},
		{4208, 43.03752438898408},
		{4209, 43.19323932573483},
		{4210, 43.15253357742187},
		{4211, 42.158082291323616},
		{4212, 42.18530485919557},
		{4213, 42.179672734709584},
		{4214, 42.08370843694731},
		{4215, 42.062223021469975},
		{4216, 41.67838942952538},
		{4217, 41.49506764318014},
		{4218, 41.59881716888323},
		{4219, 41.926024236261355},
		{4220, 42.11947567033156},
		{4221, 42.26187787539394},
		{4222, 42.65015253639434},
		{4223, 42.441042770154844},
		{4224, 43.08690587956467},
		{4225, 44.39310986424937},
		{4226, 44.94461257206579},
		{4227, 44.78804776055559},
		{4228, 45.53797859707034},
		{4229, 45.55187883708798},
		{4230, 45.88537683206737},
		{4231, 45.825185517872704},
		{4232, 46.07574947104996},
		{4233, 46.060884716412254},
		{4234, 45.76160043499644},
		{4235, 45.76931072837809},
		{4236, 45.51959493255436},
		{4237, 45.554677642229564},
		{4238, 45.52426093953901},
		{4239, 46.109945660686115},
		{4240, 45.99257414377729},
		{4241, 46.07894808606258},
		{4242, 44.286086348874676},
		{4243, 44.4865400708932},
		{4244, 44.234913797823154},
		{4245, 44.16033974830902},
		{4246, 43.44966049381201},
		{4247, 43.08447818539749},
		{4248, 43.205930480034986},
		{4249, 43.27174171858989},
		{4250, 43.37829207780095},
		{4251, 43.10867398251995},
		{4252, 43.143056746791466},
		{4253, 43.184584390833216},
		{4254, 43.2384965784782},
		{4255, 43.64269672822312},
		{4256, 43.95024182576365},
		{4257, 44.34389839798999},
		{4258, 44.779096753781424},
		{4259, 44.81584343357247},
		{4260, 44.862897971461486},
		{4261, 44.809095330679426},
		{4262, 44.82258689695792},
		{4263, 44.94257959545199},
		{4264, 44.70649326700459},
		{4265, 44.83403801956116},
		{4266, 44.69375741776056},
		{4267, 44.246909347585806},
		{4268, 44.31131992100262},
		{4269, 44.30870396354328},
		{4270, 45.277184693727},
		{4271, 45.325741450057016},
		{4272, 45.35787945327525},
		{4273, 46.26696427252453},
		{4274, 46.3349833461467},
		{4275, 46.262367003647526},
		{4276, 45.88415082220467},
		{4277, 45.69088772324733},
		{4278, 46.30744653697344},
		{4279, 46.109976226034966},
		{4280, 46.00659771921764},
		{4281, 45.98327288475588},
		{4282, 45.93683158647156},
		{4283, 45.827017445641175},
		{4284, 44.898780786668084},
		{4285, 44.857946447901604},
		{4286, 44.78129029650326},
		{4287, 44.460979780768575},
		{4288, 43.781370713844545},
		{4289, 43.80346385692286},
		{4290, 43.70136533034899},
		{4291, 43.59605587452415},
		{4292, 46.60711089281704},
		{4293, 43.20999116221206},
		{4294, 43.61705810567689},
		{4295, 43.10115465773163},
		{4296, 43.384267987646375},
		{4297, 43.335817467294284},
		{4298, 43.450535768186434},
		{4299, 43.27624559738883},
		{4300, 43.74264705432361},
		{4301, 43.80047106789878},
		{4302, 43.1793620620206},
		{4303, 44.06844891439754},
		{4304, 44.122958015118876},
		{4305, 44.16868478059633},
		{4306, 43.763355009011846},
		{4307, 44.24245012746741},
		{4308, 43.6667108237416},
		{4309, 43.44755931800053},
		{4310, 43.8388540119679},
		{4311, 43.84684532813053},
		{4312, 43.891606967283025},
		{4313, 43.61609318450177},
		{4314, 43.50391313644561},
		{4315, 43.4370124294098},
		{4316, 43.320512498718685},
		{4317, 43.43317762888966},
		{4318, 43.441565229865844},
		{4319, 43.89166237920707},
		{4320, 43.79297096639801},
		{4321, 42.86616397750371},
		{4322, 42.492817070603124},
		{4323, 42.463597734904255},
		{4324, 42.099203979340935},
		{4325, 43.148352379283736},
		{4326, 43.14393983822119},
		{4327, 43.802920465762334},
		{4328, 43.787438894126545},
		{4329, 44.8390211407645},
		{4330, 43.9210831019679},
		{4331, 44.160162794499165},
		{4332, 44.603706152703076},
		{4333, 44.6029921449433},
		{4334, 45.71896157142412},
		{4335, 45.7186083077643},
		{4336, 46.132237714004425},
		{4337, 45.29810811659616},
		{4338, 45.44893704926406},
		{4339, 44.56756839730384},
		{4340, 44.84857422415435},
		{4341, 44.59101148982434},
		{4342, 45.832174813792975},
		{4343, 46.24568332480992},
		{4344, 46.27972490525816},
		{4345, 46.220037760327614},
		{4346, 46.26785697686892},
		{4347, 45.91029124256099},
		{4348, 46.01333033455775},
		{4349, 45.55640134102827},
		{4350, 45.07332328943664},
		{4351, 44.05034495051119},
		{4352, 44.37631641742659},
		{4353, 44.47510379397457},
		{4354, 44.481536987385375},
		{4355, 45.09815452419815},
		{4356, 44.23545269297595},
		{4357, 44.35190888940349},
		{4358, 44.44376512206195},
		{4359, 44.35377582256026},
		{4360, 44.182956840667266},
		{4361, 44.4297999192324},
		{4362, 44.13473879958124},
		{4363, 44.055995402850435},
		{4364, 44.005471335161786},
		{4365, 43.25758809794344},
		{4366, 43.817908095986404},
		{4367, 43.822056761477334},
		{4368, 44.14477410502386},
		{4369, 43.797486505074176},
		{4370, 43.79800963408963},
		{4371, 42.8796997482786},
		{4372, 42.88556424491737},
		{4373, 42.77727527309622},
		{4374, 42.35778308150795},
		{4375, 42.16975466552083},
		{4376, 41.93028553518078},
		{4377, 42.09563715627173},
		{4378, 41.17204610689614},
		{4379, 41.78572427771177},
		{4380, 41.92150205271964},
		{4381, 42.09741338103042},
		{4382, 43.69386481163199},
		{4383, 42.96742035210106},
		{4384, 42.343580110976134},
		{4385, 42.22399465949221},
		{4386, 42.6091386627358},
		{4387, 42.568896188906415},
		{4388, 42.64410302487118},
		{4389, 41.95886131685545},
		{4390, 40.80681239206415},
		{4391, 39.38331357790885},
		{4392, 39.78427361503698},
		{4393, 39.93810078981509},
		{4394, 40.13352169285055},
		{4395, 40.21778170710591},
		{4396, 40.16970625785058},
		{4397, 40.49257098503543},
		{4398, 39.526963187820655},
		{4399, 39.39632170400282},
		{4400, 38.65146414419247},
		{4401, 38.69622195198017},
		{4402, 38.82824084957831},
		{4403, 38.35894467631948},
		{4404, 38.374288310893164},
		{4405, 38.30201987865161},
		{4406, 39.040498102853704},
		{4407, 38.739032917012814},
		{4408, 38.21513757304971},
		{4409, 37.79412671670286},
		{4410, 37.572863184633725},
		{4411, 36.71659398164643},
		{4412, 36.9686014580944},
		{4413, 36.61396796753844},
		{4414, 36.21992527315708},
		{4415, 36.544907354858005},
		{4416, 36.94702029184928},
		{4417, 37.040712123103496},
		{4418, 37.86983316239789},
		{4419, 37.9133259993023},
		{4420, 37.740462632345725},
		{4421, 37.80334582875091},
		{4422, 38.06669175124991},
		{4423, 38.82569943174218},
		{4424, 38.79645166307203},
		{4425, 38.138183023387136},
		{4426, 37.343389010610906},
		{4427, 37.06893203484559},
		{4428, 36.974042644355606},
		{4429, 37.10878943344164},
		{4430, 36.92199614208741},
		{4431, 36.31927104688222},
		{4432, 36.318646750338225},
		{4433, 35.93959392178788},
		{4434, 35.957928276775064},
		{4435, 36.57803220920125},
		{4436, 36.48729877130802},
		{4437, 36.08934258703699},
		{4438, 35.92075815637091},
		{4439, 35.97790853205448},
		{4440, 37.10428719217352},
		{4441, 37.34763949569804},
		{4442, 37.54555366814172},
		{4443, 38.21664941647024},
		{4444, 38.15910673215726},
		{4445, 38.18389302943226},
		{4446, 38.15032771707228},
		{4447, 38.257644304614324},
		{4448, 37.89935320642902},
		{4449, 37.92806972052069},
		{4450, 38.44876254366418},
		{4451, 38.39207298891007},
		{4452, 37.688682446222174},
		{4453, 37.5187537924215},
		{4454, 37.51914139540737},
		{4455, 37.09540108270616},
		{4456, 37.73546306496036},
		{4457, 37.81369411954859},
		{4458, 36.83626164610845},
		{4459, 36.93994071270752},
		{4460, 36.95018579100306},
		{4461, 37.26709009753631},
		{4462, 37.15237030254232},
		{4463, 37.049394970609185},
		{4464, 37.7162630971784},
		{4465, 37.778299282663106},
		{4466, 38.730315146387746},
		{4467, 38.7451015147541},
		{4468, 38.61705719495839},
		{4469, 37.872865556491874},
		{4470, 37.89246439250997},
		{4471, 38.16788956159939},
		{4472, 37.860151832562025},
		{4473, 37.73726372456453},
		{4474, 37.69142767992998},
		{4475, 37.63232006550031},
		{4476, 37.423877674083585},
		{4477, 37.4556569056629},
		{4478, 37.96421499308509},
		{4479, 37.81211863240172},
		{4480, 38.05702647753458},
		{4481, 38.210884926434076},
		{4482, 38.312564817548534},
		{4483, 38.6074646961431},
		{4484, 38.61409094278309},
		{4485, 38.017765676373614},
		{4486, 38.13255323384495},
		{4487, 38.08371503820191},
		{4488, 39.74784443380604},
		{4489, 39.581076576624355},
		{4490, 39.560969587140264},
		{4491, 39.633230280700445},
		{4492, 39.4716793618942},
		{4493, 39.85116292164824},
		{4494, 40.13334579775557},
		{4495, 40.63456793316489},
		{4496, 40.71073257614338},
		{4497, 40.614825837238605},
		{4498, 40.52093195142476},
		{4499, 40.625992282699016},
		{4500, 40.61279333802842},
		{4501, 40.60312836705065},
		{4502, 40.45124308705166},
		{4503, 39.813179582460556},
		{4504, 39.40056608371914},
		{4505, 39.400036014144824},
		{4506, 40.068292685250626},
		{4507, 39.85851159258621},
		{4508, 40.22426761068521},
		{4509, 40.20179430857556},
		{4510, 40.195867059687664},
		{4511, 40.17841644766825},
		{4512, 40.33883934166413},
		{4513, 40.43097043722167},
		{4514, 39.97935686013634},
		{4515, 40.35201602305827},
		{4516, 40.362381301668016},
		{4517, 41.646176394626345},
		{4518, 41.090525592460445},
		{4519, 40.79730768415044},
		{4520, 40.72921199361499},
		{4521, 40.41889455210564},
		{4522, 40.379424134550845},
		{4523, 40.39436450202139},
		{4524, 40.009568103338225},
		{4525, 40.60519761105382},
		{4526, 40.20213007918731},
		{4527, 40.39015160569152},
		{4528, 39.9454660982101},
		{4529, 39.075001110176856},
		{4530, 39.13454982085495},
		{4531, 39.648058502641085},
		{4532, 39.648144119118776},
		{4533, 39.324207361611386},
		{4534, 39.53997526505957},
		{4535, 39.97557044175399},
		{4536, 40.26787350849},
		{4537, 40.17369778686508},
		{4538, 40.44006641312129},
		{4539, 40.57083491561394},
		{4540, 40.66689326871952},
		{4541, 40.65187955527909},
		{4542, 39.934274637704824},
		{4543, 40.300330136488476},
		{4544, 39.967275063438045},
		{4545, 40.042315039135865},
		{4546, 40.142830573608606},
		{4547, 40.061258001400915},
		{4548, 40.404909259369646},
		{4549, 40.450392865002364},
		{4550, 40.87269030449907},
		{4551, 41.296599615123064},
		{4552, 41.304588768416615},
		{4553, 41.90167025496052},
		{4554, 41.923025365981076},
		{4555, 42.16170196977729},
		{4556, 42.10361415553816},
		{4557, 42.10129865988418},
		{4558, 42.31197566235926},
		{4559, 42.826247977993646},
		{4560, 42.74728359084352},
		{4561, 42.667331833071664},
		{4562, 42.928631961847074},
		{4563, 43.00083178487252},
		{4564, 43.37357490404957},
		{4565, 43.2237675498371},
		{4566, 43.86137004897719},
		{4567, 43.98416288339857},
		{4568, 44.0470084423368},
		{4569, 44.11391700866567},
		{4570, 43.33351237600878},
		{4571, 43.34757034996372},
		{4572, 44.14608921239006},
		{4573, 43.315144980691606},
		{4574, 43.1647218607577},
		{4575, 42.10202383999383},
		{4576, 42.045575631273714},
		{4577, 41.92172379039672},
		{4578, 41.61112748037281},
		{4579, 41.46655883593124},
		{4580, 42.135593115850845},
		{4581, 42.28588201326873},
		{4582, 42.15472962179464},
		{4583, 41.752838975392386},
		{4584, 41.91347248013341},
		{4585, 41.92681361497359},
		{4586, 41.9507666881852},
		{4587, 41.90301229148557},
		{4588, 41.95079936213436},
		{4589, 42.01550498847925},
		{4590, 42.13012135355233},
		{4591, 42.119576747364945},
		{4592, 42.11677669401778},
		{4593, 41.43172999572018},
		{4594, 41.49388864519924},
		{4595, 41.37380290015346},
		{4596, 41.370035845679794},
		{4597, 41.81317893158739},
		{4598, 41.718756954520984},
		{4599, 41.7298553763554},
		{4600, 41.53894095306525},
		{4601, 41.75447883581544},
		{4602, 41.885739012000045},
		{4603, 42.06290639339678},
		{4604, 42.01765532193647},
		{4605, 42.09963693863422},
		{4606, 42.121746882846026},
		{4607, 42.40289053451183},
		{4608, 42.43904810148321},
		{4609, 42.073085450783054},
		{4610, 42.003884006662666},
		{4611, 42.02391206710711},
		{4612, 42.43143955801055},
		{4613, 42.418444114505455},
		{4614, 42.50130186535948},
		{4615, 42.50861804985267},
		{4616, 42.8667351445971},
		{4617, 43.2708291680743},
		{4618, 43.258875230182355},
		{4619, 43.41983125615069},
		{4620, 43.02540169744918},
		{4621, 43.03671781336059},
		{4622, 43.51115813355301},
		{4623, 43.620694664026104},
		{4624, 44.28128377598449},
		{4625, 45.28965233946238},
		{4626, 45.984997240186246},
		{4627, 45.98395822876192},
		{4628, 46.06556368008116},
		{4629, 45.196081738035325},
		{4630, 44.651596037353286},
		{4631, 45.8360412664365},
		{4632, 45.030975374089174},
		{4633, 45.34523081375098},
		{4634, 45.516897796124226},
		{4635, 45.41290103045353},
		{4636, 45.783086761335824},
		{4637, 45.729375501632695},
		{4638, 46.30543062423142},
		{4639, 46.97672701088541},
		{4640, 46.47744540724059},
		{4641, 46.43460283813155},
		{4642, 46.176535515295534},
		{4643, 46.1267773587671},
		{4644, 46.07078753090914},
		{4645, 45.90463206174146},
		{4646, 45.55249821676164},
		{4647, 46.10452178892155},
		{4648, 46.657238306997954},
		{4649, 46.676446405868745},
		{4650, 46.61931105547808},
		{4651, 46.75145539959066},
		{4652, 47.15614626698322},
		{4653, 47.87910231233235},
		{4654, 47.90315790426083},
		{4655, 48.13748706690984},
		{4656, 48.14012518669324},
		{4657, 48.12478834534791},
		{4658, 47.721710829839914},
		{4659, 47.75882142791673},
		{4660, 48.62175299672685},
		{4661, 49.44973560609625},
		{4662, 49.45655590836775},
		{4663, 49.467384079074314},
		{4664, 49.36632082510971},
		{4665, 49.361539089859946},
		{4666, 48.923792508100405},
		{4667, 48.82936960309324},
		{4668, 48.696827533051106},
		{4669, 49.021992536039356},
		{4670, 48.88792946529902},
		{4671, 49.049698787091685},
		{4672, 49.08423254107231},
		{4673, 48.92829127058939},
		{4674, 48.38289408354073},
		{4675, 48.92579198716452},
		{4676, 48.49132529951036},
		{4677, 48.54115658361094},
		{4678, 48.22070675260601},
		{4679, 47.92838519785814},
		{4680, 47.732252711009075},
		{4681, 45.99113534110142},
		{4682, 44.72899549055614},
		{4683, 44.55687451719093},
		{4684, 44.55628471776198},
		{4685, 44.44930616361875},
		{4686, 44.69514501843518},
		{4687, 44.08997798757562},
		{4688, 44.326776424034165},
		{4689, 44.5246373562509},
		{4690, 44.77024206609575},
		{4691, 44.41279660589304},
		{4692, 44.29090624408688},
		{4693, 43.49452874948875},
		{4694, 43.31906915186023},
		{4695, 42.936317507114055},
		{4696, 42.887727362033345},
		{4697, 42.05490971051252},
		{4698, 42.14072479814179},
		{4699, 40.804558836186786},
		{4700, 41.48316460835661},
		{4701, 41.52072176780903},
		{4702, 41.5348100852975},
		{4703, 41.75692669456157},
		{4704, 41.49643711525131},
		{4705, 42.35954965911466},
		{4706, 42.254443486387316},
		{4707, 42.346428230827634},
		{4708, 42.26152211995296},
		{4709, 42.978169988498536},
		{4710, 42.523208465380144},
		{4711, 42.92136840228886},
		{4712, 42.491877240874906},
		{4713, 42.43808793916936},
		{4714, 42.31386731961688},
		{4715, 42.40503671098323},
		{4716, 41.51956996493458},
		{4717, 41.51666400077679},
		{4718, 41.47426644424032},
		{4719, 41.642970570173034},
		{4720, 41.62187072416654},
		{4721, 41.15180912858265},
		{4722, 41.894760156551904},
		{4723, 42.092594714290364},
		{4724, 42.27635546119988},
		{4725, 42.69832068724076},
		{4726, 42.980504524777594},
		{4727, 45.87658310035213},
		{4728, 43.6183451512828},
		{4729, 43.92197152986545},
		{4730, 43.97403155795249},
		{4731, 43.83776224600831},
		{4732, 44.01998787494774},
		{4733, 43.74137827823208},
		{4734, 43.51842058845427},
		{4735, 43.28245959362667},
		{4736, 42.931398042043796},
		{4737, 42.81785284737095},
		{4738, 43.3507423739577},
		{4739, 42.53626377548142},
		{4740, 43.32318432190919},
		{4741, 44.1194336467924},
		{4742, 44.256900789761175},
		{4743, 44.471678262389915},
		{4744, 44.52826323886761},
		{4745, 44.28324388592612},
		{4746, 44.23885158768107},
		{4747, 44.118008732018836},
		{4748, 44.126076154771745},
		{4749, 43.920419005974225},
		{4750, 43.47296751071649},
		{4751, 43.54310189974708},
		{4752, 43.53066942512155},
		{4753, 44.49097326057114},
		{4754, 44.60107183314088},
		{4755, 42.77090560264186},
		{4756, 42.99334677650617},
		{4757, 42.929199954491246},
		{4758, 42.4663653797701},
		{4759, 42.63510417353381},
		{4760, 41.15252150405892},
		{4761, 41.65062666173321},
		{4762, 41.63569595747878},
		{4763, 41.80167602812174},
		{4764, 41.54305663974481},
		{4765, 41.09851635788159},
		{4766, 40.44563966701578},
		{4767, 40.533880450251715},
		{4768, 40.8690019227734},
		{4769, 40.88937124246862},
		{4770, 41.183809565163905},
		{4771, 41.23702889676266},
		{4772, 41.98448907911279},
		{4773, 41.95032094100293},
		{4774, 41.831147917269575},
		{4775, 41.61384516565436},
		{4776, 41.337855349523466},
		{4777, 41.31691388756026},
		{4778, 41.00824360651651},
		{4779, 40.66303052923502},
		{4780, 40.75233960523218},
		{4781, 40.74691723991323},
		{4782, 41.224220277952135},
		{4783, 41.24176898114373},
		{4784, 41.19227291002064},
		{4785, 41.455618057185966},
		{4786, 41.793314227583664},
		{4787, 41.169085632598424},
		{4788, 41.147687532911064},
		{4789, 41.308688926534245},
		{4790, 41.159786924004884},
		{4791, 41.8142881123724},
		{4792, 41.552546791536315},
		{4793, 41.63707209178867},
		{4794, 41.36580909924643},
		{4795, 40.810470695252555},
		{4796, 40.90861666010221},
		{4797, 40.9449761604474},
		{4798, 41.15181950532048},
		{4799, 41.13789218005914},
		{4800, 41.08509141277718},
		{4801, 41.05365891335565},
		{4802, 40.980911652896054},
		{4803, 41.734203454889105},
		{4804, 41.31916007992194},
		{4805, 41.4215682147432},
		{4806, 41.51414417468774},
		{4807, 42.37111482324612},
		{4808, 42.70943804743709},
		{4809, 42.666757607996296},
		{4810, 42.12147053040958},
		{4811, 42.464100810348334},
		{4812, 42.62139441696377},
		{4813, 41.94637435030276},
		{4814, 41.99684563449916},
		{4815, 42.54816794861336},
		{4816, 42.28223265087694},
		{4817, 42.24550930900591},
		{4818, 42.282491306075286},
		{4819, 42.58423495563741},
		{4820, 41.359861111288104},
		{4821, 42.10481353702761},
		{4822, 42.24941093960088},
		{4823, 41.19941990664588},
		{4824, 41.21364184446058},
		{4825, 41.72506184259513},
		{4826, 42.53601771160175},
		{4827, 42.67420054464686},
		{4828, 42.76426057329836},
		{4829, 42.78798548069922},
		{4830, 42.76148923124109},
		{4831, 43.55500018064631},
		{4832, 43.52818564137662},
		{4833, 43.71030227628026},
		{4834, 43.69559859751488},
		{4835, 42.75390903216747},
		{4836, 42.95452875778498},
		{4837, 43.365066775687346},
		{4838, 43.36814790382736},
		{4839, 43.89333739845817},
		{4840, 43.989916759215},
		{4841, 43.56253598246386},
		{4842, 43.377743454747105},
		{4843, 42.924400615882426},
		{4844, 43.12304670119044},
		{4845, 42.58288520924721},
		{4846, 42.6192440085378},
		{4847, 42.65060540788467},
		{4848, 42.41222623035344},
		{4849, 41.646360968682984},
		{4850, 41.424134917273825},
		{4851, 41.10868413269653},
		{4852, 41.172396017699},
		{4853, 41.36004588804885},
		{4854, 40.800077988280925},
		{4855, 40.86912290176179},
		{4856, 39.88125368753376},
		{4857, 40.413562336579986},
		{4858, 40.30739366800392},
		{4859, 40.4448665834484},
		{4860, 40.67992036846289},
		{4861, 39.19323041879348},
		{4862, 38.934776988261994},
		{4863, 38.6635282389286},
		{4864, 38.018950718551935},
		{4865, 38.04674518139207},
		{4866, 38.01853292656019},
		{4867, 37.71021607139222},
		{4868, 38.422846767406455},
		{4869, 38.450019969893354},
		{4870, 38.3997470792123},
		{4871, 38.34361998177123},
		{4872, 39.091980399463395},
		{4873, 39.5198852558182},
		{4874, 39.5472385581927},
		{4875, 39.52236007766908},
		{4876, 39.37969949452826},
		{4877, 40.222868236832014},
		{4878, 39.66661286220836},
		{4879, 40.283042854444616},
		{4880, 40.229716416585205},
		{4881, 40.445190873458},
		{4882, 41.20372788803875},
		{4883, 41.40919784184403},
		{4884, 41.40205136110032},
		{4885, 41.489596657482274},
		{4886, 41.63320805924114},
		{4887, 41.74501472339082},
		{4888, 41.59463339227347},
		{4889, 41.89165418164628},
		{4890, 42.429906756721266},
		{4891, 42.51566334270849},
		{4892, 42.43112651097444},
		{4893, 42.65573412390896},
		{4894, 42.5649774287204},
		{4895, 42.20578627338075},
		{4896, 42.19905186404266},
		{4897, 42.063060621486514},
		{4898, 40.8631606909062},
		{4899, 41.4015851042295},
		{4900, 40.88944459737443},
		{4901, 41.365721619037295},
		{4902, 40.08190763242786},
		{4903, 40.11091422145917},
		{4904, 40.97087104137605},
		{4905, 41.15805729486192},
		{4906, 41.27896864220977},
		{4907, 41.170130110377485},
		{4908, 41.019095916064586},
		{4909, 41.19823454109512},
		{4910, 40.71222352487721},
		{4911, 40.604096585580145},
		{4912, 40.22428726447587},
		{4913, 38.86084836942058},
		{4914, 38.70041746310922},
		{4915, 38.8194506164692},
		{4916, 38.64467182897344},
		{4917, 38.639870243656844},
		{4918, 38.929014450694986},
		{4919, 38.80087908949201},
		{4920, 38.474735694735124},
		{4921, 38.66300047873025},
		{4922, 38.684455038039566},
		{4923, 38.29265621342109},
		{4924, 38.49017410077791},
		{4925, 38.88748571310341},
		{4926, 39.34561928924596},
		{4927, 39.41078759839496},
		{4928, 39.753432401888055},
		{4929, 39.66725496587569},
		{4930, 39.64503039721447},
		{4931, 39.74434633216348},
		{4932, 39.24422485817575},
		{4933, 39.357176347703124},
		{4934, 39.083868743073175},
		{4935, 38.77958154299146},
		{4936, 38.784739437051876},
		{4937, 38.84416930772779},
		{4938, 38.531983904133995},
		{4939, 38.13839982421345},
		{4940, 38.19834116299139},
		{4941, 38.384878790359856},
		{4942, 37.41788048991505},
		{4943, 38.32825078053794},
		{4944, 37.27571543485687},
		{4945, 37.381643210593964},
		{4946, 37.139400383368596},
		{4947, 37.686425234248986},
		{4948, 37.603432975092},
		{4949, 37.64359097418195},
		{4950, 37.23329232367846},
		{4951, 36.994849462220934},
		{4952, 36.826705097979186},
		{4953, 36.10850429296769},
		{4954, 36.10515278531009},
		{4955, 36.27585866868438},
		{4956, 35.877106950223286},
		{4957, 35.06902417472288},
		{4958, 35.68954496717155},
		{4959, 34.429975016070486},
		{4960, 33.9039081124664},
		{4961, 34.50994734260132},
		{4962, 34.24726508634311},
		{4963, 34.27196566995411},
		{4964, 34.53854603237626},
		{4965, 33.18029169046981},
		{4966, 33.74849740086445},
		{4967, 33.7660814446232},
		{4968, 33.690731394344716},
		{4969, 33.75687045286519},
		{4970, 33.94382441333488},
		{4971, 33.56341790269826},
		{4972, 33.64387610469928},
		{4973, 33.603199106999526},
		{4974, 33.55490176741914},
		{4975, 33.76379562158691},
		{4976, 33.76315401597445},
		{4977, 32.8267558087521},
		{4978, 32.45241276131267},
		{4979, 32.55733463964615},
		{4980, 33.319801723754786},
		{4981, 33.59341126615778},
		{4982, 33.73081923269426},
		{4983, 34.37346764688176},
		{4984, 35.07176548766099},
		{4985, 35.37591959109057},
		{4986, 35.471233571041964},
		{4987, 35.13016122517179},
		{4988, 34.65099719562258},
		{4989, 34.37111074112498},
		{4990, 34.51523331821902},
		{4991, 34.357189444832144},
		{4992, 34.36407537164143},
		{4993, 34.34124435265085},
		{4994, 34.71701325993874},
		{4995, 34.535111124444725},
		{4996, 34.12819607955428},
		{4997, 33.476805201968794},
		{4998, 33.21364571883214},
		{4999, 34.07622604782128},
	},
	{
		{0, 0.1914345437063385},
		{1, -3.4947066313635853},
		{2, -0.9012035232019713},
		{3, -1.2917234135635458},
		{4, -1.6201434723336083},
		{5, -0.11959001409322911},
		{6, 0.06991019914916345},
		{7, 1.8891389677842017},
		{8, 2.871091958695281},
		{9, 1.1787358303375146},
		{10, -2.0768580265100094},
		{11, 3.794369215409862},
		{12, 1.5276510011875892},
		{13, 0.48684884260988703},
		{14, 1.6236249468526378},
		{15, -0.24846588954417326},
		{16, 0.5218634555370912},
		{17, 1.5632739797608988},
		{18, -1.7140974489224434},
		{19, 0.48847898637375486},
		{20, -2.431355287741206},
		{21, -1.3047332002869514},
		{22, -0.056262024497085006},
		{23, 3.5280937799870453},
		{24, 1.431121485137867},
		{25, -1.1605230787409155},
		{26, 1.531155772131144},
		{27, 2.4421145713003467},
		{28, 0.07528067356218182},
		{29, -1.9472656022334065},
		{30, -1.6203621394151857},
		{31, 2.005336574225008},
		{32, 0.6057826871490144},
		{33, 1.8000737806803708},
		{34, 1.6215940425083741},
		{35, -0.19060198061318312},
		{36, -1.544810626264314},
		{37, 1.711336220094734},
		{38, -1.921805906299878},
		{39, -0.813649126888345},
		{40, -0.5761299283902863},
		{41, 1.6860178215600943},
		{42, 0.7043996825596841},
		{43, -0.5391619157688288},
		{44, -0.5671736960385907},
		{45, 2.914019458989815},
		{46, 1.1719608720494954},
		{47, 0.5550525224245844},
		{48, 3.7572954111154684},
		{49, 4.293123827593757},
		{50, 2.1806300710041686},
		{51, 0.7981645356975979},
		{52, -1.003991733674428},
		{53, 1.9534274857036642},
		{54, 0.6263315280488905},
		{55, -1.9357987011915598},
		{56, -0.633094814374466},
		{57, -1.61856108443581},
		{58, 0.8707220998375396},
		{59, 1.645145470229907},
		{60, 1.5670439746129827},
		{61, 0.5437595369770154},
		{62, 0.8861213608718086},
		{63, 0.7213355298992676},
		{64, 0.09447097996431672},
		{65, -0.7176166254326667},
		{66, 2.6574395780199938},
		{67, 3.4291173474222525},
		{68, 5.144375804019528},
		{69, 2.2440085959076197},
		{70, -1.06438249866055},
		{71, 4.11937879576129},
		{72, 1.5090335951755183},
		{73, 2.138744226081619},
		{74, 1.846894200840455},
		{75, -0.12944628869250874},
		{76, 0.2733860100995408},
		{77, 2.102847830315348},
		{78, -0.7678587235388828},
		{79, 1.2446741102733658},
		{80, -6.371245475391389},
		{81, 1.183149399051611},
		{82, -0.516701697245843},
		{83, 0.15004775959177327},
		{84, 2.5336783475235065},
		{85, 1.1620200696574},
		{86, 1.914561665492958},
		{87, 0.47461696831381145},
		{88, 2.932661399927433},
		{89, 2.2125454767105195},
		{90, 1.6437094852415006},
		{91, -2.4354202154950206},
		{92, -1.2434689178429748},
		{93, -1.701123639149033},
		{94, 0.16102472057520067},
		{95, -2.1689933751939043},
		{96, 0.15979474556292605},
		{97, -1.8835506489198282},
		{98, 2.225919900098449},
		{99, 1.056353980509323},
		{100, 0.5392521510736928},
		{101, 0.655161928523947},
		{102, 2.946448123961281},
		{103, 0.946214180388804},
		{104, 0.26627413112851805},
		{105, 0.36499584953208275},
		{106, 0.6599343574550318},
		{107, 4.160685111014811},
		{108, 1.8761842050857926},
		{109, 1.1203217007417934},
		{110, -0.5066597349350801},
		{111, 1.003204568793838},
		{112, -2.061551095492853},
		{113, -0.8499410415193758},
		{114, -2.5392624174178655},
		{115, -0.5635429549350353},
		{116, 1.3247694178929177},
		{117, 2.2777492374110513},
		{118, 1.517502018391243},
		{119, -0.44456001711365756},
		{120, -0.004273024982576673},
		{121, -2.515295713444981},
		{122, -0.6258764114592799},
		{123, -0.6075514102552615},
		{124, -0.07359117698335635},
		{125, -1.1740964103194875},
		{126, 1.9832094964379476},
		{127, 2.3884801154098225},
		{128, 1.0222850420997118},
		{129, -1.8744308023313296},
		{130, -0.5723578266591844},
		{131, -0.20744365417803964},
		{132, 5.312848317264722},
		{133, 0.928171507803693},
		{134, 1.7625607577844846},
		{135, 1.763638664029866},
		{136, 4.1926436928829265},
		{137, 1.6930089662453347},
		{138, 0.45771918687082785},
		{139, -0.9714705695595833},
		{140, -0.6127099727607993},
		{141, 2.949944584452612},
		{142, 2.4543796657493973},
		{143, 0.5529312533570623},
		{144, -1.2496301717683609},
		{145, -0.6925946006027088},
		{146, 2.9679746672265406},
		{147, -0.4599939477837862},
		{148, -1.3123905831873859},
		{149, -1.3114897892129915},
		{150, 0.4033914751854041},
		{151, 4.84605658718049},
		{152, 1.9043024790847238},
		{153, 0.4207874054886426},
		{154, 0.4429669645531573},
		{155, 1.18845208292863},
		{156, 0.6763322354409822},
		{157, 1.719787155275486},
		{158, 0.8959576416381267},
		{159, 0.5323070538962835},
		{160, 2.560762003914965},
		{161, 0.2328218526285023},
		{162, 1.1523234708781018},
		{163, -1.0871082720776921},
		{164, -0.27172483253218027},
		{165, -0.2885680296845306},
		{166, 6.425106427954243},
		{167, 2.810841525229628},
		{168, 0.4063350211557355},
		{169, -1.435789927064955},
		{170, 1.7734577470382753},
		{171, 2.839113025612226},
		{172, 8.759655219079058},
		{173, 3.3928785703278153},
		{174, 1.06013157368414},
		{175, 0.5011142780654699},
		{176, -1.7686693003724094},
		{177, -3.3532892843069764},
		{178, -0.8307467938745446},
		{179, -1.270871884262671},
		{180, -0.18504385765931847},
		{181, 3.0922039749495953},
		{182, 1.9907151401464138},
		{183, -2.7193152405597347},
		{184, 0.3882329488756855},
		{185, -0.6914547887139258},
		{186, -1.7496009432281185},
		{187, 1.9444804134838325},
		{188, -4.686997071783333},
		{189, -2.4049369407912726},
		{190, -0.9718592065669222},
		{191, -1.5371314765168353},
		{192, -1.7618524756531995},
		{193, -1.6492806293264348},
		{194, -2.667812508637397},
		{195, -0.8567156746244795},
		{196, -0.4007033464156529},
		{197, 0.08157678227560089},
		{198, 1.0251162263611484},
		{199, -0.8257654004510804},
		{200, 1.7408007890885706},
		{201, -3.3663441294319494},
		{202, 0.10094704272979182},
		{203, 2.7034427501337075},
		{204, 1.82784412413814},
		{205, 2.05356375566654},
		{206, -2.0617773007294535},
		{207, -1.2958720101234902},
		{208, 1.1256838407796659},
		{209, 1.2639254875866712},
		{210, 1.0366014827928711},
		{211, -2.5649326694095396},
		{212, 2.981750087687558},
		{213, 4.70439233766783},
		{214, -0.7898491429262997},
		{215, -0.3750502694541982},
		{216, -9.717892891169717},
		{217, -2.8788148819778354},
		{218, -1.2033779903071709},
		{219, -0.6987890129709864},
		{220, -0.5961047589163492},
		{221, -0.9122395466919095},
		{222, 0.9398462418765141},
		{223, 1.2939552044374718},
		{224, -3.4556744258749257},
		{225, -1.4512676823165467},
		{226, -0.8311134267761506},
		{227, -2.56510233962969},
		{228, -0.4871319357349335},
		{229, 1.303112430584604},
		{230, -0.6967311237318775},
		{231, -0.844824622182186},
		{232, -0.3164074698442635},
		{233, 0.5111176630532143},
		{234, -2.122635598091093},
		{235, -9.334086089393669},
		{236, -4.266857424975437},
		{237, -4.002731573278378},
		{238, 0.008254170391277516},
		{239, -3.2493841091537368},
		{240, -2.313716918479099},
		{241, -0.15781231041775434},
		{242, -0.29973641052920735},
		{243, 0.9448994631447766},
		{244, -0.2469656028254712},
		{245, 1.0757665552133373},
		{246, -0.23346214178393815},
		{247, -4.509194841032029},
		{248, -3.7095374359186435},
		{249, -2.7818707263686626},
		{250, 7.488717016798557},
		{251, 5.204956831910136},
		{252, 2.3419837429452293},
		{253, -2.722647341011651},
		{254, 0.80109904054008},
		{255, -3.4754987634021584},
		{256, -2.584114141383827},
		{257, -2.4476391373028097},
		{258, -2.9150246761326137},
		{259, -0.8672661309184031},
		{260, -1.1725528418673332},
		{261, -4.364673616828553},
		{262, -0.6442270392415905},
		{263, -1.9259012598705205},
		{264, -1.0157664374137276},
		{265, 2.176054366072152},
		{266, 1.695161561967442},
		{267, 0.853546609529878},
		{268, 0.31990152377531167},
		{269, -1.811605723582938},
		{270, -4.0787193875056},
		{271, -2.8051104789476797},
		{272, 2.189613042429574},
		{273, 0.08457153675293116},
		{274, 7.9307901304401},
		{275, 2.3587286770202147},
		{276, 1.5852118405803106},
		{277, 0.8628347302068253},
		{278, -0.09792511675652688},
		{279, 4.2174743584765295},
		{280, 2.7688877672218712},
		{281, 1.9903117033951752},
		{282, -0.16722364516277788},
		{283, 4.191334813097746},
		{284, 5.653437832039614},
		{285, 2.5807430125944206},
		{286, 0.9498955484494385},
		{287, 2.285538135290795},
		{288, -3.6228768766881823},
		{289, -2.755252159991336},
		{290, -8.842051654058304},
		{291, -3.5248693055681084},
		{292, -0.6233286891270041},
		{293, 5.1837147391217995},
		{294, 1.8145170891614328},
		{295, -1.6011395216872848},
		{296, -3.208875482468396},
		{297, 1.3127027286943918},
		{298, 0.9408887587279604},
		{299, 0.8726523204653636},
		{300, -2.9977629080080694},
		{301, 0.41337859458862414},
		{302, -2.404315930677144},
		{303, 3.365737779565705},
		{304, 0.8355014589979994},
		{305, -1.2940576722230808},
		{306, -3.0844537505736147},
		{307, -4.314730825481584},
		{308, 1.9744452791678582},
		{309, 5.295895251068488},
		{310, 2.6264847886532},
		{311, 0.41846762975042917},
		{312, 0.2161439132481338},
		{313, -3.177606420884945},
		{314, -3.1885750205466357},
		{315, -5.948572689763958},
		{316, -0.025607593745959},
		{317, -0.9247199889121526},
		{318, 1.0679878219895718},
		{319, 0.49264902354690737},
		{320, -1.5477640987304628},
		{321, -0.616920843537844},
		{322, -1.3142669989042493},
		{323, -4.772593862210578},
		{324, -3.5999977028689303},
		{325, 2.473491559452559},
		{326, -2.0979530291173623},
		{327, 0.12107711436982893},
		{328, -0.03888999523767664},
		{329, 1.3928448016975756},
		{330, -1.7880476516853978},
		{331, -4.273769378522248},
		{332, -3.9305938111922156},
		{333, -0.6388761694227411},
		{334, -2.5809339087924816},
		{335, -1.045958670265378},
		{336, -4.0496752959036675},
		{337, -0.8872962081351521},
		{338, -0.001049821261518702},
		{339, 3.0980970769019613},
		{340, 1.2176177751926238},
		{341, 0.09950811716233882},
		{342, -0.5984208938626945},
		{343, -5.480953933976579},
		{344, -2.8294877023867078},
		{345, 0.04896616996286052},
		{346, -7.124188517291678},
		{347, -4.09890717690312},
		{348, -1.0573448701199746},
		{349, -0.35891182383272374},
		{350, 5.068268022329972},
		{351, 1.342847149988731},
		{352, 0.48648418235079777},
		{353, -0.45694466050623916},
		{354, 1.725520510531638},
		{355, -1.4555671865194495},
		{356, 0.8731575438527951},
		{357, -5.739025179756},
		{358, 5.780315712954591},
		{359, 2.0955752464565065},
		{360, 0.8728188902293061},
		{361, -2.9154136130843167},
		{362, -8.722478830520327},
		{363, -3.4288134699912907},
		{364, 1.5881904400889102},
		{365, 3.6885989767493035},
		{366, 1.4257634144770264},
		{367, 0.8203173917777384},
		{368, -6.194417962223472},
		{369, -2.3579947208087444},
		{370, -0.8501973488362331},
		{371, 1.6429124556431134},
		{372, 1.7521097827374783},
		{373, -0.6485459557402405},
		{374, -1.19054257131004},
		{375, -2.5079512145360705},
		{376, -1.6142789812138787},
		{377, 1.493120853714955},
		{378, 0.1502189552245657},
		{379, 1.965753775585022},
		{380, 2.556277377292205},
		{381, 0.9580054901104009},
		{382, 1.2506661650370159},
		{383, 0.4452499039539756},
		{384, -0.7890080994183144},
		{385, 1.1987270812835835},
		{386, 0.25338048454722245},
		{387, 0.185937733253716},
		{388, 0.6545941251512096},
		{389, 0.7709942139441459},
		{390, 6.228216500767825},
		{391, 3.9317896605690783},
		{392, 2.426753023419997},
		{393, 1.0428286555491084},
		{394, 0.5553075440869526},
		{395, 0.7274155268590293},
		{396, -0.23018042258105564},
		{397, 6.764618345199662},
		{398, 2.5217186614753024},
		{399, 1.2638711667293046},
		{400, -0.6629872204824352},
		{401, -1.565903352753156},
		{402, 0.016349396123733206},
		{403, -3.5080026794915415},
		{404, -5.034120440662629},
		{405, -1.6231883850345745},
		{406, 1.7460362889141436},
		{407, 2.38385159893849},
		{408, -1.4077710083562036},
		{409, 2.830508600462809},
		{410, 3.2235384235166733},
		{411, 10.680162966307382},
		{412, 3.1503344124374237},
		{413, 0.44975918496117073},
		{414, 1.9743524162643762},
		{415, 3.8112723944386278},
		{416, 2.5952251826597994},
		{417, 2.782416931607755},
		{418, 1.0859203580782602},
		{419, -0.5950496195834831},
		{420, 0.006446387318461311},
		{421, -3.3371252647158993},
		{422, -1.7738901802447735},
		{423, 2.1646793240115008},
		{424, -2.1737042717456854},
		{425, -0.698900566406741},
		{426, 1.0277893465324768},
		{427, 0.7280176394459692},
		{428, -1.070039249965446},
		{429, 1.3719854196367511},
		{430, 2.082375803270201},
		{431, 1.5949388227783636},
		{432, 2.0450765071096346},
		{433, 1.6277950703453512},
		{434, 1.7885510774656295},
		{435, 0.6543433124067691},
		{436, 0.25526736341238876},
		{437, 3.1626111377950044},
		{438, -0.9420397328235681},
		{439, -3.99214829980081},
		{440, 5.926245253925511},
		{441, 4.002669567003816},
		{442, 1.1463759187271014},
		{443, -0.838323971716292},
		{444, -2.1478453299511395},
		{445, 5.040462206388455},
		{446, -1.1665147598848877},
		{447, -5.767199178697785},
		{448, -3.788076493252116},
		{449, -2.4117641182868783},
		{450, 7.428849070328937},
		{451, 2.8907181411531693},
		{452, -1.474095846971491},
		{453, 0.21638163666260024},
		{454, -0.4180384078231692},
		{455, 1.9754725415064809},
		{456, 0.6171839703004453},
		{457, -0.7068646577807739},
		{458, 3.456474060246713},
		{459, 0.9148891125351392},
		{460, 1.9323983868216643},
		{461, 0.6658973843054514},
		{462, 2.8818483248013265},
		{463, 2.1958098430458244},
		{464, -2.4199517011682765},
		{465, -2.6962072816522826},
		{466, -4.450158760656216},
		{467, -1.714735285612154},
		{468, -0.6797236071338637},
		{469, -0.830031164897902},
		{470, -0.5173091105825354},
		{471, -0.7542165344029724},
		{472, 1.4266739459271878},
		{473, 2.923268271701274},
		{474, 1.324894302880223},
		{475, -1.5236035898006808},
		{476, 1.0670328486306233},
		{477, 0.1640119488555175},
		{478, 0.8069651668455848},
		{479, 3.2637034425480893},
		{480, 1.7758455595232152},
		{481, -6.187334607725115},
		{482, -5.194746201695879},
		{483, -5.2940283763606},
		{484, 5.478690596756662},
		{485, 3.925248837284192},
		{486, 2.3872932263665168},
		{487, 0.7475748353929562},
		{488, 3.3162404840511686},
		{489, 1.0486103412852126},
		{490, 3.105025481182797},
		{491, -2.7163367774294467},
		{492, -0.1874594826483008},
		{493, 1.2886644937661624},
		{494, -0.7265025956842696},
		{495, 0.26901130847247595},
		{496, -0.39318780757090566},
		{497, 2.927962080344786},
		{498, 1.033554306801928},
		{499, -1.4938938794085295},
		{500, -1.1713677003070506},
		{501, 0.4349070647450149},
		{502, -1.6765616896701363},
		{503, 0.9803733627235269},
		{504, 4.219458402551453},
		{505, 4.901846260516893},
		{506, 1.5556333333710586},
		{507, 0.029560801919495172},
		{508, 0.49406448911624856},
		{509, -0.09544421470320144},
		{510, -0.853012774660948},
		{511, -3.087811525386968},
		{512, 4.575936435452966},
		{513, 2.053673612587058},
		{514, -2.3545526667133214},
		{515, -2.172153007701302},
		{516, -2.557612633917088},
		{517, -0.5295302974271412},
		{518, -1.4251486430631748},
		{519, -1.6049620871068024},
		{520, -4.438075881194919},
		{521, -1.8056534625916354},
		{522, -0.7006808889927207},
		{523, -2.7316607023649624},
		{524, -1.518735990180021},
		{525, 1.9164977043143034},
		{526, 3.042743425585678},
		{527, 1.5508371266356353},
		{528, 3.3296781521893197},
		{529, 0.13637620373321746},
		{530, -0.6940547317320769},
		{531, -0.5248128507193897},
		{532, -0.3093799839893248},
		{533, -1.318529177734886},
		{534, 3.0734615938849985},
		{535, 1.364796941037128},
		{536, 2.7208077555716415},
		{537, -1.8539194130521746},
		{538, 5.1208728215766985},
		{539, 0.9243576203951707},
		{540, 0.284722674932578},
		{541, -1.3796325597715227},
		{542, -2.853896057173713},
		{543, -6.64979622922069},
		{544, -0.3512102421745871},
		{545, 0.8374314849987796},
		{546, -1.917736042253964},
		{547, -0.49308614746566204},
		{548, 1.1131183843396806},
		{549, 1.0435926091237036},
		{550, 3.0926028434199377},
		{551, 1.132955076351417},
		{552, 0.8829127295051288},
		{553, 0.8106936892043155},
		{554, -5.973083026326699},
		{555, -2.171808373671146},
		{556, -2.331011404477491},
		{557, -0.2245890413296009},
		{558, 0.63198778475827},
		{559, 0.8107149948523004},
		{560, 2.393561994243787},
		{561, 3.3040616281628425},
		{562, -1.4356639132312219},
		{563, 1.0282839248481226},
		{564, 0.8867489153855117},
		{565, 0.18027075917740956},
		{566, -1.3559177946979581},
		{567, -1.632007271209547},
		{568, -7.58913614620494},
		{569, -2.980336111732328},
		{570, -1.5265333690392953},
		{571, -3.2059265194754683},
		{572, -1.3939910721276518},
		{573, 0.9143774579024243},
		{574, 0.1916928234676105},
		{575, 0.9684076188855067},
		{576, -0.6123610795238493},
		{577, -0.35273748480225775},
		{578, -0.9871026635957046},
		{579, 0.0421448144751686},
		{580, 0.39116219919394424},
		{581, 2.19153593939719},
		{582, 0.7219026215079757},
		{583, -0.04718691700485933},
		{584, -0.37530525168303136},
		{585, -0.8628488026666725},
		{586, -1.4794558546514847},
		{587, -1.6431402742994978},
		{588, -0.7097071169167412},
		{589, -0.3717221921617687},
		{590, -0.45671370862561994},
		{591, -0.20378366272470277},
		{592, 0.5039991666650537},
		{593, -0.1524275550651076},
		{594, -1.3398789140867537},
		{595, 0.25725932553913466},
		{596, 0.1225828189342533},
		{597, -0.22733429241420783},
		{598, -0.6394987034621353},
		{599, 0.7491886221036683},
		{600, -0.2877049554705119},
		{601, 0.686964389352406},
		{602, 0.021674610183780696},
		{603, 3.4812137369614455},
		{604, 0.9133678801852028},
		{605, -1.3596553236078919},
		{606, 1.512956610157548},
		{607, -1.535634141846325},
		{608, -0.5957350233668306},
		{609, -3.5517044925428736},
		{610, -2.3195659383809635},
		{611, 3.1807960743517683},
		{612, 1.028863552784148},
		{613, -4.909337140803901},
		{614, -6.052581382408974},
		{615, -2.554279162283053},
		{616, -2.2865909478751343},
		{617, -3.8844764757928383},
		{618, -1.5574783998158384},
		{619, 0.4218355580654204},
		{620, 0.2663329757698134},
		{621, 4.103454499667303},
		{622, 0.5618766930882393},
		{623, -0.537739264283743},
		{624, 3.146933323699527},
		{625, 2.4807393099261716},
		{626, 0.864865216588693},
		{627, 0.9263274349150967},
		{628, 0.07772825860401789},
		{629, -0.4276678358704391},
		{630, -1.525561603450226},
		{631, -0.5251757334912615},
		{632, -0.7679406482513123},
		{633, 2.007707004838056},
		{634, -1.5585645659426035},
		{635, -1.2361847257102874},
		{636, -1.1858958590318345},
		{637, -0.3446618741535768},
		{638, -0.5462017780057382},
		{639, 4.580715771548457},
		{640, 2.2089411940437698},
		{641, 1.0352047317927364},
		{642, 4.268552882862046},
		{643, 1.3158532660360782},
		{644, 0.8852957574672027},
		{645, 2.9428971707789415},
		{646, 1.1962164604248657},
		{647, 0.37345231165532083},
		{648, 0.27032862775926464},
		{649, 0.10814361644854498},
		{650, 0.10138263090037708},
		{651, -0.39588850668702125},
		{652, -2.802767410612327},
		{653, -1.1123863257386444},
		{654, -2.308729372856542},
		{655, -1.0798971868310354},
		{656, 1.3129170692721122},
		{657, 0.7393410913588886},
		{658, 3.6651012236017277},
		{659, -0.051046478928732375},
		{660, 2.8657924708893456},
		{661, 0.9657261671286878},
		{662, 0.8572019488426964},
		{663, 0.07713828925886501},
		{664, -5.059821883249842},
		{665, -3.371332360125992},
		{666, -3.1808007129214086},
		{667, -0.8975641323453148},
		{668, -2.561358651494711},
		{669, -1.8198764983528042},
		{670, 0.6349517458649239},
		{671, 2.140792860360975},
		{672, 0.7359313548164224},
		{673, 0.36249264596929187},
		{674, -0.4021198521615083},
		{675, 0.420916167625928},
		{676, 0.09153426309042893},
		{677, -0.563528020369867},
		{678, 0.8827229485218783},
		{679, 0.07186545692953644},
		{680, -1.6926523140631595},
		{681, -0.7396376183150042},
		{682, -0.4715180811285632},
		{683, -2.6151217281663484},
		{684, -2.2120850540825483},
		{685, -0.19942840662563655},
		{686, 1.90472165474662},
		{687, 0.44291319984030836},
		{688, 0.40762463907671365},
		{689, -2.8007048351079185},
		{690, 4.606816710736399},
		{691, 1.3875603305034827},
		{692, -0.005844796177935652},
		{693, 0.11127712449794935},
		{694, 0.17430005383126396},
		{695, 5.251324165463329},
		{696, -0.5406960567928363},
		{697, -0.6105622090905771},
		{698, -0.7321600860623754},
		{699, 0.45527923350066796},
		{700, 2.699908559423271},
		{701, 0.5974310629535398},
		{702, 1.2389816784116179},
		{703, -0.47064007556310244},
		{704, 0.2740311551135818},
		{705, -0.08193568618507148},
		{706, -0.09741612095404859},
		{707, -0.03557582417880731},
		{708, -0.22059369824149594},
		{709, 0.025378669969027393},
		{710, 0.08669341465848773},
		{711, -0.29349687312360845},
		{712, -0.5352404751080106},
		{713, 0.30602684279033987},
		{714, 0.9517109390674441},
		{715, 1.2838993200785203},
		{716, -3.84532715538551},
		{717, -1.597232117189238},
		{718, -0.0673026220482491},
		{719, 0.3062652010317667},
		{720, -0.23477845189737118},
		{721, -1.2860389808116095},
		{722, 0.6471225663234876},
		{723, 3.685362922185199},
		{724, 1.8780105558443014},
		{725, 1.725255206628551},
		{726, -1.496492475135371},
		{727, -3.0904418171845056},
		{728, -0.6031342827894787},
		{729, 0.9980242464281925},
		{730, 5.484394280593881},
		{731, 5.263024938222562},
		{732, 2.3158808594437943},
		{733, 1.0375823628037384},
		{734, 0.9080197201361612},
		{735, -0.3266292000867431},
		{736, 0.3740649888018903},
		{737, -0.8951118242983469},
		{738, -0.008694146006163195},
		{739, 1.5760419482452819},
		{740, 1.6568965566033338},
		{741, 1.362080339041464},
		{742, 0.1172494560102651},
		{743, -0.5243469856547921},
		{744, -1.0764533421297946},
		{745, -1.3953609508651772},
		{746, 1.2575520284032842},
		{747, -0.5688454976597641},
		{748, -1.236726597050751},
		{749, 0.9549921667672163},
		{750, -0.03411113433626417},
		{751, 0.18843797456841602},
		{752, 3.5325442097447035},
		{753, -2.060639543075947},
		{754, -0.4818234343441015},
		{755, -0.0029557205143259935},
		{756, 0.7604652561796137},
		{757, 2.6385528245113754},
		{758, 1.9805984940536887},
		{759, 1.3219218985121803},
		{760, 1.308735979682801},
		{761, 1.04913607487119},
		{762, -0.006535345252570435},
		{763, 0.65790927603002},
		{764, 0.09816155346956429},
		{765, -2.93994394976615},
		{766, -4.397035860157518},
		{767, -2.297548449235193},
		{768, -2.3772375486805193},
		{769, -0.29404482476376814},
		{770, -1.0672470463841006},
		{771, 0.29494817946107393},
		{772, -0.17785317897980307},
		{773, 0.15573117695629046},
		{774, 1.6944586741518857},
		{775, 0.4500577511307101},
		{776, 3.361302702898243},
		{777, 1.7476223131749937},
		{778, 4.164177219197379},
		{779, 0.2641500670144885},
		{780, 0.6310495900166455},
		{781, -2.2416041466491694},
		{782, -0.21141364966509935},
		{783, -0.17861656087987615},
		{784, -4.551560622611826},
		{785, -0.8503502147562148},
		{786, -1.9840941270085284},
		{787, 0.8350622231372448},
		{788, 0.3959723263828643},
		{789, -1.8773496150203761},
		{790, -1.6253236173123664},
		{791, -0.667292338618356},
		{792, -0.8527503425168907},
		{793, 0.9426302610657384},
		{794, -3.052536953812949},
		{795, -1.9091022132088074},
		{796, 2.1586801848042554},
		{797, 2.634988213645425},
		{798, -3.150935083329458},
		{799, -1.6618302945595709},
		{800, 1.0115293199254833},
		{801, 0.04425817580322494},
		{802, 0.5040229585908809},
		{803, 0.30116949581622493},
		{804, 0.12731949042708884},
		{805, -3.487769943836523},
		{806, -2.5165290241220335},
		{807, -0.449692179231186},
		{808, -5.700310245011747},
		{809, 0.9530533042230185},
		{810, 2.343457197097518},
		{811, 4.208818120160133},
		{812, 2.526627704880281},
		{813, -0.2335243848280566},
		{814, 4.6373849251015455},
		{815, 1.463990397326367},
		{816, -0.09795617128605538},
		{817, -3.04684056114075},
		{818, -1.4839354307809165},
		{819, 2.3328273786128877},
		{820, 1.2503747371457512},
		{821, 1.0080764940154037},
		{822, 4.619127136660354},
		{823, 3.2956410102433313},
		{824, 2.352743934533067},
		{825, -0.11644128315691316},
		{826, 0.004654656225646182},
		{827, 1.4300675115990378},
		{828, -3.078938149973093},
		{829, -2.814316956416535},
		{830, 0.2558259480845799},
		{831, -0.0036211937165492175},
		{832, 0.836167482875796},
		{833, 2.909919750926119},
		{834, 2.2598763230700794},
		{835, 2.011491869840956},
		{836, 1.6165952150832996},
		{837, 0.7662737298383531},
		{838, 4.4914156318969605},
		{839, 5.273887014210655},
		{840, 1.9942327616203093},
		{841, 0.8304087766440931},
		{842, 0.8465742196317071},
		{843, 1.9986742892499467},
		{844, -0.09293240554373905},
		{845, -0.7223956916271372},
		{846, 0.032929861852148856},
		{847, 1.2094575991518906},
		{848, 0.48134226000043706},
		{849, -2.055849696262347},
		{850, 2.7343299903530864},
		{851, 1.2017529158632534},
		{852, -0.6509522570277488},
		{853, 0.8336117543710233},
		{854, 1.447031461678498},
		{855, 0.6772064632632366},
		{856, 2.184096582295235},
		{857, 0.8684547494066969},
		{858, -0.8376040188930207},
		{859, -0.9220382239599365},
		{860, -1.30510461044829},
		{861, 4.449606805276277},
		{862, 2.1997107515660437},
		{863, 1.8467097574217237},
		{864, 1.3490892876368203},
		{865, -0.10081947852141526},
		{866, 4.143247293571973},
		{867, -0.832124123639743},
		{868, -0.1432821519023044},
		{869, -0.23585170541286354},
		{870, -0.9305991564033012},
		{871, -0.5322563258809004},
		{872, -1.1509011959654973},
		{873, -1.4945513697647927},
		{874, -5.594617805429758},
		{875, -4.488060950178165},
		{876, 4.955508988475826},
		{877, 0.7392227737605825},
		{878, 0.08920194151455543},
		{879, -4.337423062090759},
		{880, -2.26041002007487},
		{881, -7.820650280623007},
		{882, -0.5484904590064499},
		{883, 1.161876885865979},
		{884, -2.2706184638744693},
		{885, -1.8636722909093641},
		{886, -1.3741159192694186},
		{887, 1.4924958324217754},
		{888, 4.2172500741526715},
		{889, 2.728950596185547},
		{890, 1.3004787616573679},
		{891, 0.3742087117039523},
		{892, 0.8318856779947059},
		{893, 0.5199511341064373},
		{894, -0.09086336078394952},
		{895, 2.50696915687815},
		{896, 1.444123642942994},
		{897, 0.5341605087531349},
		{898, 0.9382727728752811},
		{899, 0.6698783132824384},
		{900, 0.6868974203111965},
		{901, 3.463542789497685},
		{902, 1.4147582495079924},
		{903, 0.80714593278944},
		{904, 0.5114969122276751},
		{905, 0.22741231880237425},
		{906, -0.8889489229090706},
		{907, -4.611429213511713},
		{908, 0.160806082932204},
		{909, 2.6321586135076167},
		{910, -2.011491226038337},
		{911, -2.1622537111007087},
		{912, 0.7656718785015024},
		{913, 1.1002585768178086},
		{914, 0.8283339978126004},
		{915, 0.3330853816871364},
		{916, 0.07843656716092111},
		{917, 0.8754550082260723},
		{918, 0.5754249350887833},
		{919, 2.5262511207313536},
		{920, 0.9040900948879211},
		{921, -4.06927762558898},
		{922, 1.2383971730580643},
		{923, 3.7287420205652744},
		{924, -1.6343959270627226},
		{925, 4.7595239935470435},
		{926, 1.8234504073525968},
		{927, -0.17477633124742264},
		{928, 1.100164502138179},
		{929, -1.796265655903103},
		{930, -3.550669008999186},
		{931, -0.3427821548910792},
		{932, -0.8086726732116141},
		{933, -5.456704909724856},
		{934, 0.6135004380473457},
		{935, 0.1263456959949772},
		{936, 3.581162950258055},
		{937, -0.8266432641855588},
		{938, 2.829509797008712},
		{939, 0.7917926915385811},
		{940, 1.967022332141343},
		{941, 7.649741059579023},
		{942, 9.32635957623812},
		{943, 2.851186443552971},
		{944, 3.7624134757008303},
		{945, 0.1625127823862813},
		{946, -0.267785231175632},
		{947, 1.25177695167891},
		{948, -3.0075485449729764},
		{949, -0.6410323025242585},
		{950, 0.9892760314378537},
		{951, 0.3751200755510685},
		{952, -0.8514310777827894},
		{953, -3.6462259490834543},
		{954, -1.0293697477160744},
		{955, -2.2042046790038605},
		{956, -0.4154195725291746},
		{957, -0.008812675140503018},
		{958, 1.2509971550815455},
		{959, 0.09937023163718761},
		{960, 0.14018984476612104},
		{961, 0.7064849455368751},
		{962, 1.850491191425549},
		{963, -0.37583938673560124},
		{964, -0.13641066683325123},
		{965, -1.7390494173320847},
		{966, -0.9378941760958647},
		{967, 0.9846224877247118},
		{968, -3.0614148328826043},
		{969, -1.1784509628642545},
		{970, -0.8944315830340808},
		{971, -0.4688990977377501},
		{972, -1.538063287491706},
		{973, -0.31638693182749844},
		{974, 0.07447213507753242},
		{975, -0.4452362718533946},
		{976, 0.9943926377718899},
		{977, -1.2705281910620296},
		{978, -0.9520413472692737},
		{979, -0.16459694990741974},
		{980, -4.387148133666137},
		{981, -0.12480357776919737},
		{982, -0.3652699919855605},
		{983, 2.6403437073868403},
		{984, -1.6439994814716699},
		{985, -1.2229204760930563},
		{986, 1.6437110625477154},
		{987, 0.9741703557133164},
		{988, -5.461880122106444},
		{989, -7.631367894446474},
		{990, -5.947892290735805},
		{991, -2.509428205307425},
		{992, -1.5815940063612142},
		{993, -0.09541106835633817},
		{994, 0.13547818356703675},
		{995, -1.6842493662941616},
		{996, -0.6324785739109252},
		{997, -0.9797264066846805},
		{998, -5.353432863380168},
		{999, -4.942345069680021},
		{1000, -1.9767032477414341},
		{1001, -0.12173340867975857},
		{1002, -1.8989777876514837},
		{1003, -2.1889228924625894},
		{1004, 0.5018846756127787},
		{1005, -1.0474753082525514},
		{1006, 1.013410534742523},
		{1007, -3.1981081522821677},
		{1008, -1.9728349545267032},
		{1009, -1.503900935213173},
		{1010, -0.11505664994574721},
		{1011, 2.712615743858978},
		{1012, 7.202777962756665},
		{1013, 3.6759016554672783},
		{1014, 6.892831759764684},
		{1015, 1.7545575103581537},
		{1016, 0.6635217333449616},
		{1017, 1.1360561980958919},
		{1018, 4.019354120741262},
		{1019, -1.0679048912566123},
		{1020, -0.21971038617821625},
		{1021, -4.17853108551789},
		{1022, -1.2712540464131559},
		{1023, 0.1560021224324546},
		{1024, 0.7915711838674928},
		{1025, 3.425445933271888},
		{1026, 4.7573798051141365},
		{1027, -0.0220388095263524},
		{1028, -2.5052040533672137},
		{1029, 0.21298859294202122},
		{1030, -1.7854912051136465},
		{1031, -3.3288265301758604},
		{1032, -4.465125942489238},
		{1033, -1.2467103510020325},
		{1034, 0.38735431470353615},
		{1035, 0.16659504040877535},
		{1036, 3.6308213651629315},
		{1037, -1.744593729602403},
		{1038, -3.02025119607125},
		{1039, -0.7905559998484648},
		{1040, 1.232349753899285},
		{1041, 2.2964233470166517},
		{1042, 1.403072029697316},
		{1043, -1.5192045610379377},
		{1044, 0.4114782598614126},
		{1045, 1.4164686387539098},
		{1046, 1.7613414873072861},
		{1047, 0.2541207461157947},
		{1048, -1.1220241495291858},
		{1049, -0.37487161979387107},
		{1050, -1.493168299403439},
		{1051, -0.45583010898505405},
		{1052, -1.858561776380648},
		{1053, 1.522464819832893},
		{1054, 2.4540788628251713},
		{1055, 1.6372138134886458},
		{1056, -0.1545691119435818},
		{1057, -0.44927807672786335},
		{1058, -0.6916965100022838},
		{1059, -3.1294713452722114},
		{1060, -1.5695471823983764},
		{1061, -1.2242795987907498},
		{1062, 1.9924985837201024},
		{1063, 4.179807688764242},
		{1064, -2.744878735812537},
		{1065, -4.018176225952731},
		{1066, -1.8731783609587007},
		{1067, -0.925641143708177},
		{1068, -0.2508275011513015},
		{1069, 0.9372807324083351},
		{1070, 0.37737852401911615},
		{1071, 2.5451647486609623},
		{1072, 2.5383636856535166},
		{1073, -1.276173850066572},
		{1074, -0.6499246632086646},
		{1075, -3.5004264934042704},
		{1076, -0.7691664017569197},
		{1077, -2.3764523675276585},
		{1078, -1.091421058999182},
		{1079, 1.3033996837869193},
		{1080, -0.689712042925962},
		{1081, 0.09766257817028567},
		{1082, 1.8031356601652129},
		{1083, 0.5078555033706204},
		{1084, -0.159712734317946},
		{1085, -0.9846037125075181},
		{1086, -1.580786583842342},
		{1087, -0.0746674245985971},
		{1088, 3.3708413612975026},
		{1089, 1.3532495352153515},
		{1090, -1.564191223139765},
		{1091, -0.5308588040759545},
		{1092, -0.5282134927854645},
		{1093, -1.1588588206890484},
		{1094, 0.913966567770696},
		{1095, 0.36319742841928093},
		{1096, -0.17091525711921793},
		{1097, 0.37181353425579794},
		{1098, 6.6157138361338035},
		{1099, 5.4260939351888275},
		{1100, 3.9291122403980037},
		{1101, 2.304573303281604},
		{1102, 1.169259735963037},
		{1103, 0.06676533144926844},
		{1104, 0.7885500895650779},
		{1105, 1.193404762116266},
		{1106, -0.891819909412454},
		{1107, -4.807673638549589},
		{1108, 2.2024395949442632},
		{1109, 2.5774193231133045},
		{1110, 2.392600002368141},
		{1111, -3.2757918800353742},
		{1112, -1.130894435673593},
		{1113, -0.2140526636075635},
		{1114, -0.0703302341800644},
		{1115, 3.208084523407074},
		{1116, 0.8401756045637071},
		{1117, 0.328289561712496},
		{1118, 3.435519902892176},
		{1119, 0.4715924123423577},
		{1120, -2.937201816972818},
		{1121, -8.840127063755203},
		{1122, -3.0475205154096443},
		{1123, -7.958275039564295},
		{1124, -4.590745138952405},
		{1125, -1.6226812218138473},
		{1126, -1.342164794844372},
		{1127, 2.9465125787639943},
		{1128, 3.8319234906634128},
		{1129, 1.1838323760964398},
		{1130, 0.6973278350730985},
		{1131, 1.5531861282355153},
		{1132, 1.3626308027509335},
		{1133, 0.5752023252692363},
		{1134, 1.2023680067165063},
		{1135, 2.918649102983747},
		{1136, 1.0761418203808395},
		{1137, 1.0700378273716287},
		{1138, 0.6056407062212603},
		{1139, -0.11560287389589938},
		{1140, 4.250569510286743},
		{1141, 3.4181100136448785},
		{1142, 1.7424591897044317},
		{1143, 2.2430498578624025},
		{1144, -3.756099892949023},
		{1145, -1.8874201295707569},
		{1146, 4.060576084471169},
		{1147, 1.5234789106671527},
		{1148, -0.2919670436878542},
		{1149, -0.7564562392043133},
		{1150, 0.5889854598137041},
		{1151, 3.6116549735770294},
		{1152, 4.132270098028595},
		{1153, 0.795669610562663},
		{1154, 0.5565354949602446},
		{1155, 0.5659554731354154},
		{1156, 0.23485377889604814},
		{1157, 1.042240934084846},
		{1158, 5.182362108862529},
		{1159, 4.33649442668553},
		{1160, 0.7380303189703217},
		{1161, -1.7630159735261146},
		{1162, -6.326983492467595},
		{1163, -5.7186839161664835},
		{1164, 0.13641811715775898},
		{1165, -1.7030012712983946},
		{1166, -3.5355595256293704},
		{1167, -1.1417643303314091},
		{1168, -0.5194495446741861},
		{1169, -0.1413138190770573},
		{1170, -0.18536172890429603},
		{1171, -2.1387189400389652},
		{1172, 3.248418448720487},
		{1173, -0.6037596933211551},
		{1174, -0.8847028822138476},
		{1175, -0.4009736625550283},
		{1176, -0.5235606425892978},
		{1177, -4.735635544354674},
		{1178, -1.5968810418405757},
		{1179, -2.4206981264633343},
		{1180, 1.6853284305568277},
		{1181, -0.14959471412671943},
		{1182, -0.2268572716983429},
		{1183, 8.009430712501162},
		{1184, 4.226693442585662},
		{1185, -2.7433060385802985},
		{1186, -3.510303678699679},
		{1187, -0.17182221493036187},
		{1188, 0.15748915273786873},
		{1189, 3.8143673488118215},
		{1190, 1.9503660015828477},
		{1191, 0.890433988471184},
		{1192, 1.9130763695252966},
		{1193, -1.833027876528427},
		{1194, -2.742044645454723},
		{1195, -0.47488004309141507},
		{1196, -4.035884771532241},
		{1197, 0.42429221706635944},
		{1198, 2.2359243767424712},
		{1199, 0.549362076041766},
		{1200, 0.24440770273570517},
		{1201, -0.3360615035146291},
		{1202, -0.2778907745872328},
		{1203, 0.9138009123933042},
		{1204, -0.1479654415183143},
		{1205, 1.7136413165491882},
		{1206, 2.934289013152436},
		{1207, 1.0731940800896742},
		{1208, 0.13560518058483445},
		{1209, 2.38358758420854},
		{1210, 2.734428529452748},
		{1211, -4.041789356108857},
		{1212, -4.539042049881521},
		{1213, -2.9841112565601926},
		{1214, -1.0670613855314817},
		{1215, -1.894942046183513},
		{1216, 0.6081106279793247},
		{1217, 1.159388018989162},
		{1218, 0.9016236593863172},
		{1219, 0.3592332555514793},
		{1220, -1.728486314938787},
		{1221, -1.368522486308104},
		{1222, 0.9866238913431316},
		{1223, 0.2666651798390664},
		{1224, -0.7935790749200822},
		{1225, 4.707864233341095},
		{1226, 1.4591719461798989},
		{1227, 0.6586627066978925},
		{1228, 0.5819364093323061},
		{1229, 0.1710374431762603},
		{1230, -1.3492726338089613},
		{1231, -1.8389200384707585},
		{1232, -2.907575053060525},
		{1233, 5.088232847839355},
		{1234, 4.027463664053041},
		{1235, -2.9194808724671137},
		{1236, 2.7228062682717695},
		{1237, 0.15542207064329416},
		{1238, -3.2317099411923533},
		{1239, 0.16843764866992372},
		{1240, 0.23781910393282835},
		{1241, 1.5391542298775456},
		{1242, 0.6860091654713558},
		{1243, -1.931761537517844},
		{1244, -0.7698550164539355},
		{1245, -1.6013411768899104},
		{1246, 3.0360238348438857},
		{1247, -1.209484556649047},
		{1248, 3.025370588462953},
		{1249, -2.1932739777537082},
		{1250, -2.6783230286925908},
		{1251, -2.5489684149076473},
		{1252, -1.0902658237730243},
		{1253, -1.9709134454445885},
		{1254, 0.8712455768683711},
		{1255, -2.352879289131102},
		{1256, -1.149460127625121},
		{1257, 3.164888250331585},
		{1258, 1.6850995737230814},
		{1259, 0.5145586393616726},
		{1260, 1.239414133200514},
		{1261, 2.1556600143258144},
		{1262, 0.8408219812514216},
		{1263, 0.8530818538777675},
		{1264, 0.4298743245939972},
		{1265, 0.013969632999970671},
		{1266, 0.4664384990167074},
		{1267, -1.7298944328109824},
		{1268, 1.8191530530787374},
		{1269, 0.5973059928173138},
		{1270, 1.5718912216089098},
		{1271, 1.0187481705349997},
		{1272, 0.8106703277783249},
		{1273, 2.1583928654362308},
		{1274, -1.883645599867616},
		{1275, 0.5169472957691834},
		{1276, -0.17051801876269596},
		{1277, -0.8890570536695241},
		{1278, -1.6344792969011104},
		{1279, -4.470560590150113},
		{1280, -1.6584456743993545},
		{1281, 1.3905372668120353},
		{1282, 1.5268333009639767},
		{1283, 0.5763501816661231},
		{1284, 1.1751574007221077},
		{1285, 3.5731366973359573},
		{1286, 0.96214985911665},
		{1287, 1.4553118505762621},
		{1288, -1.6257309118425205},
		{1289, -0.6098605928816531},
		{1290, 0.49168833052301386},
		{1291, -1.820871878693475},
		{1292, -0.5807867351904447},
		{1293, -3.328443942055976},
		{1294, 2.1960960535054572},
		{1295, 0.7442515033343204},
		{1296, 1.0864364897347742},
		{1297, -3.454261078002848},
		{1298, 0.7332993293882608},
		{1299, 0.06965036865486188},
		{1300, -0.08089269639186826},
		{1301, 0.5603074816091945},
		{1302, -2.382621484278481},
		{1303, -2.020300144585899},
		{1304, 0.7771656239482074},
		{1305, -0.27183435119523836},
		{1306, 1.2129845782199662},
		{1307, 0.1197848852179424},
		{1308, -0.8434772984250948},
		{1309, 1.3303078231056003},
		{1310, -4.646142168470022},
		{1311, -1.8600333947505843},
		{1312, 1.0264896014978127},
		{1313, -2.522645750470158},
		{1314, 3.735572937790156},
		{1315, 0.7139323849882611},
		{1316, 1.466282643684217},
		{1317, -0.5223469479060654},
		{1318, -0.30482216161330633},
		{1319, -0.5330693731101168},
		{1320, 0.14509226123376906},
		{1321, -0.8402377505523949},
		{1322, -0.49335548299061094},
		{1323, 2.2071384112188834},
		{1324, 0.39444322373240415},
		{1325, 0.2571271030304063},
		{1326, 0.3628442760659169},
		{1327, 1.110637158316231},
		{1328, -3.3713200538414765},
		{1329, -1.5285040994721997},
		{1330, -1.039644700563335},
		{1331, -0.13693802583341802},
		{1332, 0.24012723097354446},
		{1333, 0.8510226907903072},
		{1334, 0.3018028871080348},
		{1335, -0.28913989977705223},
		{1336, -2.8484680086981276},
		{1337, 0.5659934602757057},
		{1338, 3.87317444923262},
		{1339, 1.5514423587825519},
		{1340, -2.1190859701862377},
		{1341, 2.9986633245404573},
		{1342, 5.761367019276657},
		{1343, 2.5005399620250897},
		{1344, 9.684705700423464},
		{1345, 6.294505614598388},
		{1346, 3.1911568425324655},
		{1347, 2.595472568358627},
		{1348, -0.4137307176595506},
		{1349, 2.3177206538332493},
		{1350, 1.512782522872891},
		{1351, -6.100936437444041},
		{1352, -2.4829092142916465},
		{1353, -0.8470704266288768},
		{1354, -1.3565074574089562},
		{1355, -0.10721479675572276},
		{1356, -0.3493225339452013},
		{1357, 0.8707227253411013},
		{1358, 0.37719939228559607},
		{1359, 0.7100766029640788},
		{1360, 0.08630717107864053},
		{1361, -0.0607856486584007},
		{1362, 0.14459160845136923},
		{1363, -2.6674080488634755},
		{1364, -4.38112630781154},
		{1365, -4.735694488678014},
		{1366, -3.3722319665970133},
		{1367, -6.106114766786172},
		{1368, 2.1376644264930906},
		{1369, 1.336356044453791},
		{1370, 0.7354617257688426},
		{1371, -0.05846108438666758},
		{1372, -0.02635698914501041},
		{1373, 2.406016748273669},
		{1374, 1.1378651340140518},
		{1375, 0.7875136139278269},
		{1376, -1.560405669422732},
		{1377, -0.19412509223968694},
		{1378, -0.13763366593352688},
		{1379, -0.41090174774459265},
		{1380, -0.5128217389659675},
		{1381, -0.48411275366343093},
		{1382, -0.3006514512260015},
		{1383, -3.0052959236866217},
		{1384, -0.6522613166824651},
		{1385, -1.2417553208879637},
		{1386, -1.5980893266129066},
		{1387, 1.6326005659741591},
		{1388, -1.2298354901656758},
		{1389, -1.3872238759747595},
		{1390, -1.277317964923307},
		{1391, 1.1851692799565265},
		{1392, 0.5218809238864076},
		{1393, 1.4954416611266375},
		{1394, 0.5373669339877764},
		{1395, -2.391972542216768},
		{1396, -0.5517288816237749},
		{1397, -0.16629955632665802},
		{1398, -4.2632387399198635},
		{1399, -4.635100691231299},
		{1400, -7.110001736114018},
		{1401, -2.1245097650238245},
		{1402, -0.40197256663046255},
		{1403, -0.21992779683008618},
		{1404, -4.859866723650084},
		{1405, -0.7746397920852106},
		{1406, -2.6824408508699005},
		{1407, -0.8521537373508848},
		{1408, -0.8530214328853423},
		{1409, 2.327362071521189},
		{1410, 4.3693438730486305},
		{1411, 2.029888413390829},
		{1412, -0.05201064009262335},
		{1413, -0.7906025200523498},
		{1414, -0.05627283619163653},
		{1415, -1.9418234843816597},
		{1416, -2.6677090791172806},
		{1417, -1.1799672998632535},
		{1418, -1.4736701548596058},
		{1419, 1.8392104626082608},
		{1420, 0.797178900262403},
		{1421, 3.837307350311048},
		{1422, 1.5263117843414418},
		{1423, 1.9652461584508931},
		{1424, 0.961231420212767},
		{1425, 1.4684512916784902},
		{1426, -0.8221798818322279},
		{1427, -1.1171873145543691},
		{1428, -0.5010538038622496},
		{1429, 0.8370108308261766},
		{1430, -2.3915541878247084},
		{1431, -3.7058750275822367},
		{1432, 3.809018095100254},
		{1433, -2.0999398889399155},
		{1434, -1.5038867105894775},
		{1435, -3.281960511874818},
		{1436, 4.5411702489644465},
		{1437, 3.137020974742362},
		{1438, 0.29597097648032844},
		{1439, 0.48580317325584255},
		{1440, 1.0277484201700107},
		{1441, 5.920296997542591},
		{1442, 1.7553805982706767},
		{1443, -0.2762937727400562},
		{1444, -0.4779774025131348},
		{1445, -6.856485641014716},
		{1446, -3.1038900953497603},
		{1447, -1.2508745023996033},
		{1448, 3.6345606635115475},
		{1449, 2.3429317086950627},
		{1450, 4.149518625547424},
		{1451, 0.8190939868855394},
		{1452, 7.370984629851606},
		{1453, 2.8797942108407812},
		{1454, 1.057625031975485},
		{1455, 2.823711774081786},
		{1456, -0.6611660956146441},
		{1457, -0.33521690008483085},
		{1458, -2.8317895022922004},
		{1459, 2.220909487893608},
		{1460, -1.3177631121947575},
		{1461, -1.8487462405591268},
		{1462, -0.09740058409466923},
		{1463, -6.2316802622909115},
		{1464, -4.351006990946448},
		{1465, -0.2852885567718342},
		{1466, -1.908371558847051},
		{1467, -1.7333891968418333},
		{1468, 1.5242302186846874},
		{1469, -3.190048294533906},
		{1470, 0.7370423674214097},
		{1471, 2.161838485877223},
		{1472, 1.073784698393274},
		{1473, 8.456899590062646},
		{1474, 4.602963300213601},
		{1475, 1.9007872069281564},
		{1476, 0.4924108136780714},
		{1477, -2.908330837836086},
		{1478, 0.08013486167399386},
		{1479, 0.3728648282595119},
		{1480, -0.4414854059521493},
		{1481, -2.8637255283901566},
		{1482, -2.8407418171283947},
		{1483, -0.37961028023359344},
		{1484, 5.09928534187917},
		{1485, 1.3412692856240354},
		{1486, 2.0472903023214117},
		{1487, 0.4446945395737501},
		{1488, 0.6001913106734689},
		{1489, -2.101043204413802},
		{1490, 0.6456120982210549},
		{1491, 0.7985919601583051},
		{1492, 2.3067028906894826},
		{1493, 1.061753540305504},
		{1494, 0.5198520982767226},
		{1495, 1.0801608428343479},
		{1496, 0.4353328751701283},
		{1497, -0.9452293088332749},
		{1498, 0.07316547423422132},
		{1499, -1.441592087673726},
		{1500, -0.7918021468252867},
		{1501, -0.19516044750878903},
		{1502, 3.1122859739494517},
		{1503, 4.553142438333481},
		{1504, 0.20705701395977982},
		{1505, -8.360493536168361},
		{1506, -3.6163548850088016},
		{1507, -5.086508516603473},
		{1508, -3.8163509100358555},
		{1509, -3.2505799548248193},
		{1510, -2.977247144147009},
		{1511, 3.905530154303458},
		{1512, 2.754797438253135},
		{1513, 1.183823123841138},
		{1514, -1.3152941369792872},
		{1515, 0.09876048458507225},
		{1516, 1.7552290398039303},
		{1517, 0.17750449157994352},
		{1518, -1.5010138919299845},
		{1519, -0.7259438627927103},
		{1520, -0.5453547004260415},
		{1521, 0.4499383209974993},
		{1522, -2.780894877749338},
		{1523, -0.7412784615120118},
		{1524, -1.2434634185770994},
		{1525, -0.8184716117097173},
		{1526, -0.28145796183726807},
		{1527, 0.9048689138321924},
		{1528, 0.8098004110171007},
		{1529, 0.34789785366220066},
		{1530, 0.2179884993047545},
		{1531, -1.7920567091359185},
		{1532, -0.003213539380083952},
		{1533, 0.2239419165604665},
		{1534, 1.7927268849659557},
		{1535, -0.9166733909167643},
		{1536, -0.35871729402484287},
		{1537, 4.080018394073955},
		{1538, -0.374792910265352},
		{1539, 3.2961011764587185},
		{1540, 0.07952187786264542},
		{1541, -6.222166659122913},
		{1542, -5.447502952676054},
		{1543, -1.6522366853305663},
		{1544, -0.0484909497947541},
		{1545, -0.01978831600980841},
		{1546, -0.008459012833394093},
		{1547, -1.5598198817998117},
		{1548, 6.478665268389666},
		{1549, 2.920998083962441},
		{1550, 0.39247381225111777},
		{1551, 0.7131014352547447},
		{1552, 0.4822696457567224},
		{1553, 0.31337781919658353},
		{1554, 0.17460257782245217},
		{1555, 3.9861961918290865},
		{1556, 4.280030644159919},
		{1557, 1.5915113077236331},
		{1558, 0.7977654150091268},
		{1559, 0.8787956993854376},
		{1560, 0.7574926911030345},
		{1561, 0.0807751016148261},
		{1562, 0.5639956263216438},
		{1563, 5.917155238044591},
		{1564, 4.025280856480598},
		{1565, 3.177256637404212},
		{1566, 1.4368996047355975},
		{1567, 4.1091520516503826},
		{1568, 3.8027768213970123},
		{1569, -1.1080618294953268},
		{1570, 3.734874618993904},
		{1571, 0.7798845305977561},
		{1572, 1.657354001861104},
		{1573, 0.4472991372666527},
		{1574, -0.26687716846245546},
		{1575, 6.696426303724373},
		{1576, 4.609355871026603},
		{1577, 0.29985905795131496},
		{1578, -0.487541530141561},
		{1579, 0.5261064414614154},
		{1580, 0.3045794598318048},
		{1581, 1.8436637656583943},
		{1582, 1.972252443889294},
		{1583, -0.4584776637570749},
		{1584, -0.256665445392951},
		{1585, 0.3079822649759647},
		{1586, 0.275229824442014},
		{1587, 4.2552259467915965},
		{1588, 1.6753866129854698},
		{1589, 2.3470872265727105},
		{1590, 0.9550614002826752},
		{1591, 5.57820691420845},
		{1592, 6.967459707438015},
		{1593, 3.9456537877685607},
		{1594, 0.5481719151110764},
		{1595, 0.22327871581005201},
		{1596, 0.3537527984067912},
		{1597, -1.551168778687852},
		{1598, 0.050036094027675726},
		{1599, -0.7904583095875892},
		{1600, -0.711190300797069},
		{1601, -0.40063982850452073},
		{1602, -3.0895299904589666},
		{1603, -3.7895029523264907},
		{1604, -2.8950027912232548},
		{1605, -1.1810596041013601},
		{1606, 1.361870570561297},
		{1607, 2.0402209535292686},
		{1608, 0.3647696227413807},
		{1609, 2.3502488376338677},
		{1610, 1.1686748133697509},
		{1611, 3.045246870152562},
		{1612, 5.891967345464777},
		{1613, -0.0898926111524534},
		{1614, -0.7417274443593768},
		{1615, -0.4441786669809267},
		{1616, -2.2526193475832286},
		{1617, 1.9618096398271123},
		{1618, 6.455698311540212},
		{1619, 4.704815978582291},
		{1620, -1.7951023059742703},
		{1621, -2.0687205616873876},
		{1622, -1.1836988807125999},
		{1623, -0.021717983642627614},
		{1624, 0.19879682219653433},
		{1625, 0.7880379904121191},
		{1626, -1.2969384708023008},
		{1627, -0.9299132054109676},
		{1628, -8.904882111178363},
		{1629, -0.23925097008503293},
		{1630, -0.9619039510091345},
		{1631, 1.5152807979525158},
		{1632, 1.2408265691277374},
		{1633, 0.2126201485591661},
		{1634, -0.015097384314178522},
		{1635, 3.462104377282962},
		{1636, 2.429598918288754},
		{1637, -0.3309915832173491},
		{1638, -1.1350375930343553},
		{1639, -0.29212232050937426},
		{1640, 2.7399648781647867},
		{1641, 0.5400642049066471},
		{1642, -0.9874320474895332},
		{1643, 0.022949446539447416},
		{1644, 1.9629807142283215},
		{1645, -4.57058358724791},
		{1646, -1.8437763357746826},
		{1647, 1.5757765299758155},
		{1648, 0.6557612209014597},
		{1649, 1.645675812773225},
		{1650, -0.26379861735208154},
		{1651, 0.5888908168417802},
		{1652, 0.69638977380988},
		{1653, -0.9541013188421699},
		{1654, -3.6789239794544857},
		{1655, -6.098330022219628},
		{1656, -2.1055549439213728},
		{1657, 1.3903060713588298},
		{1658, -0.2826125368179414},
		{1659, -0.15731078444657381},
		{1660, 0.31589191764032876},
		{1661, -1.3105755631257054},
		{1662, -2.3517786944753345},
		{1663, -3.76963645254161},
		{1664, -1.51815290646032},
		{1665, -0.16870201698547654},
		{1666, 7.011838225463239},
		{1667, 2.694939897557991},
		{1668, 1.341287087009806},
		{1669, -5.497899317545717},
		{1670, -2.2556705552561573},
		{1671, 4.998478903448072},
		{1672, 3.7162316220288627},
		{1673, 1.3299949839141096},
		{1674, -0.42407219366132853},
		{1675, -2.090388813550331},
		{1676, -1.5555903875870327},
		{1677, 0.5235885782552866},
		{1678, 0.4715632223202757},
		{1679, 2.2042694466880417},
		{1680, 1.5371226166768661},
		{1681, 0.5861429454626592},
		{1682, 3.010336383542044},
		{1683, 1.2819509088584535},
		{1684, 0.9702766986335079},
		{1685, 2.8793332543644534},
		{1686, -2.2502195041812323},
		{1687, -5.801788229804142},
		{1688, -3.57235353592964},
		{1689, -5.476977195341316},
		{1690, -1.9284229941607667},
		{1691, 0.8019051706936664},
		{1692, 3.607736113541032},
		{1693, -0.26580636954414416},
		{1694, 0.17807955747368387},
		{1695, 4.090092603696913},
		{1696, 1.991240033399493},
		{1697, -0.02029776869873612},
		{1698, -0.2102366831101744},
		{1699, 0.4785700480151526},
		{1700, 1.2215594979288416},
		{1701, -0.8136060516223772},
		{1702, -1.218270543585871},
		{1703, -0.7425257242496361},
		{1704, -0.5787885551825825},
		{1705, -1.7486509700269153},
		{1706, -2.445384710689969},
		{1707, -3.381106009587705},
		{1708, -2.2467460880091195},
		{1709, 0.5783655995127321},
		{1710, 0.6367913691240249},
		{1711, 0.17938075958109273},
		{1712, 2.9623567751496584},
		{1713, 0.07409938623653578},
		{1714, 4.230276519738167},
		{1715, 1.1629333825666488},
		{1716, 1.1026378455932202},
		{1717, 0.97976589599163},
		{1718, 2.215504081171289},
		{1719, 1.6266022408912777},
		{1720, -7.302905224148934},
		{1721, -0.3594226648662415},
		{1722, -0.1918601899212488},
		{1723, 1.8676244558775439},
		{1724, 1.1165704642302714},
		{1725, -4.374255244922507},
		{1726, 3.7711656707375605},
		{1727, 1.595760727218944},
		{1728, -0.5192181446946155},
		{1729, -5.883320496617901},
		{1730, -2.686299602215407},
		{1731, -1.4123061710138118},
		{1732, -0.30414005398382443},
		{1733, -0.05513997605290202},
		{1734, -0.35417512673400336},
		{1735, 2.0089210923677387},
		{1736, 1.7156117099380412},
		{1737, -0.30958795682794216},
		{1738, -1.6800057619558209},
		{1739, -0.7673558578551819},
		{1740, 1.189747988831154},
		{1741, 3.4299543800612113},
		{1742, 2.3564094546807044},
		{1743, -0.934026536404255},
		{1744, -0.5858126836094147},
		{1745, 1.0227216197241338},
		{1746, 0.44685014092908026},
		{1747, -0.18644369560540477},
		{1748, 0.2660382272489602},
		{1749, 0.10728676158944059},
		{1750, 0.492055295445045},
		{1751, -2.1533966782181113},
		{1752, 0.2941882782459959},
		{1753, 2.5476702578864487},
		{1754, -0.13308171344716646},
		{1755, 0.06127734854472514},
		{1756, 0.6235011334771682},
		{1757, -4.1848714467696455},
		{1758, -1.7733657895403252},
		{1759, -0.5886761632640042},
		{1760, -0.2686983222729117},
		{1761, -0.11185465120909478},
		{1762, -1.3482172736717872},
		{1763, -0.14703622215299406},
		{1764, -0.8943359894816696},
		{1765, -0.6182383203627311},
		{1766, -0.3533698291199343},
		{1767, 5.531781510064832},
		{1768, 3.7621896032254663},
		{1769, 3.499278713263857},
		{1770, 1.5083511138991512},
		{1771, 2.7011213230664346},
		{1772, 1.7988513343563242},
		{1773, -1.017414590380442},
		{1774, -0.5132377647731874},
		{1775, 0.0554345523358622},
		{1776, -0.20553782385350033},
		{1777, -0.05873301280634831},
		{1778, -6.664700586932413},
		{1779, -1.0338404986157304},
		{1780, -1.3761092659264877},
		{1781, -0.6181464956303887},
		{1782, -0.29823566420822967},
		{1783, -0.3859394503305176},
		{1784, 0.16366422972298045},
		{1785, -2.8522926713556256},
		{1786, 0.04872241374958097},
		{1787, 1.3213731139462377},
		{1788, -6.1730361074815505},
		{1789, 5.654036711284216},
		{1790, 1.917428290372935},
		{1791, -1.3643323331016168},
		{1792, -5.463554146256742},
		{1793, -2.2960746587783514},
		{1794, -3.4588912771661686},
		{1795, 0.10734282982334875},
		{1796, -5.500960653141919},
		{1797, -3.41777398184099},
		{1798, -0.9108499879826484},
		{1799, -0.7810785860774376},
		{1800, -1.226472192188076},
		{1801, -4.092145362158016},
		{1802, -3.0567263847848176},
		{1803, -0.4367294194812975},
		{1804, -3.319740373299292},
		{1805, -0.822739711492482},
		{1806, -0.4145174351035504},
		{1807, -0.45494195921524827},
		{1808, -1.2641232276635632},
		{1809, 3.5033674263560526},
		{1810, 6.454182337371713},
		{1811, 5.381388733082188},
		{1812, 2.178132904590227},
		{1813, 0.675292713174831},
		{1814, 2.746943358458573},
		{1815, 0.3245676763820312},
		{1816, 0.3351406396200248},
		{1817, 1.587179838053857},
		{1818, 0.4574668064765296},
		{1819, 0.18906501768529987},
		{1820, 1.7118767821703427},
		{1821, -2.0982888183019295},
		{1822, 2.7159195503679614},
		{1823, 2.728109168204603},
		{1824, 0.6885653346253366},
		{1825, -1.6879912896859455},
		{1826, -0.27877692735099624},
		{1827, 0.2849073001683168},
		{1828, 0.3513267734765556},
		{1829, 0.03576722832283072},
		{1830, 0.10527151954181542},
		{1831, 0.5713038888185831},
		{1832, 0.1416259082507797},
		{1833, -0.4379988968488149},
		{1834, 0.5717761137797193},
		{1835, -0.9799711385807263},
		{1836, -4.095487077485156},
		{1837, -0.27567697529723567},
		{1838, -0.07052095148848053},
		{1839, -1.1249025910127481},
		{1840, -1.2492450116745994},
		{1841, -1.2821695698039939},
		{1842, -0.5192478685252283},
		{1843, 1.6032924292231747},
		{1844, 1.3938262705133813},
		{1845, -0.6598374904874906},
		{1846, -3.340862878202045},
		{1847, 0.5519122035614421},
		{1848, 0.4836453117525924},
		{1849, 1.4210962604748907},
		{1850, -2.450517661946151},
		{1851, -2.2642849984241447},
		{1852, -3.7218457213383047},
		{1853, -1.4173063685683605},
		{1854, 0.6966987044664344},
		{1855, -0.9346296279211808},
		{1856, -1.1130301537041474},
		{1857, -0.5403513985379464},
		{1858, 0.08368553981079299},
		{1859, 1.866412033058049},
		{1860, -0.10537254504864124},
		{1861, 1.1339489544841446},
		{1862, 1.0312511993791251},
		{1863, 2.698127519368784},
		{1864, 0.6979942946108773},
		{1865, -0.27955873520341995},
		{1866, -0.363598619137794},
		{1867, -4.79119196203075},
		{1868, -5.345286648526772},
		{1869, 1.3111841631394259},
		{1870, 0.2832868353021444},
		{1871, 1.51025351756583},
		{1872, 3.5214008170225584},
		{1873, 5.2663944180689635},
		{1874, 2.809673220275741},
		{1875, -0.42369822220483777},
		{1876, 1.3424411103237495},
		{1877, 0.8266013694003702},
		{1878, 1.8152391550678448},
		{1879, 2.7591200622794685},
		{1880, 3.9093590447370286},
		{1881, 0.4443859505898753},
		{1882, 2.486034974297612},
		{1883, 0.3927499001266622},
		{1884, -1.5761386921875646},
		{1885, -0.7315790090623888},
		{1886, -2.498252217835974},
		{1887, 2.2935881748889098},
		{1888, 1.3948274721171572},
		{1889, 0.5776029629047527},
		{1890, 2.05874790655203},
		{1891, 0.21547291695393334},
		{1892, 0.3570892252061603},
		{1893, -0.06221124604187836},
		{1894, -0.28695199835947693},
		{1895, -3.6655000207832926},
		{1896, -1.443128252350602},
		{1897, -4.036729695296774},
		{1898, -2.9266876725034097},
		{1899, -0.9161748398881531},
		{1900, -2.4159875823112102},
		{1901, -2.7722390417695664},
		{1902, -1.0375503690564416},
		{1903, 1.6027073840978998},
		{1904, 0.7478354855930643},
		{1905, -1.7297855273793372},
		{1906, -0.6848840983376177},
		{1907, -2.924163208957069},
		{1908, 1.3607393079649786},
		{1909, -0.8919494182677227},
		{1910, -1.3076172817648364},
		{1911, -3.8407823206610634},
		{1912, -2.340422072926102},
		{1913, -1.339777135789347},
		{1914, 5.190323965776324},
		{1915, 3.951602340859465},
		{1916, 2.339262890199535},
		{1917, -0.05034180310035474},
		{1918, -2.598854412102168},
		{1919, -3.210795426469667},
		{1920, -2.973491245343324},
		{1921, -0.436210845078317},
		{1922, -1.1412986327152534},
		{1923, -0.5830164500182778},
		{1924, 2.236815724255573},
		{1925, 2.4117770112942756},
		{1926, 7.893751764872615},
		{1927, 3.9538742307477523},
		{1928, 1.2277604209029704},
		{1929, 0.5348649351522733},
		{1930, 1.1960282720597442},
		{1931, 3.257698359306139},
		{1932, 1.5631434988728465},
		{1933, 2.61280291536003},
		{1934, 2.464489221766904},
		{1935, 7.7235146153450005},
		{1936, 3.5020959063935564},
		{1937, 0.4204895480712394},
		{1938, 1.7733638193078474},
		{1939, 1.5264540607012727},
		{1940, -1.0143072183751354},
		{1941, 1.9548644279425025},
		{1942, 1.8135315946268689},
		{1943, 0.8801563798462086},
		{1944, 0.36872405263617686},
		{1945, 0.5016793178234329},
		{1946, 0.7468234511571644},
		{1947, 0.5516803110059425},
		{1948, 0.07491316231767287},
		{1949, -0.5409644873153676},
		{1950, -1.7273827828612331},
		{1951, -1.6958517820034587},
		{1952, 2.571557711100016},
		{1953, 1.3591644108928387},
		{1954, 0.32452005848002774},
		{1955, 13.40662057015061},
		{1956, 5.064044656851605},
		{1957, 4.215324399881627},
		{1958, 1.2548156348825534},
		{1959, -8.017997857116901},
		{1960, -3.780352899861857},
		{1961, 4.789536353210147},
		{1962, 1.4141701561194262},
		{1963, 1.7637975267425614},
		{1964, 0.808647653868915},
		{1965, -0.09298159653100563},
		{1966, -0.5445829189050986},
		{1967, 0.23999408484862914},
		{1968, 1.7045406514144452},
		{1969, 1.2864251334403063},
		{1970, 0.23595190619905676},
		{1971, -0.16998876795567106},
		{1972, 1.7981568631939713},
		{1973, 1.8402637436984306},
		{1974, 2.6168820867379785},
		{1975, 6.73795131386144},
		{1976, 3.398955576940164},
		{1977, 1.7407524153032157},
		{1978, -2.5630717555428246},
		{1979, 0.660308187748007},
		{1980, 0.30092149715451044},
		{1981, -0.7377904638704325},
		{1982, -0.4144148915320455},
		{1983, -2.4780535764313516},
		{1984, 5.536130807875712},
		{1985, 5.09966644211762},
		{1986, 0.9464671036370131},
		{1987, 1.9575286178533995},
		{1988, 1.1310466859304777},
		{1989, 0.02589367021637473},
		{1990, -0.9086752110902208},
		{1991, 6.969753043303118},
		{1992, -1.6851724296789334},
		{1993, -0.4231415231762114},
		{1994, -1.2060445700385123},
		{1995, -0.37599475673370863},
		{1996, -1.2005041061953134},
		{1997, 0.9359792763308203},
		{1998, 3.814280824564994},
		{1999, 3.4028864793221762},
		{2000, 2.2368008825766763},
		{2001, -2.533837269958161},
		{2002, 2.601570929197199},
		{2003, 1.372697742611765},
		{2004, 0.2234123281785954},
		{2005, 0.053303021300518665},
		{2006, 0.5102248323551459},
		{2007, 3.405770213252337},
		{2008, -1.6454959618658944},
		{2009, -2.9017617186657794},
		{2010, -1.1506474683494534},
		{2011, -0.5038505334071172},
		{2012, -2.531754532884621},
		{2013, 0.6187727501350015},
		{2014, 3.230395660218953},
		{2015, 2.7897360420286645},
		{2016, 0.23496354641592465},
		{2017, 0.16282966736534202},
		{2018, 0.30860878050946144},
		{2019, -0.46454732681558286},
		{2020, -2.728371930600952},
		{2021, 0.5087594042836696},
		{2022, -0.010963135375945454},
		{2023, -2.913162840700314},
		{2024, -1.233686706823515},
		{2025, -0.5152772711141118},
		{2026, -3.203633517483002},
		{2027, -1.8983852343665344},
		{2028, -5.524058829492587},
		{2029, -4.705757340294058},
		{2030, -1.3297666958282468},
		{2031, 3.5970308674432268},
		{2032, 3.470529636068519},
		{2033, -0.8006520799753918},
		{2034, -0.7478885215868666},
		{2035, -0.30246616489287625},
		{2036, -4.385557820435351},
		{2037, -5.182673225696429},
		{2038, -2.5028392649812443},
		{2039, -0.07689337867143708},
		{2040, 0.20104667827499453},
		{2041, -3.001047690782968},
		{2042, -1.800585680382203},
		{2043, -0.6700467668164836},
		{2044, 0.8797761987955407},
		{2045, 0.026598277948519322},
		{2046, 0.3610147682670878},
		{2047, 1.4659878318511521},
		{2048, 3.4734739021424965},
		{2049, 4.693594753662188},
		{2050, -0.7525126804436022},
		{2051, -0.1429340377414655},
		{2052, 2.940047733961252},
		{2053, 5.0674653506347465},
		{2054, 0.24255686666427612},
		{2055, -1.5988537813122332},
		{2056, -0.7997497008282674},
		{2057, -0.9139025149408466},
		{2058, 1.2250856141956228},
		{2059, -0.24338384541109415},
		{2060, -0.23166139739579494},
		{2061, -1.5378508428662983},
		{2062, 0.7511011733338366},
		{2063, 0.526182883894954},
		{2064, -5.371571964442196},
		{2065, -4.2175722813378815},
		{2066, -1.6523754029380404},
		{2067, -3.6940486746408006},
		{2068, -0.30149189732720005},
		{2069, 1.2058825059025726},
		{2070, -0.7602483139899091},
		{2071, -0.10166380310160811},
		{2072, 4.112541419941205},
		{2073, 2.197615817282448},
		{2074, -2.3587072952102273},
		{2075, -0.8272849649817339},
		{2076, -0.2871762539388914},
		{2077, 2.0277219392900343},
		{2078, 0.8517925867471035},
		{2079, -0.2776758159192565},
		{2080, -1.9017133270310038},
		{2081, 1.8249482322672748},
		{2082, 0.9162178275463287},
		{2083, 0.053926892862837306},
		{2084, -1.0512064552218914},
		{2085, 2.5579517946085044},
		{2086, -0.9648914800588124},
		{2087, 0.7219826648981014},
		{2088, 1.535869027264538},
		{2089, 2.209968518628158},
		{2090, -2.1721533733380314},
		{2091, -1.582290135630036},
		{2092, 1.4022921082486717},
		{2093, -0.4788176165809881},
		{2094, 0.5922236438784418},
		{2095, 3.888944994601128},
		{2096, -1.9816758338851284},
		{2097, 2.5524673950076937},
		{2098, 0.19032303101568515},
		{2099, -0.6412555332080425},
		{2100, 0.9109249629918326},
		{2101, -1.1596685136530216},
		{2102, -1.1846318840123957},
		{2103, -3.4354858303370235},
		{2104, -1.5313381785089397},
		{2105, -4.513270443930894},
		{2106, 0.21785258344410607},
		{2107, 1.8908859069066841},
		{2108, 2.141178601733353},
		{2109, 3.56381262832591},
		{2110, 0.7926152152452789},
		{2111, -3.9413865126885717},
		{2112, -1.8828585088703222},
		{2113, -1.1559719335043117},
		{2114, 2.035820058914016},
		{2115, -0.30538879264536667},
		{2116, 3.3461052698966496},
		{2117, 0.5453745326281086},
		{2118, -2.0429441057700872},
		{2119, -0.733733159190301},
		{2120, 0.9068829996499148},
		{2121, 1.0430877419032083},
		{2122, 0.07042114536705968},
		{2123, 3.900917065678452},
		{2124, -1.7506270739621594},
		{2125, -0.11306064019178019},
		{2126, 1.9224873852248525},
		{2127, -2.31067702035073},
		{2128, -5.347880277981635},
		{2129, -2.7363104280811568},
		{2130, -1.181325014438742},
		{2131, 0.3460922898708438},
		{2132, 6.765345605877256},
		{2133, 2.8081278267589904},
		{2134, 2.291913108258975},
		{2135, -2.8567328742110556},
		{2136, -7.158972847862644},
		{2137, -1.5218091632094781},
		{2138, -1.4035913084542262},
		{2139, 5.079335565009442},
		{2140, 1.8894578159562119},
		{2141, 2.6281142000702724},
		{2142, 4.725952453225383},
		{2143, 3.3315939310586},
		{2144, 4.569311265839343},
		{2145, -1.008982272724144},
		{2146, -0.3094387214527915},
		{2147, -0.8528546562253151},
		{2148, -2.2486694607560467},
		{2149, -0.9374191311610379},
		{2150, 0.0682997767384656},
		{2151, 1.8364128440630842},
		{2152, 0.7250485587430949},
		{2153, -0.4509406746639649},
		{2154, -1.6656619165813646},
		{2155, 4.701661299922804},
		{2156, 1.725239856722534},
		{2157, -1.4827673539709745},
		{2158, 2.578991269057628},
		{2159, -0.9203885151506095},
		{2160, -2.1448729385397565},
		{2161, -0.8445950659191573},
		{2162, 0.6877527694898713},
		{2163, 4.881046790210102},
		{2164, -1.9694950903748805},
		{2165, -0.6801293201444104},
		{2166, 4.820397809628112},
		{2167, 5.454348683359417},
		{2168, 2.3559492014961485},
		{2169, 7.083227447672722},
		{2170, 3.2940670304647277},
		{2171, 0.5756532383255607},
		{2172, -6.001861298809395},
		{2173, 3.662303892088325},
		{2174, 1.562854568278409},
		{2175, 1.010595955126512},
		{2176, -3.424834550052116},
		{2177, -4.391118031989927},
		{2178, -4.480333827510096},
		{2179, -2.4748219554005937},
		{2180, -1.1641728858766203},
		{2181, 1.4182385617401523},
		{2182, 0.2598298049369104},
		{2183, 2.3878881487049313},
		{2184, 1.5882613747040253},
		{2185, 1.292313811048738},
		{2186, 1.3382494650426755},
		{2187, -0.31493922513920913},
		{2188, -1.2703682775007337},
		{2189, 0.16745729915088925},
		{2190, -0.5538534604592188},
		{2191, 2.7272040595816227},
		{2192, 2.8731812196824933},
		{2193, 1.2165560744841413},
		{2194, 0.41035729865131626},
		{2195, 2.147259676421221},
		{2196, 1.275899654078085},
		{2197, 8.013359256287037},
		{2198, 2.517809027245013},
		{2199, 2.8821075257154205},
		{2200, 1.8353697271417064},
		{2201, 2.058767106843514},
		{2202, -0.5818238004761849},
		{2203, -0.7854485514076848},
		{2204, 0.2446762635500409},
		{2205, 0.20360971065496444},
		{2206, -2.5141070905219083},
		{2207, -1.2399137721719091},
		{2208, 0.7295117610398828},
		{2209, 2.544442798251788},
		{2210, 3.6765981304373727},
		{2211, 1.8575196326843737},
		{2212, -0.61895896515988},
		{2213, -0.4784736503009458},
		{2214, 4.892172164934781},
		{2215, 4.046978517067679},
		{2216, 1.75190114974783},
		{2217, -2.7699050530316},
		{2218, -2.8790895075121004},
		{2219, -2.8888732909592783},
		{2220, -0.29056142670269536},
		{2221, 3.519856760525575},
		{2222, 0.40248648861216707},
		{2223, -0.4208394031921321},
		{2224, 0.2637138917993353},
		{2225, -3.85405040131762},
		{2226, -4.7612786216268885},
		{2227, -4.812868673936585},
		{2228, -2.700000689104999},
		{2229, 3.6668602427948267},
		{2230, 1.4872445356599924},
		{2231, 1.483680774648931},
		{2232, 0.43687266061373176},
		{2233, 1.7222876305475356},
		{2234, 3.392638647603307},
		{2235, 0.9712328977019},
		{2236, 2.4719815269750693},
		{2237, 0.7395008419626804},
		{2238, 0.4536244132396775},
		{2239, 0.20852741013854126},
		{2240, -0.7039084318734519},
		{2241, 1.0693899739672617},
		{2242, 0.4633646798319709},
		{2243, -0.25925453814005883},
		{2244, 1.3218747595984262},
		{2245, 0.5232862684783017},
		{2246, -0.48101217420693854},
		{2247, -0.1580205076636517},
		{2248, -1.4265255353679762},
		{2249, -1.2448135622026442},
		{2250, -1.0067365302296185},
		{2251, 2.0910080916752043},
		{2252, -2.7778873661537657},
		{2253, -0.009306325166254847},
		{2254, 0.5544717559451805},
		{2255, -3.196029777753724},
		{2256, 0.5509946778670918},
		{2257, 1.6584338583543548},
		{2258, 0.10174000446340159},
		{2259, -0.07306772268374886},
		{2260, 0.22481124847342135},
		{2261, 0.3192183543075873},
		{2262, -0.9380196204289794},
		{2263, -1.545382194685365},
		{2264, -0.18456344077390785},
		{2265, -2.630827555172676},
		{2266, -5.30829446836303},
		{2267, -1.8593905507615662},
		{2268, -0.6868909458645763},
		{2269, -0.7424490813510054},
		{2270, 0.16618445516520375},
		{2271, 0.6439790873872773},
		{2272, 2.406981184222743},
		{2273, 0.42522256742851483},
		{2274, 3.8059283428756947},
		{2275, 1.5558485391864694},
		{2276, -2.9441148129280004},
		{2277, -3.76697299468376},
		{2278, -1.7572308797460063},
		{2279, -1.3547730857166251},
		{2280, -0.4322706557549671},
		{2281, -4.514902307613175},
		{2282, 0.8170221994994225},
		{2283, 1.519459463243293},
		{2284, -2.0516281628868622},
		{2285, -3.2778607038182046},
		{2286, 9.380269842924411},
		{2287, 0.8208196518085042},
		{2288, -1.3383710614437179},
		{2289, -1.6549207897208662},
		{2290, -0.6689780526481571},
		{2291, -0.3373529414358622},
		{2292, -1.2839495741767135},
		{2293, -1.2035930102523464},
		{2294, -0.505097179019797},
		{2295, -0.25730162611428004},
		{2296, 1.53361864196592},
		{2297, -0.6156261764276424},
		{2298, 1.8719190124342617},
		{2299, 0.5092350731326171},
		{2300, 1.7097949220881752},
		{2301, 0.6064762442462821},
		{2302, 0.25482631231420155},
		{2303, -3.9540567228048142},
		{2304, -1.045385795114054},
		{2305, -0.8369893396351336},
		{2306, -0.3172214892523457},
		{2307, 2.5747604916540796},
		{2308, 4.776239681250443},
		{2309, 1.539887380695463},
		{2310, -1.0144360720869146},
		{2311, -2.364609607121339},
		{2312, -1.636459645507577},
		{2313, -0.6553903602441644},
		{2314, -3.199295351129731},
		{2315, 3.2839436740562187},
		{2316, 2.2077169508799868},
		{2317, 0.9479276877254036},
		{2318, -1.93579570090396},
		{2319, -4.081403959529948},
		{2320, -6.53430857383653},
		{2321, -2.358017446181436},
		{2322, 2.368681397730067},
		{2323, 2.187774845783552},
		{2324, 0.6492435977478925},
		{2325, 2.9970989970499806},
		{2326, 1.285548305490472},
		{2327, 7.498616630878812},
		{2328, 1.285305288115088},
		{2329, -0.959914985200094},
		{2330, -3.0369623716964207},
		{2331, -2.092263380055835},
		{2332, -0.5826025145790976},
		{2333, 0.7060632825669202},
		{2334, -1.976766338490524},
		{2335, 6.183174327475902},
		{2336, 2.467911218855738},
		{2337, 1.1493599611201304},
		{2338, -1.833326068108682},
		{2339, -3.4193260851893594},
		{2340, -9.90288231308415},
		{2341, -4.632311720157663},
		{2342, -8.456405887605541},
		{2343, -3.04837385086767},
		{2344, -6.3534748749040375},
		{2345, -2.508770502586404},
		{2346, -1.2108866839127876},
		{2347, -2.324380350873942},
		{2348, -2.3891308657963632},
		{2349, -1.6045933420792766},
		{2350, 1.9274169991379195},
		{2351, 0.9882205196217104},
		{2352, 2.4818559721915725},
		{2353, -0.25822336408482516},
		{2354, 0.3871376405500784},
		{2355, -6.935371057349963},
		{2356, -3.0606131547243627},
		{2357, -3.1423089276609293},
		{2358, -4.4525305471334296},
		{2359, -2.6549302172879434},
		{2360, 1.003618459931597},
		{2361, 1.752717735051477},
		{2362, 1.948928627459105},
		{2363, 0.9331198978702255},
		{2364, -1.8593460110866544},
		{2365, -1.5840759854428956},
		{2366, -0.8345982092001061},
		{2367, 2.4506034485851673},
		{2368, 1.827358341220378},
		{2369, 2.1354476936649975},
		{2370, 11.167118584430359},
		{2371, 2.6556564672947203},
		{2372, 1.5116797668674429},
		{2373, 2.146936170440836},
		{2374, -1.8886478527979849},
		{2375, -0.6244179842963303},
		{2376, 0.9545697710159557},
		{2377, -0.7967060758082687},
		{2378, 0.31351157395672863},
		{2379, -0.9077894542342008},
		{2380, -0.9442634892287118},
		{2381, -6.123530886119469},
		{2382, -2.998684071931041},
		{2383, -1.1128586725186216},
		{2384, 2.346791263302618},
		{2385, 6.519126857201609},
		{2386, -0.7513315383326851},
		{2387, 0.5609916185396582},
		{2388, 5.871005890699055},
		{2389, 3.317319950771305},
		{2390, 6.924506712140568},
		{2391, 2.4616764242847013},
		{2392, 0.014864713079566827},
		{2393, 1.1600764055157788},
		{2394, 0.6126162930676882},
		{2395, 3.31680123848806},
		{2396, 1.3539507295098638},
		{2397, 1.1316995947339983},
		{2398, -0.5197218743835353},
		{2399, -0.17664324224761538},
		{2400, -0.34737194585942927},
		{2401, -0.4030027326989405},
		{2402, -0.11016169318278628},
		{2403, 0.25760128211934175},
		{2404, -0.8619465356136948},
		{2405, -1.0623273682922427},
		{2406, -0.06372624073730665},
		{2407, -1.2762452565222377},
		{2408, 0.07483281325225144},
		{2409, 0.015489318543038616},
		{2410, -0.47960624617290654},
		{2411, -1.190036261916962},
		{2412, 2.613050313128902},
		{2413, -0.7801029162937076},
		{2414, -1.9152449191615075},
		{2415, -1.0807747793483218},
		{2416, -0.09820882581060247},
		{2417, -1.3029275675693608},
		{2418, -0.8720637124499631},
		{2419, 1.9923899617172278},
		{2420, 1.2331625625856524},
		{2421, -0.29252569796470773},
		{2422, 1.6096985097904903},
		{2423, 0.6557301672762174},
		{2424, -2.1069832099337953},
		{2425, -2.3403481065947607},
		{2426, -1.4360298970556964},
		{2427, 2.9243967261364743},
		{2428, 2.5175576452294104},
		{2429, 1.9895456754699548},
		{2430, -0.2949414286212606},
		{2431, 0.2523436142415772},
		{2432, 0.3608759126057654},
		{2433, 0.8822148191532098},
		{2434, 1.918001790319565},
		{2435, 1.488805396353943},
		{2436, 2.727072438816257},
		{2437, 1.8625001636966851},
		{2438, 0.24466601103967545},
		{2439, 0.4535609359171422},
		{2440, -0.03404206377320934},
		{2441, -1.7893095585369914},
		{2442, -1.0486419732889998},
		{2443, -4.78531546043221},
		{2444, -2.7534896498934023},
		{2445, -4.370951476742899},
		{2446, -0.1266705110245716},
		{2447, -5.722890494739227},
		{2448, -0.16013186554584724},
		{2449, -5.791395441010053},
		{2450, 0.35097690188923014},
		{2451, 1.070598000292344},
		{2452, 0.4085791197541224},
		{2453, 1.7385041906017729},
		{2454, 0.5992859532108158},
		{2455, 2.8299392934973016},
		{2456, 0.9164147831629301},
		{2457, -3.1228604072045543},
		{2458, -6.343226530986285},
		{2459, -0.7898481725938786},
		{2460, -1.528039239662501},
		{2461, 1.2117178673527835},
		{2462, -0.5190738864279494},
		{2463, -0.44277631607498735},
		{2464, -0.06503769768798741},
		{2465, -2.994407217775744},
		{2466, 0.06562334148780247},
		{2467, 0.00829164438526896},
		{2468, 4.485005058532328},
		{2469, 5.019932959093665},
		{2470, 0.42421695331849296},
		{2471, 3.375520493164291},
		{2472, 0.3121561957337018},
		{2473, 1.314394049759347},
		{2474, 1.9689969373525165},
		{2475, 0.29078494521141424},
		{2476, 5.656498222213303},
		{2477, 3.9271067455728286},
		{2478, -5.702918252681913},
		{2479, -3.7358381978063173},
		{2480, -1.156154426517952},
		{2481, 1.0847010488725808},
		{2482, 0.6155746797189414},
		{2483, 1.4117934454124488},
		{2484, 0.5546134267570109},
		{2485, 7.493490832748605},
		{2486, 2.834163261993176},
		{2487, 1.4262535512086183},
		{2488, 4.312782541430438},
		{2489, 0.21078481992274933},
		{2490, -0.6523763933795719},
		{2491, -4.710987165559514},
		{2492, -4.308470489552351},
		{2493, 1.5177338551051975},
		{2494, 3.336664030610239},
		{2495, 1.03748289139169},
		{2496, 1.2886337540299508},
		{2497, 0.10572391228256711},
		{2498, -1.1879926811762098},
		{2499, -2.3753679747529395},
		{2500, -1.094615354204732},
		{2501, 0.2790146560820532},
		{2502, 1.61663006516697},
		{2503, -1.0031720564440296},
		{2504, -0.3888113058180892},
		{2505, 0.6876922550406408},
		{2506, 3.0378824631375965},
		{2507, -5.86485412208614},
		{2508, -2.062572711791141},
		{2509, -0.2236206974772349},
		{2510, -0.17915145534715995},
		{2511, 0.7722103057774081},
		{2512, 0.2765284523809051},
		{2513, 2.1402751548586476},
		{2514, -1.6638648935469273},
		{2515, -0.6732191088640509},
		{2516, 2.9828012673380524},
		{2517, 0.7888722922331073},
		{2518, 0.6988685284029765},
		{2519, 5.432087439546415},
		{2520, 3.2069994037635734},
		{2521, 3.6026811262245566},
		{2522, -5.22340177685686},
		{2523, -1.3028167869980414},
		{2524, 0.6323959761440688},
		{2525, 1.2891945857629041},
		{2526, -1.2132949069858956},
		{2527, 0.8421039027682831},
		{2528, -1.2514778913693079},
		{2529, 5.758075143172992},
		{2530, 0.5330895110898546},
		{2531, 0.029475108824590424},
		{2532, -0.3434026292511491},
		{2533, 0.2960957010064031},
		{2534, -0.08423789931672701},
		{2535, -1.9792095789913915},
		{2536, -0.8319122870372051},
		{2537, 2.1657310283113302},
		{2538, 1.2555174211929843},
		{2539, -2.886645397070607},
		{2540, -0.29167662784616477},
		{2541, 0.32315877458149983},
		{2542, -0.521942666131929},
		{2543, 0.9706135477248169},
		{2544, 2.3137307055346263},
		{2545, -1.2788039357786327},
		{2546, -1.736665882552293},
		{2547, -1.730425962630914},
		{2548, -0.5913989882922093},
		{2549, -2.034501508116309},
		{2550, -1.0021745030550777},
		{2551, -0.3523307400407339},
		{2552, 2.778247649752519},
		{2553, 3.0335727324452133},
		{2554, 1.0707673377200715},
		{2555, 0.1328317084037181},
		{2556, 1.4975943590754541},
		{2557, 0.5579250381106804},
		{2558, -5.28142392668464},
		{2559, -1.9010141724177654},
		{2560, 2.4425499447804695},
		{2561, 2.047179685081415},
		{2562, -6.579599611432589},
		{2563, -3.8417602101669504},
		{2564, -3.466765061613761},
		{2565, -1.361628226737223},
		{2566, 0.36042109647066833},
		{2567, 0.06481537066496079},
		{2568, -7.169338799589761},
		{2569, -3.6283795132684244},
		{2570, -1.834772544723884},
		{2571, -5.21027363841999},
		{2572, -5.151623445123411},
		{2573, -4.749705886318473},
		{2574, -1.6164471288743343},
		{2575, 2.1317114566582434},
		{2576, -2.6973371554574896},
		{2577, -0.6093231635829994},
		{2578, -5.919356347489001},
		{2579, -5.115197236390259},
		{2580, -2.0249228847283316},
		{2581, -0.6710361991216449},
		{2582, -0.5957541240792815},
		{2583, 1.3860058531961412},
		{2584, 0.518879098620981},
		{2585, 2.280522947931778},
		{2586, 7.031952352627274},
		{2587, 3.2464240363409815},
		{2588, 0.787353612042264},
		{2589, 3.8069899298667513},
		{2590, 2.518233733068578},
		{2591, 1.7397715396786007},
		{2592, -0.4466150895022878},
		{2593, -0.7628207565931057},
		{2594, -2.818478375016418},
		{2595, 0.7203815180821611},
		{2596, -6.086142237136778},
		{2597, -2.7301471558226975},
		{2598, -1.1227396225080026},
		{2599, 3.6919535042821003},
		{2600, -1.2241345621669004},
		{2601, -0.01278744669635451},
		{2602, 0.2230127385025658},
		{2603, 0.0317901027506237},
		{2604, -0.4332071879192148},
		{2605, -0.06760985050759744},
		{2606, -1.2476957287553347},
		{2607, -0.49409711846678517},
		{2608, 0.9007326910768805},
		{2609, -1.1667544632688642},
		{2610, 1.842176574999661},
		{2611, 0.6977969482251878},
		{2612, 1.4598969916395697},
		{2613, 0.5783100274308494},
		{2614, 0.8919402932890941},
		{2615, -0.2981414579022361},
		{2616, 0.6103415937009841},
		{2617, 2.8454061438466023},
		{2618, -0.14139726570783062},
		{2619, -4.687415340950194},
		{2620, -2.5480221996232446},
		{2621, -11.479809387898252},
		{2622, -8.723284761210696},
		{2623, -6.550102394618799},
		{2624, -5.87969475153505},
		{2625, -1.9651884138580626},
		{2626, 2.0417213960196987},
		{2627, -0.055372954198546376},
		{2628, 0.21832564812333977},
		{2629, 1.0089347734452803},
		{2630, -0.8727334847696173},
		{2631, -1.5746296332905785},
		{2632, -3.0789136601287903},
		{2633, -1.5133562649256422},
		{2634, 2.3454763372338028},
		{2635, -1.7901019209899913},
		{2636, 0.7094094038990466},
		{2637, 5.317431896905036},
		{2638, 1.5621579403580534},
		{2639, -0.10123049744212842},
		{2640, 3.475999438167371},
		{2641, 0.6539220561007664},
		{2642, 1.5480858560041113},
		{2643, 2.2950959905294455},
		{2644, 0.42818915211387365},
		{2645, 1.2164016615556192},
		{2646, 1.0272390193522787},
		{2647, 1.436328299260126},
		{2648, -0.23643319932104156},
		{2649, 1.065681762839583},
		{2650, 1.054063255878005},
		{2651, 0.7602961503702268},
		{2652, -1.6131733707145801},
		{2653, 0.07540802328336427},
		{2654, 0.1752586879231756},
		{2655, 0.3421440163255895},
		{2656, 2.5511089008653083},
		{2657, 3.1208644970901105},
		{2658, 2.0930801902907046},
		{2659, 0.8764554308869623},
		{2660, 0.5985826318541073},
		{2661, 0.12984715200448},
		{2662, -4.779230879176144},
		{2663, -4.061120189950192},
		{2664, 0.5566476339823927},
		{2665, -1.4582793854193157},
		{2666, 0.8844858153473883},
		{2667, -0.1549834109811612},
		{2668, -2.118834806567754},
		{2669, 6.025250957638738},
		{2670, 3.1228448058697986},
		{2671, 1.70431470031208},
		{2672, 0.9433824590278245},
		{2673, -0.5485915805817524},
		{2674, -0.19360002569650636},
		{2675, 0.3937255673223166},
		{2676, -2.3822146259024173},
		{2677, -0.9168567601187981},
		{2678, -0.49997214454687744},
		{2679, 0.02654877584535581},
		{2680, -5.903752734706935},
		{2681, -8.871137060394659},
		{2682, -7.347965694424206},
		{2683, -2.4345393339062773},
		{2684, -1.040192347184557},
		{2685, -1.9702194272642157},
		{2686, -1.64612370344563},
		{2687, -0.4353414455424328},
		{2688, -1.9870276075295634},
		{2689, 0.5816778680737054},
		{2690, 4.185613487910101},
		{2691, 3.0481890776634177},
		{2692, -0.014118486172473244},
		{2693, -6.42966078269949},
		{2694, -3.6837314775594843},
		{2695, -0.9804354634735839},
		{2696, 3.9241926196540904},
		{2697, 1.819983190433026},
		{2698, -0.8414566830424961},
		{2699, -1.7607634175891935},
		{2700, 2.771281063927651},
		{2701, 1.128586824243509},
		{2702, -0.03642992213462515},
		{2703, 3.2735706006720875},
		{2704, 0.49901313661046987},
		{2705, -2.2489969780536274},
		{2706, 1.4131622747345793},
		{2707, -3.457546420579296},
		{2708, -3.487274802697844},
		{2709, 1.7054667687943108},
		{2710, -1.015962536849444},
		{2711, -0.8692863169283973},
		{2712, -1.5902530286299177},
		{2713, -4.072821335514558},
		{2714, -0.19949445669763355},
		{2715, 2.089910006794851},
		{2716, -1.5779826145558589},
		{2717, 4.504799675650527},
		{2718, -0.049293049299998115},
		{2719, -1.0920286761343985},
		{2720, -0.1457403420485937},
		{2721, 3.5913896309960545},
		{2722, 3.1568439056435293},
		{2723, 1.7185477697817308},
		{2724, -2.0889614919548594},
		{2725, 5.363630653790146},
		{2726, -1.100402528958441},
		{2727, -4.009785928202986},
		{2728, -4.136594136239643},
		{2729, -2.8817185584518814},
		{2730, -1.325034331923308},
		{2731, 1.1025273550231933},
		{2732, -1.3039324966349726},
		{2733, 3.3827539442515775},
		{2734, -2.0183327021262527},
		{2735, -2.2166014757352195},
		{2736, -5.522290009343159},
		{2737, 0.4914540027299177},
		{2738, 0.327476817679055},
		{2739, 1.5522211667538337},
		{2740, 1.4686507171232916},
		{2741, -4.260449710167407},
		{2742, -1.576246723699202},
		{2743, 1.0384925451828213},
		{2744, 1.3881143965392235},
		{2745, 1.6397578354854767},
		{2746, -1.9312025332764704},
		{2747, -2.7326261301401438},
		{2748, -9.74022899781386},
		{2749, -3.7780863854327746},
		{2750, 2.3156614191944263},
		{2751, -0.7921435977967175},
		{2752, 1.8264088975526556},
		{2753, 1.3846978748918695},
		{2754, 0.23259430458253832},
		{2755, -2.0563850069148755},
		{2756, -4.796231508265559},
		{2757, -2.6468602428925463},
		{2758, -1.2144762982458714},
		{2759, -0.11718425665772159},
		{2760, -1.3761433750147183},
		{2761, -8.858367793483852},
		{2762, -0.3014127020995758},
		{2763, 0.7607724896482395},
		{2764, 0.3026573215043018},
		{2765, 0.4540139920233084},
		{2766, 1.0225942556134076},
		{2767, -3.565260742140125},
		{2768, -1.0873770880213225},
		{2769, -1.0232305806453026},
		{2770, -0.21184656609450842},
		{2771, -0.10195136073133615},
		{2772, 0.8553482199401655},
		{2773, -0.7309870737654864},
		{2774, 0.8790815659495539},
		{2775, 0.05744497342632776},
		{2776, 0.5823586651573369},
		{2777, -0.7823636768283317},
		{2778, -2.9476104086034183},
		{2779, 0.40815734484273136},
		{2780, 0.4758091496783761},
		{2781, 0.4299970460702908},
		{2782, 1.2693917708776647},
		{2783, -3.588897741578766},
		{2784, -0.005557342794870523},
		{2785, -1.1718558556828516},
		{2786, -0.6704625123256684},
		{2787, -4.903376919823369},
		{2788, -1.9015966028492886},
		{2789, -0.708875212834064},
		{2790, -4.041787286034525},
		{2791, -0.11270571754714798},
		{2792, -0.22559697830740735},
		{2793, -0.3271446275816067},
		{2794, -3.0206054043327577},
		{2795, 1.9658336562476082},
		{2796, -2.1327629688259697},
		{2797, -0.28955055026934073},
		{2798, -1.642638576468148},
		{2799, 1.0327860271974805},
		{2800, 0.3706422515095},
		{2801, -3.6703980853955125},
		{2802, -3.9897990268255557},
		{2803, 1.1941360268050234},
		{2804, 0.3838906499169208},
		{2805, -2.255394423328145},
		{2806, -4.458104195088911},
		{2807, -1.7158987462637303},
		{2808, -1.4818418557740303},
		{2809, 0.5797863919084156},
		{2810, 0.23076029501218268},
		{2811, 0.5052177941203301},
		{2812, -1.9986590329136023},
		{2813, -3.968035382374734},
		{2814, -7.953246281514676},
		{2815, -3.793771801581811},
		{2816, -1.6625584477400954},
		{2817, -0.5084816047447149},
		{2818, -0.28681806372203905},
		{2819, -0.2824814775537562},
		{2820, 2.4063273278208315},
		{2821, 1.4708671716228383},
		{2822, 0.8373129738118636},
		{2823, -1.8638886560051886},
		{2824, -1.9104465033068148},
		{2825, -1.4702468278041714},
		{2826, -0.25146669950275624},
		{2827, 0.055133433859721886},
		{2828, -3.4863735171524137},
		{2829, -4.21848627708564},
		{2830, -1.6104946435951233},
		{2831, -1.145134063622684},
		{2832, 0.4685413738988973},
		{2833, 0.21215524203740682},
		{2834, -0.9501992988031602},
		{2835, -1.629383276145903},
		{2836, -0.7477598454918031},
		{2837, -0.6194687594949355},
		{2838, 0.6021113574752421},
		{2839, 4.734526685917843},
		{2840, -1.6506224282768298},
		{2841, -0.2791023757333173},
		{2842, 2.483955781874896},
		{2843, 1.0134730782207357},
		{2844, 2.428093845057015},
		{2845, 1.2139270578621937},
		{2846, 4.1254603721763585},
		{2847, 3.5031485660937127},
		{2848, 0.307767027692335},
		{2849, 2.1634856793283364},
		{2850, -0.5971658568728271},
		{2851, -3.721801081774384},
		{2852, -9.280662970044725},
		{2853, -4.280007111561263},
		{2854, -2.0689574616659394},
		{2855, -0.7777944683055574},
		{2856, -0.056838040579720306},
		{2857, -0.53915029394585},
		{2858, -0.2336452825335895},
		{2859, -2.122185345135698},
		{2860, -0.9975515408657263},
		{2861, 1.6471723385674126},
		{2862, 0.4282359736896837},
		{2863, -0.47309707592969297},
		{2864, -3.399919275731638},
		{2865, 1.5175701785360147},
		{2866, 1.499422461204661},
		{2867, -0.2079167302254621},
		{2868, -5.007068142903045},
		{2869, -2.036982990632476},
		{2870, -3.6427127222780293},
		{2871, -1.5726862777209611},
		{2872, -5.477396034425059},
		{2873, -8.69818323074708},
		{2874, -0.5553131554485855},
		{2875, -1.2941733175310082},
		{2876, -0.2863423988831974},
		{2877, -2.163255263418175},
		{2878, 0.1256951695918258},
		{2879, -5.117952967026661},
		{2880, -0.1458141952484675},
		{2881, 0.06910209538562319},
		{2882, -1.0390459318495051},
		{2883, -0.4778020449712107},
		{2884, 0.7354348492957297},
		{2885, -0.7908795145609446},
		{2886, -0.30319166149249865},
		{2887, 0.01251056425572393},
		{2888, -0.7016224559059285},
		{2889, 0.9168731510644221},
		{2890, 0.4753551795716026},
		{2891, 0.2683455247806897},
		{2892, 4.704720556722834},
		{2893, 3.891866176319808},
		{2894, 3.47755515191798},
		{2895, 3.0341682231555263},
		{2896, 6.872563087638162},
		{2897, 6.947281835122414},
		{2898, -0.741815214284546},
		{2899, -0.22338164651818482},
		{2900, 1.091386111563783},
		{2901, 1.165141999066148},
		{2902, 0.44739766502693},
		{2903, 0.02734914596380872},
		{2904, -1.2422567437270242},
		{2905, 0.8963931286455886},
		{2906, 0.13807517444276718},
		{2907, 0.7994874747310817},
		{2908, -0.3489606224853149},
		{2909, 5.9380528594876765},
		{2910, 5.042506145452073},
		{2911, 3.2985022864855864},
		{2912, -0.29155065113093337},
		{2913, 2.333163109838315},
		{2914, 0.7225165805078064},
		{2915, 0.8728843051463344},
		{2916, -0.21573641417559453},
		{2917, 0.6696935292265799},
		{2918, 0.8634268093891694},
		{2919, 0.6387493789922947},
		{2920, -1.066088047556552},
		{2921, 0.2870830625084744},
		{2922, 4.368042476483579},
		{2923, -0.134351119704486},
		{2924, -2.536381605980541},
		{2925, -0.831375436206817},
		{2926, -1.8641152546592674},
		{2927, 2.199788960825248},
		{2928, -6.693955043287469},
		{2929, -8.653962586832874},
		{2930, 0.8426799572847705},
		{2931, -0.8635455570726671},
		{2932, -0.43304011133499243},
		{2933, -3.6231206416054116},
		{2934, -0.21443564867919673},
		{2935, -0.35556853096643354},
		{2936, 1.4975677047897884},
		{2937, 1.9884593275498603},
		{2938, 5.59726078810444},
		{2939, 4.077303721134671},
		{2940, 0.902027035436961},
		{2941, -2.3836989520678946},
		{2942, -0.3575067245176966},
		{2943, 1.1562960013978785},
		{2944, 4.9951203638556},
		{2945, 3.392231602529498},
		{2946, 1.069286971622113},
		{2947, -1.8727101179139818},
		{2948, -1.5380782532999229},
		{2949, 1.4608659955389007},
		{2950, 0.8434628398637047},
		{2951, 3.454961594377814},
		{2952, 2.169254840779498},
		{2953, -0.6465362770127656},
		{2954, -3.001454364273441},
		{2955, -0.16378935329735445},
		{2956, 0.29467981372140506},
		{2957, 0.9773015926973174},
		{2958, 0.5846897233561579},
		{2959, 0.1949649100344332},
		{2960, 2.6520707064140465},
		{2961, 1.5620228849842217},
		{2962, -1.019638270163962},
		{2963, -1.4307766042829018},
		{2964, -2.2309021004915737},
		{2965, 1.8713225347712399},
		{2966, 3.308282618847436},
		{2967, 1.316064401463378},
		{2968, -0.17517731347318077},
		{2969, 0.05357352790932929},
		{2970, -0.20046884427478595},
		{2971, 2.4287126871785585},
		{2972, 0.4103209438701796},
		{2973, 4.629895252559033},
		{2974, 2.424224148450027},
		{2975, 0.8965357939000183},
		{2976, 2.318496786405469},
		{2977, -1.8494521649521138},
		{2978, -1.065455287881004},
		{2979, 0.4903567943394889},
		{2980, 0.19018077910596873},
		{2981, -4.82394303263925},
		{2982, -2.498122064279756},
		{2983, 2.660802259741359},
		{2984, 4.512935217504796},
		{2985, -1.6523391238435696},
		{2986, -4.99202933751446},
		{2987, -1.6857442903307478},
		{2988, -0.6605028867856009},
		{2989, 1.5936373857311834},
		{2990, -1.2981647872137374},
		{2991, -2.937490818056503},
		{2992, -0.6646705259802081},
		{2993, -1.5551974364281473},
		{2994, -1.653046545166242},
		{2995, -4.478946423892171},
		{2996, -2.046191721889008},
		{2997, 1.371271698280268},
		{2998, 0.5458322769901551},
		{2999, 0.005942356340884608},
		{3000, -0.021889156130614477},
		{3001, 0.6134669361336337},
		{3002, 0.59224448714449},
		{3003, 2.8033533789329135},
		{3004, 0.47127687368594073},
		{3005, -0.580282829702645},
		{3006, 0.08929618397126707},
		{3007, -3.9206243356281387},
		{3008, -4.867426535040597},
		{3009, -8.1174563661528},
		{3010, -1.7117671549357694},
		{3011, 1.8297340171058463},
		{3012, -0.08015933092771366},
		{3013, 0.2695439511462532},
		{3014, 0.10462876003729889},
		{3015, 0.06269144877200294},
		{3016, -6.7613848851497504},
		{3017, -2.7573050216405233},
		{3018, 4.4065126798220415},
		{3019, -0.24607973331714894},
		{3020, 0.873932316265579},
		{3021, 0.5314336597805744},
		{3022, 5.916624470840528},
		{3023, 4.6154567933921715},
		{3024, 2.0032235781044676},
		{3025, -1.2964828055712758},
		{3026, 0.6694503447123251},
		{3027, 0.9464854860298957},
		{3028, -2.553922631123597},
		{3029, 0.8139854160768611},
		{3030, 0.21184628370811365},
		{3031, 1.3691074160557315},
		{3032, 2.08559184731858},
		{3033, 0.7941314967055972},
		{3034, 1.8070735797330177},
		{3035, 0.25831466575710976},
		{3036, -2.167300372692259},
		{3037, -1.5120000137678618},
		{3038, 0.2917982668224993},
		{3039, 2.1438864514904954},
		{3040, 1.009518623335178},
		{3041, -0.18350701997622848},
		{3042, 0.5964384207474862},
		{3043, 0.24017634111824623},
		{3044, -1.3664224761006967},
		{3045, 1.3158646272421772},
		{3046, 0.8089182549742986},
		{3047, 2.012224508545627},
		{3048, 1.1016685335031176},
		{3049, -2.1032530129217197},
		{3050, 0.2415957663579016},
		{3051, 6.603661857470284},
		{3052, 4.722536879281861},
		{3053, -0.9188532692752507},
		{3054, -1.5495560068937317},
		{3055, 3.1200551841198414},
		{3056, 6.258008784111389},
		{3057, 2.5730879502652284},
		{3058, 0.5212699649366465},
		{3059, 0.75640586591271},
		{3060, 5.0936687649070995},
		{3061, -5.17013635249095},
		{3062, -2.5905790303768734},
		{3063, -2.787157322288717},
		{3064, 3.9001026059622497},
		{3065, 1.4686809384919934},
		{3066, 0.7848044357409844},
		{3067, -6.729824519456675},
		{3068, -2.901765043046497},
		{3069, -1.2162742478676174},
		{3070, -0.4225723348030648},
		{3071, -1.079539882989691},
		{3072, -2.301600022625738},
		{3073, -1.2197486525976933},
		{3074, -2.532802007599388},
		{3075, -1.701101581349481},
		{3076, -0.7396222623688208},
		{3077, 1.4226114922722812},
		{3078, 0.3411530858553503},
		{3079, 1.900405776862524},
		{3080, 0.8859082105599385},
		{3081, 0.30189466719240643},
		{3082, -0.08073282196283577},
		{3083, 0.33068267008817315},
		{3084, 0.48021460850021247},
		{3085, -2.6925990167762164},
		{3086, -1.4420260922575638},
		{3087, -0.8198984379874156},
		{3088, -0.8969272197905644},
		{3089, -0.9910802720556426},
		{3090, -0.2932561558964055},
		{3091, 0.42466525299806535},
		{3092, 2.5240774970621254},
		{3093, -1.6066494200545967},
		{3094, -1.0850536942635085},
		{3095, -4.636471326640537},
		{3096, -7.596887920058231},
		{3097, 0.11465166980345964},
		{3098, -1.8891126791392174},
		{3099, -1.9751699867030847},
		{3100, -3.1751514213694296},
		{3101, -0.9611889661723114},
		{3102, -4.2240533901527835},
		{3103, -2.920570786479122},
		{3104, -2.2563974811962124},
		{3105, -1.9911431908276187},
		{3106, -1.471262491920951},
		{3107, -7.233882407312646},
		{3108, -0.9812432329486742},
		{3109, -1.1345835328971505},
		{3110, 0.5506006412234994},
		{3111, -0.9319535960467329},
		{3112, -0.44257718999659845},
		{3113, 0.7649292874485628},
		{3114, 1.4189070453303227},
		{3115, 0.013682780472447531},
		{3116, 0.4453304390201259},
		{3117, 0.23072429684286838},
		{3118, -0.45457413551832543},
		{3119, -1.6269003205257244},
		{3120, -0.2168809116921543},
		{3121, 0.6823530912549468},
		{3122, 1.5691364939423453},
		{3123, -1.6525114644954848},
		{3124, 2.2999626651174756},
		{3125, 1.681233853755264},
		{3126, 7.4411888931211205},
		{3127, 2.8600648583309916},
		{3128, 0.015741475876890565},
		{3129, 0.06641145539379699},
		{3130, 0.16561637551356173},
		{3131, 5.7255477697068375},
		{3132, 0.9396120518607889},
		{3133, -5.843592958896051},
		{3134, 2.0373279219093217},
		{3135, 0.9255171996545122},
		{3136, 1.5387651536265383},
		{3137, 0.7734746025607809},
		{3138, -1.220235017285657},
		{3139, 3.8594057536703574},
		{3140, -3.9646837626434603},
		{3141, -1.3464501309343706},
		{3142, -1.1623802103107939},
		{3143, -0.26021831850718774},
		{3144, -0.27817132535068545},
		{3145, -1.0648016799279123},
		{3146, -1.3419451340211141},
		{3147, -0.22110719054434785},
		{3148, -0.6648196449121446},
		{3149, -0.6648349896858936},
		{3150, -0.4589808585786279},
		{3151, 1.822423477189921},
		{3152, 1.1706108216399547},
		{3153, -0.5893982156626945},
		{3154, 9.757573174589815},
		{3155, 11.301914954647897},
		{3156, 3.1184217168479753},
		{3157, 0.45558003098376587},
		{3158, 6.043824373186178},
		{3159, 1.6229384965620581},
		{3160, 1.567477448347181},
		{3161, -0.12905479133661002},
		{3162, -0.9677316472650409},
		{3163, -2.4284911517850025},
		{3164, -1.1310784334795847},
		{3165, -1.9029098719634414},
		{3166, -2.3121383810668372},
		{3167, -3.3869206136379475},
		{3168, -1.315944705126176},
		{3169, 0.016764854289352704},
		{3170, -4.597281137640761},
		{3171, -1.809411650071589},
		{3172, -0.05053167168292494},
		{3173, -0.14077855189943075},
		{3174, -0.946283544744339},
		{3175, -2.6432441306526098},
		{3176, -0.37385713584580726},
		{3177, -0.112025339113283},
		{3178, 1.2438402283317953},
		{3179, 0.7469360477167348},
		{3180, -2.553282799520712},
		{3181, -1.7508355623075782},
		{3182, 2.374392078904858},
		{3183, 1.049840870475472},
		{3184, -0.8412292145242308},
		{3185, -1.6438026107202877},
		{3186, 3.501613308865459},
		{3187, 2.820020443046028},
		{3188, 1.1876728708406252},
		{3189, 0.6965919451351277},
		{3190, -3.2139453866878256},
		{3191, -3.905929171434189},
		{3192, -5.687279299607525},
		{3193, -0.7248789865410732},
		{3194, -0.5724531183231758},
		{3195, -0.28437639886909216},
		{3196, -0.19749465362127966},
		{3197, -1.6349883264419576},
		{3198, -1.0785663997844335},
		{3199, -3.6737207007421038},
		{3200, -0.6663196505287051},
		{3201, -1.5442439523670262},
		{3202, -1.761647300681016},
		{3203, 2.9231669719969893},
		{3204, 4.849726594132493},
		{3205, 2.0693736032662637},
		{3206, 1.6592345178172296},
		{3207, 3.6989776122395615},
		{3208, 2.7454432206287125},
		{3209, 1.3259717389978052},
		{3210, 2.8048813940429076},
		{3211, 1.6466007965792189},
		{3212, -1.4421239426390573},
		{3213, 0.23856429328902062},
		{3214, 1.374935386377447},
		{3215, -1.369551864388458},
		{3216, -0.7308283336504797},
		{3217, -1.4206844775068834},
		{3218, -0.10963990411163987},
		{3219, 0.19497018259819865},
		{3220, -0.011753537951629159},
		{3221, 0.02136090514022639},
		{3222, 3.6333720542356507},
		{3223, -0.25547422217692795},
		{3224, -2.301892733950634},
		{3225, -0.8249512858502619},
		{3226, -0.24433811126608435},
		{3227, 0.23956878303343004},
		{3228, -2.3211986873178656},
		{3229, -0.8496337356615131},
		{3230, 1.2171397029400246},
		{3231, 3.218168461298179},
		{3232, 0.9018732908095002},
		{3233, 1.9342532718865273},
		{3234, 0.18749522279858644},
		{3235, 0.25325117811722886},
		{3236, 2.6557751830602743},
		{3237, -2.3663055588663076},
		{3238, -1.7268393934537039},
		{3239, -1.6363013891386577},
		{3240, -0.6547638136365955},
		{3241, -1.4777698841237814},
		{3242, -4.630028585861867},
		{3243, -2.8637642256343923},
		{3244, -4.232264768685361},
		{3245, 0.3010055659085549},
		{3246, 0.15540377267650218},
		{3247, 1.4389893984328834},
		{3248, 2.5546121896222065},
		{3249, 1.310049106011679},
		{3250, 0.06679864975632283},
		{3251, -0.2224707953487741},
		{3252, -0.8837789190237955},
		{3253, 0.10145267675204322},
		{3254, -0.8542916876352927},
		{3255, -0.28665507257198974},
		{3256, 0.46050585047116804},
		{3257, 0.7314462388653797},
		{3258, -4.549639766253245},
		{3259, 4.834701982846138},
		{3260, -3.3334490115060196},
		{3261, -0.72185911463243},
		{3262, 1.0992778888508903},
		{3263, -3.614681464520967},
		{3264, -2.288177944055445},
		{3265, -0.9037161745003117},
		{3266, -2.772010123623196},
		{3267, -0.6942963432754331},
		{3268, 0.14674930170337414},
		{3269, -0.4922051761586409},
		{3270, -3.09399224065798},
		{3271, -1.1961419386544727},
		{3272, 0.2865145722035036},
		{3273, 0.794105599482598},
		{3274, 1.188467845239849},
		{3275, 1.8127518783408376},
		{3276, -0.4746690010622917},
		{3277, 0.12077453597862242},
		{3278, -5.28395938452937},
		{3279, 1.639714665604051},
		{3280, -1.4393075572773382},
		{3281, -4.721419078972631},
		{3282, -1.7920040144368525},
		{3283, -1.5173170810140246},
		{3284, 1.7425218657932477},
		{3285, -0.6063318087728284},
		{3286, -1.4771862527008974},
		{3287, 0.38090120811096506},
		{3288, 1.9621688973051805},
		{3289, -3.2151574527926714},
		{3290, 3.921579427937406},
		{3291, 0.8522262740519965},
		{3292, 0.29858286728633976},
		{3293, 0.382013634168903},
		{3294, 2.6185909319504383},
		{3295, -0.28664952275206335},
		{3296, 0.21675753510010892},
		{3297, 0.03425218077967597},
		{3298, -0.04070483792528275},
		{3299, 0.04605325174367114},
		{3300, -0.837963271928355},
		{3301, 1.0496707599453088},
		{3302, -0.9119769623353375},
		{3303, 0.5563993203799009},
		{3304, -1.7468638628022155},
		{3305, -3.527281756557568},
		{3306, -1.0815097550867363},
		{3307, 0.4983546153095182},
		{3308, 1.2777178190411465},
		{3309, 3.302433769405343},
		{3310, 3.3347878356801646},
		{3311, 0.07743859540899045},
		{3312, -0.7304062587814399},
		{3313, -0.32974829469939426},
		{3314, -1.2277651270676229},
		{3315, 1.9194326778561845},
		{3316, 4.557973533323188},
		{3317, 2.9049794701968796},
		{3318, 1.7069903254979097},
		{3319, 0.014900999680489746},
		{3320, 0.359232733322449},
		{3321, -1.4895166606402215},
		{3322, -0.44446189579445916},
		{3323, 0.9420802126888999},
		{3324, 0.5940824283464141},
		{3325, 0.08227770291097797},
		{3326, -0.007913751505067984},
		{3327, -3.44850073113839},
		{3328, -1.1037844163914003},
		{3329, -0.8833883016746005},
		{3330, -0.23692071471592796},
		{3331, -1.4479259414637702},
		{3332, -1.3989708295021677},
		{3333, -0.7375778243106275},
		{3334, 3.441179924537405},
		{3335, 1.0144910924012736},
		{3336, -0.4720292137005732},
		{3337, -3.4392315261815805},
		{3338, -0.03813743017542759},
		{3339, -0.019536953093575216},
		{3340, 1.1845639121623557},
		{3341, 0.48224005420927957},
		{3342, 0.17010477968792756},
		{3343, -0.5878043634773589},
		{3344, 0.27492603156515605},
		{3345, 2.214362173983477},
		{3346, 0.43930430459527386},
		{3347, 2.3900130527791354},
		{3348, 1.4633748630598356},
		{3349, 1.9015932852067396},
		{3350, 0.7134666679626732},
		{3351, -1.3779273782321606},
		{3352, 3.3557498258870306},
		{3353, 1.5362283765911147},
		{3354, -0.803789819640666},
		{3355, -1.7851577744936489},
		{3356, -0.623733947090548},
		{3357, -0.29645932465283503},
		{3358, 1.6550341126123285},
		{3359, -0.5420083703048337},
		{3360, -1.6183585289614444},
		{3361, -0.5867243677861461},
		{3362, -0.04343336562009084},
		{3363, 2.4534104736510387},
		{3364, 4.160301001587996},
		{3365, 4.6168572057284685},
		{3366, 3.8579498564215644},
		{3367, 1.5257096356355018},
		{3368, 2.1599163501331504},
		{3369, 0.6049463969981193},
		{3370, -0.742965855868237},
		{3371, -2.972177090547649},
		{3372, -2.594909142580179},
		{3373, -0.906107390411771},
		{3374, 1.512108118926779},
		{3375, 1.2643535020912688},
		{3376, -2.9663855678247297},
		{3377, -1.7577474046486354},
		{3378, 0.3197108121355806},
		{3379, -1.3425438232056053},
		{3380, -0.9303367860079368},
		{3381, -0.7669858346184695},
		{3382, 0.31683659280229154},
		{3383, 0.6663792495854027},
		{3384, 0.7581243386273291},
		{3385, -1.294234375398452},
		{3386, 0.8574099487672004},
		{3387, 1.5448268911390846},
		{3388, 3.696855493047551},
		{3389, -0.33283905435911953},
		{3390, 2.3576584374338596},
		{3391, -2.094309715300025},
		{3392, -2.372401183017178},
		{3393, -1.491667361375127},
		{3394, -0.9182426494332743},
		{3395, -0.5668619249642253},
		{3396, -0.26827408961629773},
		{3397, -0.9489968106849853},
		{3398, 0.5219182241835842},
		{3399, -1.263439829398148},
		{3400, 1.4466405190786569},
		{3401, -0.00902702507022457},
		{3402, -1.6660122917071492},
		{3403, -0.6769869646011409},
		{3404, 0.5258550786280587},
		{3405, -0.3104575592757068},
		{3406, -0.19850360350652757},
		{3407, 2.303818208162693},
		{3408, 3.5217295411320433},
		{3409, -4.453591766367026},
		{3410, -2.6181050959873344},
		{3411, 0.8905377903316667},
		{3412, 1.8282777461920623},
		{3413, 1.35723814478877},
		{3414, 0.9164080061569506},
		{3415, -2.1016430296880735},
		{3416, 0.4052258840313093},
		{3417, 2.534329108346545},
		{3418, 0.447902221517246},
		{3419, -2.8947081033695183},
		{3420, -0.3612702282796464},
		{3421, 3.9439526694836093},
		{3422, 3.7302593688902803},
		{3423, 1.5779766290914872},
		{3424, -3.984050085807984},
		{3425, -1.678037382918316},
		{3426, -6.838972983218538},
		{3427, 1.9144706300644039},
		{3428, 2.0295962928738756},
		{3429, -1.5096599071465233},
		{3430, -0.6735166514124974},
		{3431, 0.6957713867547404},
		{3432, -1.1174833319895452},
		{3433, -0.7556752707704717},
		{3434, -0.3894017705533286},
		{3435, -3.327675686943565},
		{3436, -0.35332420499511297},
		{3437, -3.269675168663621},
		{3438, -1.1374963621711838},
		{3439, -0.10507551834010248},
		{3440, 0.24012493177409464},
		{3441, 1.0433239525692803},
		{3442, -3.457331834784135},
		{3443, -0.8419813526148783},
		{3444, -1.771794613807026},
		{3445, -0.5002542251675972},
		{3446, -1.0324364026857087},
		{3447, -1.2470030202791786},
		{3448, -1.241993304293668},
		{3449, -0.15031931254503045},
		{3450, 2.8077625878366277},
		{3451, 1.3102720890350639},
		{3452, 0.9900297677910657},
		{3453, -0.12514785268008055},
		{3454, -0.7611220477014748},
		{3455, 0.9952682149779505},
		{3456, 0.5141390084095174},
		{3457, 0.16364672080123516},
		{3458, -2.244745284504577},
		{3459, 2.482143184019395},
		{3460, 2.810907191784789},
		{3461, 0.5280419829620705},
		{3462, 4.02456469669884},
		{3463, 1.6627531031308167},
		{3464, 3.435877214413484},
		{3465, -2.828522333093734},
		{3466, -3.0116602601906846},
		{3467, -1.196464552964846},
		{3468, 1.773338371205283},
		{3469, -1.5609837461970595},
		{3470, 0.4656114009008133},
		{3471, -4.873443424350407},
		{3472, -1.5244178475368768},
		{3473, -1.5769679053043584},
		{3474, 5.3109211590756695},
		{3475, 3.2755563099190326},
		{3476, 1.5743265882749105},
		{3477, 1.1142770792861567},
		{3478, 0.3958727515723008},
		{3479, -0.4118125762660592},
		{3480, -1.2537467233047062},
		{3481, -3.466798042967472},
		{3482, -1.365985778452746},
		{3483, -0.42149124913148},
		{3484, -0.1742792788239082},
		{3485, -0.353226357439042},
		{3486, -0.8924737895572716},
		{3487, -4.676561869301092},
		{3488, 0.07933192150047352},
		{3489, -0.05975549314175007},
		{3490, 0.2866022425368644},
		{3491, -2.292922133073971},
		{3492, 4.6825643262642735},
		{3493, 2.7871771210756315},
		{3494, 0.7391955182403545},
		{3495, -0.3286925592677028},
		{3496, 0.9875390799583313},
		{3497, -3.247243658186449},
		{3498, -3.635035504985291},
		{3499, -1.5480676384699568},
		{3500, 0.2464664722685832},
		{3501, 0.4402521278204557},
		{3502, 5.270795892703029},
		{3503, 2.8042840307712966},
		{3504, 0.9462329467776176},
		{3505, 4.083396433142653},
		{3506, 1.9549087755026735},
		{3507, 0.16950002359844074},
		{3508, -5.621626699206365},
		{3509, 0.7065464722430463},
		{3510, -0.13077056815642224},
		{3511, 1.054445256609005},
		{3512, 2.275456165193968},
		{3513, 1.522726795164989},
		{3514, 3.613290525107368},
		{3515, 7.641241371078061},
		{3516, -0.8596201483779455},
		{3517, 3.8383082775973665},
		{3518, 4.415536447532871},
		{3519, 4.925444188582757},
		{3520, 0.5529614115127446},
		{3521, -0.43249773935493385},
		{3522, -0.3978731569685626},
		{3523, 2.369258624946799},
		{3524, 0.11770305274025916},
		{3525, -0.2578544359620139},
		{3526, 0.4436567116513679},
		{3527, -2.0567264988569036},
		{3528, -1.3031023140703584},
		{3529, -0.5677076061354076},
		{3530, -0.32569038241559284},
		{3531, -0.3763919144990273},
		{3532, 0.37043547850101466},
		{3533, -1.3365595082069066},
		{3534, -2.1213348849814486},
		{3535, 2.6705428141883254},
		{3536, 1.1490225959669937},
		{3537, -1.403844095804133},
		{3538, -1.2107872921871796},
		{3539, -1.915345446963727},
		{3540, 2.3402350255202062},
		{3541, 1.2938466844903567},
		{3542, 1.0476000137979424},
		{3543, 0.42858056243474657},
		{3544, 1.4174426905469966},
		{3545, 1.6794181342290262},
		{3546, 1.0441517752627636},
		{3547, 0.4535503817633613},
		{3548, 2.9867039327249554},
		{3549, 3.866279224928834},
		{3550, -0.9998712163569154},
		{3551, -1.5359997925869895},
		{3552, -3.350138782735296},
		{3553, 0.776572306531955},
		{3554, 1.724499212677416},
		{3555, 3.7005067041712887},
		{3556, -1.4306309705193057},
		{3557, 1.1042140950104118},
		{3558, -3.0720237631568423},
		{3559, -2.292693963673056},
		{3560, 2.0631655685116437},
		{3561, 0.5209021693288604},
		{3562, -0.8516657965665261},
		{3563, -0.6692847810035192},
		{3564, -4.936828660287376},
		{3565, -1.0275786350860048},
		{3566, 2.0836937258738093},
		{3567, 5.038756843295271},
		{3568, 2.6138052449635953},
		{3569, 1.6702403645036632},
		{3570, 0.6771004419799107},
		{3571, 1.0012042330108164},
		{3572, -0.8684664464699946},
		{3573, 3.0057117627900882},
		{3574, -2.5842947892441828},
		{3575, -0.8264679790036954},
		{3576, -1.8206411475201065},
		{3577, 2.041200349259358},
		{3578, 2.7401530695434198},
		{3579, 4.002299395552687},
		{3580, 4.189786170973092},
		{3581, 1.7420421361967129},
		{3582, -0.08620519905246926},
		{3583, -0.14610869269882912},
		{3584, 0.718559899854734},
		{3585, -2.6781823290433806},
		{3586, -1.0539034526402162},
		{3587, -0.12943651733523087},
		{3588, -0.39575196081465225},
		{3589, 3.3853687651990527},
		{3590, -0.7602870075192598},
		{3591, -0.5035957932759519},
		{3592, 1.5579661417136255},
		{3593, 0.6310252072360384},
		{3594, 0.3084611921089204},
		{3595, -0.43867754216855454},
		{3596, -2.9870298227672825},
		{3597, -7.452151240243173},
		{3598, 1.025026788234816},
		{3599, 0.8431018734564414},
		{3600, -0.007710004445343199},
		{3601, 2.02952620490366},
		{3602, 0.6518418515927574},
		{3603, -1.7005366683681715},
		{3604, 0.3280229403098107},
		{3605, 0.5393576992077274},
		{3606, 0.37692073889648303},
		{3607, -5.3881045158602285},
		{3608, -2.2245706813004036},
		{3609, -6.251763810973281},
		{3610, -3.075774811824537},
		{3611, -3.8836941002906906},
		{3612, 1.155803661808641},
		{3613, 5.2337018370705035},
		{3614, 3.760409409708025},
		{3615, 1.5699018637090039},
		{3616, 2.1419714492105975},
		{3617, -4.358271898563193},
		{3618, -5.513322924100752},
		{3619, -4.406630347423569},
		{3620, 0.1966789557454678},
		{3621, 0.7520648686155538},
		{3622, -0.13281827409806252},
		{3623, -0.20027365005112527},
		{3624, 4.357115314297288},
		{3625, 2.6178110923762605},
		{3626, 2.886222769436089},
		{3627, 4.106097417737272},
		{3628, 2.2217681804418286},
		{3629, 0.5782209291992357},
		{3630, 0.39722656325807243},
		{3631, 0.10186702404312531},
		{3632, 0.059131764158733766},
		{3633, 2.2827620028740485},
		{3634, -0.6464373942320885},
		{3635, 0.20005310852881675},
		{3636, -0.8903914826383801},
		{3637, 1.052767810265875},
		{3638, 0.8978650565777415},
		{3639, -0.6542901763717243},
		{3640, 1.416555235775439},
		{3641, 0.24045288985471575},
		{3642, 0.09531593901029105},
		{3643, -1.383902178406173},
		{3644, 1.367363249232728},
		{3645, 0.2534727954556751},
		{3646, 0.9554610145041108},
		{3647, -0.689573060493075},
		{3648, -4.000765244129101},
		{3649, 0.48818205394824177},
		{3650, 0.5310130909524258},
		{3651, 0.194516550805345},
		{3652, -0.40557558979081376},
		{3653, -2.9956308409383787},
		{3654, -6.4614724994835795},
		{3655, -1.5048528488579056},
		{3656, -0.647230372456216},
		{3657, -0.3806586424868038},
		{3658, 0.9475810382217537},
		{3659, 0.48370616021752055},
		{3660, 1.3121486184480695},
		{3661, 0.035495062178133474},
		{3662, -6.113538037038059},
		{3663, -2.0354848727814514},
		{3664, 2.1475176334116783},
		{3665, 3.1444353305796118},
		{3666, 1.7135313889545842},
		{3667, 5.898123225098903},
		{3668, 3.612394872736382},
		{3669, 1.483430007107794},
		{3670, 0.8287511982151257},
		{3671, 0.5328197602270268},
		{3672, -0.19291632539275477},
		{3673, 0.4805196600996498},
		{3674, 1.3797367665693825},
		{3675, 3.667176507414775},
		{3676, -2.5593234489302716},
		{3677, -2.344054869106955},
		{3678, -1.9252892029569812},
		{3679, -4.647325115310373},
		{3680, -2.062789418154639},
		{3681, 0.21858357055940036},
		{3682, -7.078871226782706},
		{3683, -0.9122620893253444},
		{3684, -0.5576425335815458},
		{3685, -3.8668101970384736},
		{3686, 0.5733177267483516},
		{3687, -1.5333563956926417},
		{3688, 4.308471448130455},
		{3689, 1.158882929946008},
		{3690, 1.1079084728530937},
		{3691, 0.1070707828820412},
		{3692, 2.436832427243785},
		{3693, 2.0121070406310544},
		{3694, 1.2118256186294054},
		{3695, 0.7629104511972078},
		{3696, 0.7716886145745803},
		{3697, -0.20137379738103017},
		{3698, 2.635432019964343},
		{3699, 4.044974986831757},
		{3700, 1.339620107669096},
		{3701, -1.894798039376997},
		{3702, 3.6466574852412323},
		{3703, 0.5498081857587207},
		{3704, 0.5359422991070406},
		{3705, 4.31024131197643},
		{3706, 0.48500372850882867},
		{3707, 4.458159210621078},
		{3708, 2.5820038928857283},
		{3709, 1.2827178303780218},
		{3710, 0.1274493147196024},
		{3711, -0.6668797955966457},
		{3712, -0.4987469813424097},
		{3713, -0.6463874863117756},
		{3714, -2.0050618552727313},
		{3715, -1.363629419702245},
		{3716, -1.690540985181837},
		{3717, 1.2798086355033744},
		{3718, 0.5311118149323346},
		{3719, -2.3613775223822326},
		{3720, -0.4958881615432906},
		{3721, -6.107733880501552},
		{3722, -3.5549452208531194},
		{3723, -1.8929491827803278},
		{3724, -5.360098637751946},
		{3725, -0.6855885078767818},
		{3726, 4.512926181533091},
		{3727, 8.262553435511572},
		{3728, -0.35786748699372284},
		{3729, -2.248657862092917},
		{3730, -1.533826882414015},
		{3731, 1.5942677446024778},
		{3732, 1.3955966779682822},
		{3733, 2.5951560236478746},
		{3734, 0.26789948313106027},
		{3735, 0.5654149958289705},
		{3736, 0.02816436639558223},
		{3737, 0.05911086680261711},
		{3738, -1.4160321817691317},
		{3739, 2.9842219091452025},
		{3740, 4.961376415983401},
		{3741, 5.7561874066047505},
		{3742, -4.426453987167468},
		{3743, -0.7523160525530008},
		{3744, 1.0760354036727577},
		{3745, 1.061988513739246},
		{3746, 0.16115154670813608},
		{3747, 0.12092430567976764},
		{3748, 3.5224902390534734},
		{3749, 1.2991048632489155},
		{3750, 0.782189422861838},
		{3751, -0.0014212383308839227},
		{3752, -7.339067843187849},
		{3753, -5.602182556426843},
		{3754, -3.102418390999827},
		{3755, -0.4290644654621022},
		{3756, -0.019144569222709046},
		{3757, -1.2166990306468117},
		{3758, 2.112595961356108},
		{3759, 1.5483888134915245},
		{3760, 3.1299322729737677},
		{3761, 0.5801127530387525},
		{3762, 1.0523157667297522},
		{3763, 0.5772632008120191},
		{3764, 0.4615755668132708},
		{3765, -2.5518677493368367},
		{3766, -0.8024676079465085},
		{3767, -0.3614059416064488},
		{3768, 0.2138625753113225},
		{3769, 0.030977677906406056},
		{3770, -0.299639184090898},
		{3771, -3.7463220570725975},
		{3772, -1.6964185239732832},
		{3773, -0.8082588963951713},
		{3774, -0.9193081941740062},
		{3775, -0.3046608434374557},
		{3776, 0.9253415150380421},
		{3777, 1.2510449625853781},
		{3778, 3.485466208969293},
		{3779, 1.6555584770633813},
		{3780, 1.2259600670986335},
		{3781, 0.21651047706400345},
		{3782, -1.4181618805302225},
		{3783, -2.9110349041377264},
		{3784, -1.5644934762408225},
		{3785, -0.19535461604564902},
		{3786, -1.5938982282076615},
		{3787, -0.6841708835515129},
		{3788, -0.3277255305855965},
		{3789, 1.9635106212634863},
		{3790, 6.794649793111887},
		{3791, 3.035075731392759},
		{3792, -1.079739185672618},
		{3793, 1.4735629685282676},
		{3794, 0.05208771191570871},
		{3795, -3.7583439143053},
		{3796, -1.2550037467330013},
		{3797, 1.6811922840179006},
		{3798, 1.8263103481947345},
		{3799, 1.184801176687889},
		{3800, 0.5392977614220245},
		{3801, 2.2931603295208007},
		{3802, 2.238114237900401},
		{3803, -0.31088386772142684},
		{3804, -0.2465302724040442},
		{3805, 3.539480501502044},
		{3806, 1.349817758510164},
		{3807, 0.8179195795365517},
		{3808, 0.0874014751081702},
		{3809, 0.019008480630096667},
		{3810, 0.9269920733367198},
		{3811, 0.6009254518892028},
		{3812, 3.089124327332928},
		{3813, 1.0060062245895998},
		{3814, 1.3715127137125078},
		{3815, -1.1553138110026606},
		{3816, -5.442180687137575},
		{3817, -3.491109173677031},
		{3818, -1.5681393936164818},
		{3819, 0.2178832841449776},
		{3820, 0.908094343457441},
		{3821, -0.7350204980991375},
		{3822, -0.40416846133574325},
		{3823, -2.385984529090015},
		{3824, -0.4816718358226062},
		{3825, 0.2209950376333574},
		{3826, 0.105380696771281},
		{3827, 2.6222838704294067},
		{3828, -7.005058657205559},
		{3829, -1.357780948428518},
		{3830, -0.2445025885961341},
		{3831, -1.6267870439590153},
		{3832, 0.13045803726132088},
		{3833, -2.8329086673131934},
		{3834, -2.253093763029299},
		{3835, -4.3847885619892555},
		{3836, -0.7944160246928461},
		{3837, -0.21638408786581492},
		{3838, -0.2896340665366428},
		{3839, 3.997238624684931},
		{3840, 1.6207301614017904},
		{3841, 4.352108502755206},
		{3842, 1.632962860832281},
		{3843, -0.010343447598495259},
		{3844, 2.6517742384029344},
		{3845, -0.8298164393688887},
		{3846, -2.8968623064555388},
		{3847, -1.6729841233127258},
		{3848, 0.2516268809345146},
		{3849, 6.879837374755038},
		{3850, 3.493804514508959},
		{3851, 1.280853198020833},
		{3852, 2.0655175258904817},
		{3853, 2.0994819287189337},
		{3854, -3.5450779001325268},
		{3855, -1.6610593232161173},
		{3856, 0.003710678998797068},
		{3857, 2.1740186633662053},
		{3858, 2.0166168916512075},
		{3859, 0.33230885101830077},
		{3860, -1.403471823418949},
		{3861, 6.706226251356585},
		{3862, 3.268723699626104},
		{3863, 3.896856540280541},
		{3864, 3.93933089432305},
		{3865, 6.850167434144977},
		{3866, 0.9582912694284986},
		{3867, 1.746991304777294},
		{3868, -0.3282550495014688},
		{3869, 4.517314467064612},
		{3870, 1.9035633193764503},
		{3871, -5.034729709568222},
		{3872, -2.144342084452062},
		{3873, -4.5434323724342685},
		{3874, -1.4159030214120705},
		{3875, -0.8270335898964435},
		{3876, -0.4464097897780446},
		{3877, -0.8609421566557751},
		{3878, -0.46061621759681887},
		{3879, 0.10529118720512318},
		{3880, 0.38192151535376867},
		{3881, 1.0234271202453358},
		{3882, -3.0423480247891983},
		{3883, -0.13319026427254554},
		{3884, -0.052757911920418364},
		{3885, -1.8564293570709383},
		{3886, 1.3879709623689143},
		{3887, 0.7891482874174489},
		{3888, -0.09441436418063598},
		{3889, -1.4771837020972594},
		{3890, -0.8651066056425056},
		{3891, -0.3506582933941955},
		{3892, -2.7089677548850988},
		{3893, -0.0859540137494278},
		{3894, -1.0992781314599054},
		{3895, -0.025789401178551774},
		{3896, 3.7258448360092795},
		{3897, 8.501982660406188},
		{3898, 5.463803678790135},
		{3899, 2.894799137836932},
		{3900, 0.5245509828913806},
		{3901, 0.2339359105669125},
		{3902, 1.845244853656182},
		{3903, 3.6586613642257255},
		{3904, 4.494229512537741},
		{3905, 1.8567131594492843},
		{3906, -0.28480318623368006},
		{3907, 5.4394435899846005},
		{3908, 4.13109098909338},
		{3909, 1.696224497898017},
		{3910, 0.8527638003800077},
		{3911, 0.14248770902152988},
		{3912, 0.5245412640319627},
		{3913, -0.3310254649256237},
		{3914, -1.499861872132789},
		{3915, -0.40603595905101825},
		{3916, 4.347741790761696},
		{3917, 2.665025771220038},
		{3918, 0.867051212853433},
		{3919, -1.5335478960488629},
		{3920, 1.815079328056662},
		{3921, 1.9113271908644371},
		{3922, 4.133438982040487},
		{3923, -1.8110020383262935},
		{3924, 1.1814910247775439},
		{3925, 0.3966463455123105},
		{3926, 1.267243825972283},
		{3927, -0.5934840798398582},
		{3928, -2.150509218935407},
		{3929, 0.11903799138482485},
		{3930, 0.9894118543043009},
		{3931, -2.4705029808605135},
		{3932, -0.3674534785093143},
		{3933, -2.258663059167097},
		{3934, -1.298227865074035},
		{3935, -0.4779775437038967},
		{3936, -2.885687093328418},
		{3937, -1.663071481926528},
		{3938, -0.12926581392816772},
		{3939, 0.08661312917993472},
		{3940, -1.2101709011270756},
		{3941, 0.2569054398492012},
		{3942, -3.023614722629027},
		{3943, -3.4070541067370526},
		{3944, -0.3319321063854075},
		{3945, 0.37849890177954426},
		{3946, 0.14742610678575932},
		{3947, -2.357765788763721},
		{3948, -0.6479487506266413},
		{3949, -1.947147505980847},
		{3950, 1.274314588064696},
		{3951, 0.7460938702941141},
		{3952, -1.052728238704216},
		{3953, -0.49210455374700457},
		{3954, -0.5089012113644384},
		{3955, 0.051742778838548414},
		{3956, -2.718575659455488},
		{3957, -0.7059834566078866},
		{3958, 3.709708060274754},
		{3959, 4.5350980398821505},
		{3960, 2.300768730266061},
		{3961, 2.1964540731907576},
		{3962, -0.6915100849454563},
		{3963, -5.101859714020331},
		{3964, -2.259295182274847},
		{3965, 0.44812009795780916},
		{3966, -0.775750710042171},
		{3967, 1.9018564432122282},
		{3968, 2.711881366695429},
		{3969, 2.3993498155163566},
		{3970, 7.071876383455166},
		{3971, 1.6306733494146701},
		{3972, -3.4666689696938127},
		{3973, 0.05631527624437993},
		{3974, -0.20572225055724644},
		{3975, -1.723578796082768},
		{3976, -1.6655364247033486},
		{3977, 5.731626265125767},
		{3978, 2.608867574104167},
		{3979, 1.070709360849436},
		{3980, 0.18091707247977495},
		{3981, -0.6258481987804936},
		{3982, -0.24046411604600337},
		{3983, -0.5908317197289353},
		{3984, 2.640325294928367},
		{3985, -0.04801721615432952},
		{3986, 3.1735494172420737},
		{3987, 1.4573293549461055},
		{3988, 0.30380800342699754},
		{3989, -0.9246033606228896},
		{3990, -1.9603772720743762},
		{3991, -1.969991707260833},
		{3992, -1.3921050694916082},
		{3993, 3.249875792813785},
		{3994, 1.2679152762389778},
		{3995, 2.783836821273316},
		{3996, 0.6804539520277095},
		{3997, 2.471908314262278},
		{3998, 0.07234615343719464},
		{3999, -0.015566899127408398},
		{4000, -1.8240883616595596},
		{4001, -2.282073515080191},
		{4002, -2.8693368194003575},
		{4003, -1.729903733351549},
		{4004, -0.6358122408358943},
		{4005, -1.9470813817208907},
		{4006, -5.156035703224898},
		{4007, -3.867015615534824},
		{4008, -1.2229241380606968},
		{4009, -0.8755673316340512},
		{4010, -5.593558803826821},
		{4011, -0.7695492492942761},
		{4012, -0.8580939414198058},
		{4013, -0.8141964690698247},
		{4014, -0.4550411503623416},
		{4015, 5.857393619236056},
		{4016, 2.408615609925268},
		{4017, -2.210565947679017},
		{4018, -0.925944615805411},
		{4019, -3.3867772550731226},
		{4020, -2.9470250903456314},
		{4021, -1.1108729845227605},
		{4022, -1.1453599359920261},
		{4023, 2.90255907562918},
		{4024, 1.8274030105654235},
		{4025, -4.468186547307697},
		{4026, 1.0071761153370464},
		{4027, -0.256091238996385},
		{4028, -0.3555391954446541},
		{4029, 0.41459292859274877},
		{4030, 3.1783259919377973},
		{4031, 1.582637149058331},
		{4032, -0.08490256163868182},
		{4033, -0.22998782874798007},
		{4034, -3.517534338304362},
		{4035, -6.919043654660112},
		{4036, -0.6207442810319024},
		{4037, -0.9373386342396219},
		{4038, -0.11675088027683023},
		{4039, 1.5596767809662595},
		{4040, -4.794023219296949},
		{4041, -2.182378328420883},
		{4042, -1.9140789280075325},
		{4043, -0.308073670116254},
		{4044, 1.8306807404194096},
		{4045, -1.677332375426406},
		{4046, -2.0345526446778135},
		{4047, 8.076895259029955},
		{4048, 2.749578648611859},
		{4049, 5.994802310655114},
		{4050, 3.13959876357364},
		{4051, 1.374822673765422},
		{4052, -0.021399229104740347},
		{4053, -3.815941034842902},
		{4054, -3.7092199819338383},
		{4055, -1.2274443510618704},
		{4056, -0.582250907570955},
		{4057, -2.0467943400777617},
		{4058, -5.328769056773165},
		{4059, -2.199914536683484},
		{4060, 0.8513998945839506},
		{4061, -2.0076293008993193},
		{4062, -4.602371365358537},
		{4063, -1.8667034689693431},
		{4064, -0.01143880430433919},
		{4065, -0.6586743542664091},
		{4066, 0.5126112673733887},
		{4067, 0.4554983307307227},
		{4068, -0.4130855904112666},
		{4069, -0.2920727880965147},
		{4070, -3.6596319348482464},
		{4071, -1.6876472660726374},
		{4072, -1.01624030074247},
		{4073, -0.5593104079465473},
		{4074, -1.021218608869819},
		{4075, 0.14724943572573995},
		{4076, 0.5168105958146991},
		{4077, 0.08962116423761896},
		{4078, 3.666578587743252},
		{4079, 0.5585256393384805},
		{4080, -2.673814291005599},
		{4081, -1.6256944867522631},
		{4082, -0.335153163256277},
		{4083, -2.26210821272076},
		{4084, -0.25909934978007443},
		{4085, 1.6277809185831362},
		{4086, 0.5860261750223161},
		{4087, 2.5593201122887637},
		{4088, 3.6632206772663256},
		{4089, -2.7432421443564277},
		{4090, -0.3347229339495924},
		{4091, -0.16733191769389585},
		{4092, -1.1956207129467546},
		{4093, -1.3303934593169067},
		{4094, 0.6375205632906452},
		{4095, 2.3555738143240084},
		{4096, 3.092417686855228},
		{4097, 1.1321672126906694},
		{4098, 1.0875107918160796},
		{4099, 0.12353451960558426},
		{4100, 0.406658819556935},
		{4101, 1.291603385574436},
		{4102, 0.5077222136752811},
		{4103, 0.09002111077513873},
		{4104, 0.20722364317148212},
		{4105, -0.10114591211051856},
		{4106, -3.975290254394115},
		{4107, 1.0571194239916903},
		{4108, 1.4961230974162687},
		{4109, 0.5809378394397109},
		{4110, 1.3917219579762288},
		{4111, 0.44497036201916185},
		{4112, 2.5489999251725055},
		{4113, -1.5378880621178206},
		{4114, 0.0057082995271172665},
		{4115, -0.7282227050795992},
		{4116, -0.018523801070291113},
		{4117, -2.222979567257638},
		{4118, -2.12385299267095},
		{4119, -0.7331963606497975},
		{4120, -0.29353720476761025},
		{4121, 2.8778830217801046},
		{4122, 0.07399324463663981},
		{4123, 0.2834896461594688},
		{4124, 0.5872067970181483},
		{4125, 0.24809573613508132},
		{4126, 4.755865329227087},
		{4127, 2.946992163146769},
		{4128, -0.5292548535403847},
		{4129, 1.8033337125133275},
		{4130, 0.1565619384973671},
		{4131, 2.7033442093023905},
		{4132, 0.760382432683696},
		{4133, 1.5713357666791916},
		{4134, 3.7866673730342417},
		{4135, 1.6785945303591159},
		{4136, 3.227986256224302},
		{4137, 0.8395730463488306},
		{4138, 2.0571991607562863},
		{4139, 1.5701221261782727},
		{4140, -0.4827908417024569},
		{4141, 0.24033488402394204},
		{4142, 2.7047667435333556},
		{4143, -0.3063629255305016},
		{4144, 0.9189445745183489},
		{4145, 1.3860124050071232},
		{4146, 0.12516910755581961},
		{4147, -3.9644957843349045},
		{4148, -0.5746893388215171},
		{4149, -0.26482325895960845},
		{4150, -1.176471517129221},
		{4151, 3.763394839221223},
		{4152, 1.1859474461344748},
		{4153, 1.5607732003184946},
		{4154, 0.5645393605980283},
		{4155, -2.4319959467482257},
		{4156, 0.735542103278018},
		{4157, 0.8260026931171969},
		{4158, 0.5522829817526795},
		{4159, -0.13881404367777836},
		{4160, -1.2696298409862525},
		{4161, -0.776817037202012},
		{4162, 3.460033892108722},
		{4163, 6.346017286862609},
		{4164, 2.8970696928306054},
		{4165, 0.871110264150726},
		{4166, 0.6303462615053403},
		{4167, 2.8054791425300696},
		{4168, 3.255638725785135},
		{4169, 3.1663549719731794},
		{4170, 0.7989019366961099},
		{4171, 4.333424658540002},
		{4172, 1.5884465013726772},
		{4173, 0.08666369573054633},
		{4174, 2.077431941592718},
		{4175, -0.7028012893561617},
		{4176, -1.4067986827313894},
		{4177, -2.3810090979635117},
		{4178, -3.9719713593215715},
		{4179, -2.4640447158715677},
		{4180, -3.8475343194317437},
		{4181, -0.7383606524860985},
		{4182, -3.938447869668466},
		{4183, -2.071896788966474},
		{4184, -1.101315669612277},
		{4185, -0.4406158034704644},
		{4186, 2.376053343181054},
		{4187, -1.8733436472668474},
		{4188, 1.4902025634222253},
		{4189, 0.6295845879562204},
		{4190, 0.6443232084688073},
		{4191, -4.1626928250973165},
		{4192, 0.9744698905956357},
		{4193, -1.1040774424052024},
		{4194, 0.15306818923440435},
		{4195, -0.1739879516506123},
		{4196, -0.5653210032820589},
		{4197, -1.67142039909384},
		{4198, 0.41416716505916296},
		{4199, 2.112423399816226},
		{4200, 0.8841872652341336},
		{4201, -2.6260122945212467},
		{4202, 1.508592791616904},
		{4203, -2.302877113128602},
		{4204, 1.5850788698928753},
		{4205, 3.296945248936624},
		{4206, 0.4487631301866849},
		{4207, -3.575136876782827},
		{4208, -2.984155379854266},
		{4209, -1.1147829680799552},
		{4210, -0.74361664322346},
		{4211, -0.7435824305434449},
		{4212, -2.1048992800970394},
		{4213, -2.900604569814485},
		{4214, -2.168895349780052},
		{4215, -0.8381038296582247},
		{4216, 0.7472629024783604},
		{4217, 0.5470226264117923},
		{4218, 3.9409174202678594},
		{4219, 2.6508957977493726},
		{4220, 0.9910793555613732},
		{4221, 0.4889560686733936},
		{4222, -2.6605999233266266},
		{4223, -1.1371722590595297},
		{4224, -0.12032404483012904},
		{4225, 0.5922995638866825},
		{4226, 3.1600622631524806},
		{4227, 1.4115893973412503},
		{4228, 0.4617034989857701},
		{4229, 0.6251330318201022},
		{4230, 0.444317109194026},
		{4231, 0.09202538670577946},
		{4232, -0.9048740271006598},
		{4233, 1.684249022971444},
		{4234, 3.219145272371575},
		{4235, 0.968810202061136},
		{4236, -0.18661135038501386},
		{4237, -4.525329491251166},
		{4238, -1.5360548670343257},
		{4239, 4.0463501937297774},
		{4240, 0.8280346890086904},
		{4241, 2.16131653252869},
		{4242, 0.5891384494532454},
		{4243, 1.9689747435729763},
		{4244, 0.8537255998228777},
		{4245, 0.31227777240697036},
		{4246, 2.566311391781835},
		{4247, 0.25166816297036587},
		{4248, 0.13645073055490686},
		{4249, -0.8737308778681316},
		{4250, 0.44529225057506755},
		{4251, 1.3724123878203303},
		{4252, 1.0731050674691502},
		{4253, 0.3773163144207022},
		{4254, 2.8427804775291805},
		{4255, 1.5428670860674225},
		{4256, 1.3634123574622943},
		{4257, -2.4626947786142677},
		{4258, -1.3063705993740828},
		{4259, 0.6726916799803032},
		{4260, 2.9285274298689403},
		{4261, 1.181038380913006},
		{4262, -0.15212037720128063},
		{4263, -0.5632230378096998},
		{4264, -2.181923267393287},
		{4265, -2.1455133091735474},
		{4266, -1.9595297139655994},
		{4267, 2.019956268156402},
		{4268, -3.9280249421022893},
		{4269, 0.0056047764939819444},
		{4270, 3.9494880156969656},
		{4271, -0.21111312474970312},
		{4272, -1.0460611954421413},
		{4273, -1.8663766484425388},
		{4274, -0.48102386662525604},
		{4275, 0.9743508266573128},
		{4276, 3.0756000759984587},
		{4277, 8.930330818057417},
		{4278, 10.724030560757402},
		{4279, 5.715098334382074},
		{4280, 1.3902566895212076},
		{4281, 2.4359860114494087},
		{4282, 0.07208280440138148},
		{4283, 1.4826982550338428},
		{4284, 4.171497638572084},
		{4285, 1.504543930203364},
		{4286, 2.203452858669924},
		{4287, -0.15432860562580042},
		{4288, 1.4106683666663138},
		{4289, 0.7280799661963546},
		{4290, -2.9369902493894},
		{4291, -1.169260936076521},
		{4292, 1.8180135663851178},
		{4293, 10.476036772936595},
		{4294, 3.610809121844307},
		{4295, 1.6715507477347478},
		{4296, 0.7844161813947715},
		{4297, 2.195189430513417},
		{4298, 0.9040474819734833},
		{4299, 0.9119335830721167},
		{4300, -2.808641760458209},
		{4301, -0.7162164192398612},
		{4302, -3.5566387623810267},
		{4303, -1.7600933574895234},
		{4304, -5.707988669873386},
		{4305, -2.2779427188066115},
		{4306, 0.18719153845883962},
		{4307, 0.23762465158330873},
		{4308, -0.04869625804953623},
		{4309, -1.8126565117290667},
		{4310, -3.1088562061577147},
		{4311, -2.361526854975107},
		{4312, -0.07240090157902657},
		{4313, 1.617408793629731},
		{4314, 0.7215390687996323},
		{4315, -1.7879624640644702},
		{4316, -0.673655535665817},
		{4317, 1.497069207598817},
		{4318, 1.1344300973144823},
		{4319, 0.07607107154896792},
		{4320, -2.6683334328644275},
		{4321, -3.674759138069418},
		{4322, -1.4231725347891266},
		{4323, 0.9665011134470454},
		{4324, 1.5038928686983677},
		{4325, -0.6745012983347918},
		{4326, -0.8706925206483735},
		{4327, -2.389984789421191},
		{4328, -0.6424824055658296},
		{4329, -0.9884420105319683},
		{4330, -0.5037985797554977},
		{4331, -0.5315673067051389},
		{4332, -4.344679581715628},
		{4333, 0.29530174559629785},
		{4334, -1.2005569939171372},
		{4335, 1.2308917585422505},
		{4336, -2.9108752594362217},
		{4337, -6.820144796160701},
		{4338, -3.046761908870466},
		{4339, -1.6561425727573658},
		{4340, -2.7966046864746463},
		{4341, 0.13725605567049337},
		{4342, 1.1811760771793736},
		{4343, 1.0462995896360667},
		{4344, 2.6856249295345505},
		{4345, -1.936510360648504},
		{4346, 0.760042612465019},
		{4347, -1.6626600857558154},
		{4348, -1.3131284392763654},
		{4349, -1.355150570724133},
		{4350, -0.1747401302844846},
		{4351, 0.18668135300973382},
		{4352, 2.369597550490902},
		{4353, -4.539777121306794},
		{4354, 1.0855165957075767},
		{4355, 0.1607188961927749},
		{4356, -1.0804640925391273},
		{4357, 1.768798029847328},
		{4358, 0.10621187060840709},
		{4359, 1.0047661360686655},
		{4360, 1.1334217242763538},
		{4361, 2.8435832747029473},
		{4362, 3.7556090322271185},
		{4363, 1.565685967510826},
		{4364, 1.413048350900734},
		{4365, 2.172482115660905},
		{4366, 1.4838064930520893},
		{4367, 2.478375932415275},
		{4368, -0.4526187422013995},
		{4369, -0.6721699292152334},
		{4370, -0.5189790658200166},
		{4371, 0.4991765645406925},
		{4372, 0.20855086301840858},
		{4373, 0.04043925516571027},
		{4374, -0.09234238507356421},
		{4375, 0.16739840596970965},
		{4376, 2.1676935920652247},
		{4377, -6.667480475082275},
		{4378, -2.992016172949934},
		{4379, -2.389635321431008},
		{4380, -0.3029411166114785},
		{4381, -0.12438452439788496},
		{4382, 0.3739873101077285},
		{4383, 1.8337780690793204},
		{4384, 1.6231811951588733},
		{4385, -0.08549599226953508},
		{4386, 2.4196079983403362},
		{4387, 3.305779499939982},
		{4388, -0.19996826390670952},
		{4389, 3.727474459523985},
		{4390, 1.2022233963125264},
		{4391, -0.4248721972353111},
		{4392, 5.7914949526091135},
		{4393, 2.231836966720819},
		{4394, -6.178315817030297},
		{4395, -2.7839694641012374},
		{4396, 2.6550580322087267},
		{4397, 1.203378148170068},
		{4398, -2.345092255964026},
		{4399, 0.34970834932896255},
		{4400, -1.1649703361035426},
		{4401, 2.0498069212568044},
		{4402, 0.6041341716097318},
		{4403, -0.3841664778404307},
		{4404, -0.9510993984530962},
		{4405, 5.121933838962404},
		{4406, 1.8431142708843449},
		{4407, 0.2662996598532498},
		{4408, 0.85146581495856},
		{4409, 0.3727715876656369},
		{4410, 0.5295979154537124},
		{4411, -0.226616507878515},
		{4412, 7.346299135337522},
		{4413, 7.064469183424043},
		{4414, 3.870882926054535},
		{4415, 1.5141463067033745},
		{4416, 0.9038985739685654},
		{4417, 0.45401136936009673},
		{4418, 1.6334606968378198},
		{4419, -0.20746347844570823},
		{4420, -4.275742485443478},
		{4421, -4.15181898786355},
		{4422, -1.666681169030956},
		{4423, -2.01472469857058},
		{4424, 0.0954201619972932},
		{4425, -4.530559172548947},
		{4426, 2.4541762462398378},
		{4427, 0.4903887910286054},
		{4428, 0.34994484919954594},
		{4429, 0.7460174288887461},
		{4430, -0.1426705741385349},
		{4431, -1.1357890031481415},
		{4432, 0.5317006179299856},
		{4433, 0.024409147929956437},
		{4434, 0.8453288188515287},
		{4435, 3.107199329306928},
		{4436, 0.7024053204332132},
		{4437, 0.6862463908178116},
		{4438, 2.2643706404599215},
		{4439, -1.8936970095136325},
		{4440, 1.7563900212115469},
		{4441, 1.9339790580000469},
		{4442, -2.634799554516289},
		{4443, 2.7094795000805947},
		{4444, 2.3785823716061407},
		{4445, 1.958712908707469},
		{4446, -1.7437311689300459},
		{4447, -3.2665462741348352},
		{4448, -9.094197277422168},
		{4449, -4.757163678168436},
		{4450, 2.455147283264317},
		{4451, 8.010608586202348},
		{4452, 2.4560472483808056},
		{4453, 0.7851166476874631},
		{4454, -0.9543486858967865},
		{4455, -5.294219650391986},
		{4456, -2.840044006991233},
		{4457, -3.560127059373878},
		{4458, -1.9939179269009824},
		{4459, 1.925930687705752},
		{4460, -2.587251381360806},
		{4461, -2.1697177581299605},
		{4462, -1.2001901171194915},
		{4463, -0.6654790825819603},
		{4464, -0.10681295517180212},
		{4465, -3.0335949422102284},
		{4466, 4.2705894739018815},
		{4467, 1.1265602474453198},
		{4468, -0.46322501124433335},
		{4469, 0.6084871259647298},
		{4470, -0.007314307360012706},
		{4471, 1.9906328238128792},
		{4472, 3.949461672000644},
		{4473, 3.792382311907316},
		{4474, 4.2332852699404775},
		{4475, 2.2206676268497083},
		{4476, 1.6272380144900214},
		{4477, 1.5672440782423647},
		{4478, -0.37970690840253396},
		{4479, 6.703495044604546},
		{4480, 2.4777285721997915},
		{4481, -2.694276178406354},
		{4482, -0.37348888922708856},
		{4483, -1.8741194560749652},
		{4484, -1.0925729966423758},
		{4485, 0.06823251014929799},
		{4486, -0.13029595133982985},
		{4487, -0.045444475770359025},
		{4488, -2.716380229546698},
		{4489, 0.9337297206642796},
		{4490, 3.978261546893857},
		{4491, 1.6006520695724036},
		{4492, -0.6009200838218196},
		{4493, -0.6367546511742368},
		{4494, -1.3642481657991867},
		{4495, -5.250592258299262},
		{4496, -2.956590806734569},
		{4497, -1.297046727706237},
		{4498, -0.8215856266030106},
		{4499, -0.6097452016687026},
		{4500, -0.9996524685674063},
		{4501, -1.3175351791255987},
		{4502, -0.5498093695693438},
		{4503, 4.948753709818669},
		{4504, 0.9356118084755916},
		{4505, -0.019108773786087607},
		{4506, 1.2179384828674626},
		{4507, 1.2592065169601598},
		{4508, -1.1900004738388792},
		{4509, -1.227104678672239},
		{4510, -1.3553180290176987},
		{4511, -0.36559859384607},
		{4512, -0.6493126055185777},
		{4513, 0.7747005861753706},
		{4514, -0.2846366515233368},
		{4515, 4.3136864881009975},
		{4516, 0.6951680459533383},
		{4517, 0.35497744430756367},
		{4518, -1.6006606723109393},
		{4519, 7.397683690530439},
		{4520, 3.690511814720172},
		{4521, 0.9687244117980266},
		{4522, -0.8126645453998025},
		{4523, -0.43725255102451105},
		{4524, 1.3205732424089855},
		{4525, 2.4134051956560723},
		{4526, 1.1989035923708666},
		{4527, 0.8174057718435397},
		{4528, -4.794644343367631},
		{4529, -3.095224252694956},
		{4530, -3.0257512250655783},
		{4531, -2.8773352758072877},
		{4532, -1.438618037151039},
		{4533, -0.5614381504755518},
		{4534, 0.24628287143312655},
		{4535, 3.588034506097846},
		{4536, 2.0238533069354143},
		{4537, 1.2101054554072723},
		{4538, 4.888295529168028},
		{4539, 0.586304967638011},
		{4540, -0.19631058156701467},
		{4541, -1.270134194916562},
		{4542, 1.8174339711391436},
		{4543, 1.6487508531598043},
		{4544, -0.006734569016869751},
		{4545, 0.39767807657588394},
		{4546, -0.36017487888610367},
		{4547, -3.489314500475214},
		{4548, -3.0193919510501113},
		{4549, -0.7216726919493521},
		{4550, 1.1324991517268865},
		{4551, 1.272291554701719},
		{4552, -3.8283405349246555},
		{4553, -7.490687754534086},
		{4554, -8.100999558699858},
		{4555, -4.4477726969253375},
		{4556, 1.9568200817498747},
		{4557, 3.8433778399515917},
		{4558, -3.3821323214154697},
		{4559, -1.5083212504152927},
		{4560, -2.807051791401827},
		{4561, -5.798313704138256},
		{4562, -2.454702089972893},
		{4563, -1.0304553473952451},
		{4564, 4.939088709800628},
		{4565, -2.212148249660287},
		{4566, 2.7004565629939283},
		{4567, 0.4412813704372146},
		{4568, 1.1833258908350537},
		{4569, -0.9612585607464537},
		{4570, -0.7689715395984924},
		{4571, -0.7194476278639234},
		{4572, -0.2996623431633636},
		{4573, -1.864340890262391},
		{4574, -2.7502782120729377},
		{4575, -1.0115589569868149},
		{4576, -1.355516253907687},
		{4577, -0.6776207898842496},
		{4578, 6.674441071597661},
		{4579, 1.5924123970755921},
		{4580, 0.9378031411451044},
		{4581, -3.6005892628607614},
		{4582, -1.5399453265159382},
		{4583, -2.6354107375740314},
		{4584, -0.9750942148407489},
		{4585, -5.71868422917731},
		{4586, -7.247651437403871},
		{4587, -2.1705366496810994},
		{4588, -0.9255520790657801},
		{4589, 1.0834770828922653},
		{4590, 0.1963752612685663},
		{4591, 0.04197461429001619},
		{4592, -0.16604438538856928},
		{4593, -1.2472828356790808},
		{4594, 2.9502331798694614},
		{4595, -0.9488619468610553},
		{4596, -0.2385769646480507},
		{4597, -0.8466486080087486},
		{4598, -0.5121024761547124},
		{4599, 2.525630308124496},
		{4600, 2.3312536957852905},
		{4601, 1.4655485087667866},
		{4602, 2.950662862754819},
		{4603, 1.4895901239007938},
		{4604, -0.42183041220642326},
		{4605, -0.37558407383724235},
		{4606, 0.27811807366842906},
		{4607, 0.7878133659651422},
		{4608, 0.19564593257660678},
		{4609, 0.5782104528352848},
		{4610, 2.399435083320623},
		{4611, 1.319709807854943},
		{4612, -0.8067365662068473},
		{4613, 0.7992451551840142},
		{4614, 1.0664545666467848},
		{4615, 0.5751098397288027},
		{4616, 2.0192802416530324},
		{4617, 0.4827167711666865},
		{4618, 0.09486042789500534},
		{4619, 0.6102991292577258},
		{4620, 1.3992411996034553},
		{4621, -0.3115845268763554},
		{4622, -0.08319237752024729},
		{4623, 1.6123210756148714},
		{4624, 3.7991902581116266},
		{4625, 0.707378256410034},
		{4626, 0.38355370447307646},
		{4627, 0.5711966764053436},
		{4628, -2.602546764050603},
		{4629, -1.34623457856895},
		{4630, 0.5402289114931638},
		{4631, -1.9981750290031006},
		{4632, -2.824903056248726},
		{4633, -1.2645142428111962},
		{4634, -5.474426075301093},
		{4635, -2.523028856198459},
		{4636, -0.2816236362539698},
		{4637, -0.25621706657614707},
		{4638, -0.10872999894088639},
		{4639, -0.8079026865669083},
		{4640, 0.13679013782134591},
		{4641, -1.3636775922561384},
		{4642, -2.3168121385221943},
		{4643, -1.1352123198230877},
		{4644, -2.500972997914568},
		{4645, -2.6164744032913765},
		{4646, -5.554592483059352},
		{4647, 0.39547923023483733},
		{4648, 0.854287577353447},
		{4649, -1.0869642898196856},
		{4650, -2.7047864053796022},
		{4651, 2.251259112777023},
		{4652, 9.407346665506479},
		{4653, 3.7533858183442965},
		{4654, 3.1925061067103364},
		{4655, 4.4617202143886665},
		{4656, 2.212014980276105},
		{4657, -0.060131664785403216},
		{4658, 7.642392039133967},
		{4659, 2.542004547918301},
		{4660, 1.1279765433428888},
		{4661, 1.6212497899449543},
		{4662, -2.770798840472041},
		{4663, 4.145827203352061},
		{4664, 1.0564188551677995},
		{4665, 0.8525889496586063},
		{4666, -4.827326408454258},
		{4667, -2.940481981870345},
		{4668, 5.052921405457672},
		{4669, 4.9026055308903205},
		{4670, 2.0871961878677845},
		{4671, -0.17333089102320942},
		{4672, 0.2522468545679819},
		{4673, 1.5443280774489654},
		{4674, 1.9603013644294278},
		{4675, 1.0674642774175453},
		{4676, 0.5379180622274371},
		{4677, -1.140154227866732},
		{4678, -1.7394671682091374},
		{4679, -0.9981841962631384},
		{4680, 3.712308997412607},
		{4681, 0.7992475259645505},
		{4682, 1.0418822039238274},
		{4683, -0.3603681955613036},
		{4684, -0.1978522326519181},
		{4685, 1.2349888447349793},
		{4686, 1.0646508970910555},
		{4687, -1.7911150702372909},
		{4688, -0.38591941485836473},
		{4689, -4.108856353434841},
		{4690, 1.1236845121746355},
		{4691, -0.18280218448030783},
		{4692, -6.69523066094518},
		{4693, -3.3085375193526727},
		{4694, -1.5283187974669863},
		{4695, -1.3533397829581384},
		{4696, 1.0733006802059244},
		{4697, -4.3883760664790525},
		{4698, -3.8954350438250187},
		{4699, 0.8254707501828615},
		{4700, 3.1297247604032674},
		{4701, 2.5388103456307745},
		{4702, -0.14945565902111446},
		{4703, 0.055781455679992216},
		{4704, -0.40515124719241696},
		{4705, 2.170691566472983},
		{4706, -2.0279610089095885},
		{4707, 3.229312303186303},
		{4708, -0.43101841871435687},
		{4709, -0.19118126645564018},
		{4710, 0.30411036439430467},
		{4711, -0.2688548061606313},
		{4712, -0.6752139661384496},
		{4713, -0.5041384836574827},
		{4714, 2.367082843621476},
		{4715, -0.783182373031726},
		{4716, -0.18944147644329762},
		{4717, -1.9586092294033044},
		{4718, 2.1620681570455225},
		{4719, -0.16141600798067746},
		{4720, 3.2075776859801413},
		{4721, 1.0183533685799957},
		{4722, -1.7080687421326803},
		{4723, 0.18440719807808925},
		{4724, 5.712998632364829},
		{4725, 2.369161817602541},
		{4726, 0.37156117842568737},
		{4727, 1.858710733625652},
		{4728, 5.514065605944318},
		{4729, 2.5083543114013693},
		{4730, 2.4136816846208102},
		{4731, 1.1277988308955025},
		{4732, -0.3149152221707343},
		{4733, 0.26392682192513506},
		{4734, -1.7257447945643514},
		{4735, 2.966386741890815},
		{4736, 0.9742725896024123},
		{4737, -0.5149122531467037},
		{4738, 0.11751676845141168},
		{4739, -1.695596948138374},
		{4740, 2.596856657089344},
		{4741, 3.3571905598677363},
		{4742, 1.6188311057313367},
		{4743, -0.8439886715986941},
		{4744, 3.02318829049801},
		{4745, -3.77577881323399},
		{4746, -3.7463694518238735},
		{4747, 1.2432267950430334},
		{4748, -4.692695417878392},
		{4749, -2.390857284634945},
		{4750, -4.34577098334832},
		{4751, 1.3102528075552806},
		{4752, 0.2447351567985186},
		{4753, 0.09001265630962721},
		{4754, 0.1827544337388772},
		{4755, 2.4445638937790837},
		{4756, 1.3162538371686088},
		{4757, 1.831962578289522},
		{4758, 3.2015148461749185},
		{4759, 5.204633743199489},
		{4760, 5.394010917996627},
		{4761, 2.928167760516333},
		{4762, 1.9772469641601984},
		{4763, -0.6166951595066459},
		{4764, -1.1676893014751861},
		{4765, 0.44121398940602746},
		{4766, 1.3890050094059294},
		{4767, 1.246121041656619},
		{4768, 4.858178223164077},
		{4769, -1.6470310461483908},
		{4770, -0.6221357550161467},
		{4771, -0.23825194575433123},
		{4772, 0.3657858669562647},
		{4773, 0.49786995458604616},
		{4774, 2.2940697403302943},
		{4775, 2.355475795878255},
		{4776, 0.8579398554962228},
		{4777, -1.0989638934569577},
		{4778, 0.23010264057912538},
		{4779, 0.27227004031748125},
		{4780, 5.84931276015174},
		{4781, 2.3944319566017933},
		{4782, 1.119431326261401},
		{4783, 2.575287068171496},
		{4784, -0.1329566764692025},
		{4785, -2.4132430061506644},
		{4786, -2.0877092568151987},
		{4787, -1.8512771494368312},
		{4788, 1.4439530468253723},
		{4789, 0.9377375292594714},
		{4790, 0.6842835330949164},
		{4791, 2.533201350364261},
		{4792, 2.0703283330730313},
		{4793, 0.01024859421887514},
		{4794, 2.2148427094191447},
		{4795, 3.310380568545128},
		{4796, 3.181256534502562},
		{4797, -0.8288426548243919},
		{4798, -3.5763170156434265},
		{4799, -6.152565225010233},
		{4800, -1.939030005717324},
		{4801, -1.0626647908906293},
		{4802, -0.6336369061379366},
		{4803, -1.282111075788885},
		{4804, 0.11780356386210622},
		{4805, -1.8538412312522439},
		{4806, -4.329150563557341},
		{4807, 0.6466910151562291},
		{4808, 2.654330103308584},
		{4809, 0.5771757567196507},
		{4810, 0.8648281485144467},
		{4811, 0.7311973440575287},
		{4812, 1.734095330246719},
		{4813, 0.07202498543618185},
		{4814, -1.9790439068656012},
		{4815, -3.021090671729249},
		{4816, -0.1008065952326671},
		{4817, -0.32176473697901575},
		{4818, -0.15322689408646292},
		{4819, -0.3906749618873405},
		{4820, -1.06930474231111},
		{4821, -2.685543621657825},
		{4822, -0.5639649369974775},
		{4823, -1.2718355663720502},
		{4824, -1.4295259096313646},
		{4825, -1.653421663502435},
		{4826, -0.7942386249588679},
		{4827, -0.1135225087602709},
		{4828, 0.6174683389123125},
		{4829, 1.7776891185739003},
		{4830, 0.49368403611360856},
		{4831, -0.16216773180079563},
		{4832, 0.4171480827197546},
		{4833, 2.7430312522104074},
		{4834, 0.1378863999781319},
		{4835, 0.9257715128192511},
		{4836, 0.8949090567067058},
		{4837, -2.120857886730965},
		{4838, 0.06063826717477927},
		{4839, 1.316879237377766},
		{4840, 0.17235219008946806},
		{4841, 2.7186016852551456},
		{4842, -2.629341559278801},
		{4843, -1.39819769693825},
		{4844, 0.0505392580151206},
		{4845, 0.6975583765889697},
		{4846, 5.282296720565897},
		{4847, 2.524734163972368},
		{4848, 0.8550409750927},
		{4849, 3.212299702844635},
		{4850, 3.8318102503596174},
		{4851, 0.2859544341989202},
		{4852, 4.0097953482453645},
		{4853, 2.202704095183943},
		{4854, 3.173155137285125},
		{4855, 1.1387763272081264},
		{4856, -2.239103295319526},
		{4857, -1.2475069044327225},
		{4858, -0.9246837996846835},
		{4859, -5.484369447376061},
		{4860, -0.393333650840721},
		{4861, -1.3700612843074804},
		{4862, -0.6045267717963987},
		{4863, -0.7745651278216691},
		{4864, -0.11804197831945898},
		{4865, 2.642215055609161},
		{4866, 1.5102066564321164},
		{4867, 3.1329272817355287},
		{4868, 1.0102304695554887},
		{4869, 0.13167505092598614},
		{4870, 1.3996231838556024},
		{4871, 0.02956656701826854},
		{4872, 1.2254405765926737},
		{4873, 0.22281937232316323},
		{4874, -0.041726769815379},
		{4875, 0.15632396878903487},
		{4876, 0.959413538322629},
		{4877, 3.557091410889755},
		{4878, 2.6081666938052805},
		{4879, 1.351610047291514},
		{4880, 0.15053566419989195},
		{4881, 0.11947001088780972},
		{4882, -0.0004862823174552125},
		{4883, 4.0571954468096525},
		{4884, 3.0220517485785505},
		{4885, 0.24199187333548378},
		{4886, -0.8664819296012896},
		{4887, -3.1081870825997773},
		{4888, 3.954294143776898},
		{4889, -2.085693226576981},
		{4890, -2.668399127624096},
		{4891, 0.7490119537821371},
		{4892, -0.5920788817462851},
		{4893, -0.32293918264878235},
		{4894, 2.966014996547112},
		{4895, -8.02630092740655},
		{4896, -0.5166625595792476},
		{4897, -7.894731927175839},
		{4898, -2.628765094978702},
		{4899, -0.3568486552406481},
		{4900, -0.003672173201065082},
		{4901, 1.1309450310000193},
		{4902, 0.8870196938718216},
		{4903, 0.20440220428953804},
		{4904, 0.5794337626839388},
		{4905, -0.506783949813211},
		{4906, -1.318891870607134},
		{4907, -3.514057687016755},
		{4908, 0.3102012340399152},
		{4909, 0.039507125792953776},
		{4910, 0.22216100107841033},
		{4911, 0.10800575553808826},
		{4912, 0.35540034022195294},
		{4913, -0.5143566974653374},
		{4914, 0.3764559281112929},
		{4915, -0.21899706171164457},
		{4916, -2.392159371871195},
		{4917, -1.3692465375673415},
		{4918, 0.01754552931307507},
		{4919, 1.3153201818471942},
		{4920, 2.8718991134470455},
		{4921, 4.371385473896687},
		{4922, 0.7267615141439405},
		{4923, -0.7866827418848996},
		{4924, -0.7674138010538658},
		{4925, -1.2946003994431583},
		{4926, 1.6156891757529985},
		{4927, 1.6514183375970304},
		{4928, -0.20163984525959422},
		{4929, 1.6331147261277983},
		{4930, -0.4263471213757257},
		{4931, -0.10499912536710572},
		{4932, 1.9572672751122535},
		{4933, 6.114137943176568},
		{4934, 0.9204473513949778},
		{4935, 0.3130311603277955},
		{4936, 3.7178148278984695},
		{4937, -0.22976290869503702},
		{4938, -1.402445923610105},
		{4939, -3.263251413159843},
		{4940, -0.9137779659371708},
		{4941, -0.9408093602715949},
		{4942, -0.8093049071420847},
		{4943, -2.5648140570362923},
		{4944, -0.9478816164738884},
		{4945, -0.8588358454671889},
		{4946, -0.9048495065466595},
		{4947, 3.153392290182028},
		{4948, 1.2713634918137844},
		{4949, -0.012874348709935646},
		{4950, -0.3528948587211178},
		{4951, -1.6302416750492448},
		{4952, -1.9195611378384263},
		{4953, -1.655199232857213},
		{4954, 2.116591473668242},
		{4955, -0.18584337035164888},
		{4956, -2.8449029712278135},
		{4957, -3.644379518699819},
		{4958, -3.1518295822604734},
		{4959, -4.377783874045203},
		{4960, -1.7447904428964078},
		{4961, -0.7344264428039533},
		{4962, -3.0125624023628257},
		{4963, 0.7562292436053228},
		{4964, 0.2729120218260934},
		{4965, -2.305969625992171},
		{4966, -1.705580968400545},
		{4967, -0.5204860682562008},
		{4968, 1.8737431282723342},
		{4969, 0.22892178886606063},
		{4970, 0.10639281807191012},
		{4971, 1.7053340293551373},
		{4972, 0.590741694960817},
		{4973, 6.799049981047992},
		{4974, -1.9163018970939785},
		{4975, -0.7217509410867016},
		{4976, -3.024730046486979},
		{4977, 4.233341079131715},
		{4978, 4.160135282015402},
		{4979, -1.0094104100937795},
		{4980, 0.3551984443515547},
		{4981, 0.23553183348294526},
		{4982, 3.2678247766479243},
		{4983, -1.0437988712409358},
		{4984, -0.4553981016014734},
		{4985, -1.8941782284414685},
		{4986, -0.8390873207157827},
		{4987, -0.514539915419365},
		{4988, -0.7011793330779448},
		{4989, 2.350413669895398},
		{4990, 0.5046792457656368},
		{4991, -0.413771582203163},
		{4992, -1.3533624386611818},
		{4993, -0.49031243398209756},
		{4994, -0.4858191206009566},
		{4995, -0.2777943061752596},
		{4996, -0.5587905516604237},
		{4997, 2.8479512294406004},
		{4998, 7.167016287499241},
		{4999, 3.616596050725063},
	},
	{
		{0, 100.0},
		{316224, 106.3849},
		{320544, 106.3907},
		{321408, 106.3862},
		{322272, 106.3901},
		{323136, 106.3802},
		{325728, 106.3839},
		{326592, 106.3894},
		{327456, 106.389},
		{328320, 106.3851},
		{329184, 106.3932},
		{331776, 106.3882},
		{332640, 106.392},
		{333504, 106.3803},
		{334368, 106.3937},
		{335232, 106.3785},
		{337824, 106.3874},
		{338688, 106.3802},
		{339552, 106.3851},
		{340416, 106.3842},
		{341280, 106.3906},
		{343872, 106.3854},
		{344736, 106.3844},
		{345600, 106.3861},
		{346464, 106.3844},
		{347328, 106.3803},
		{349920, 106.3806},
		{350784, 106.3919},
		{351648, 106.3871},
		{352512, 106.388},
		{353376, 106.3778},
		{355968, 106.3788},
		{356832, 106.3853},
		{357696, 106.3935},
		{358560, 106.3887},
		{359424, 106.3908},
		{362016, 106.3815},
		{362880, 106.3796},
		{363744, 106.3888},
		{364608, 106.3794},
		{365472, 106.386},
		{368064, 106.3785},
		{368928, 106.3844},
		{369792, 106.3877},
		{370656, 106.3843},
		{371520, 106.3883},
		{374112, 106.3927},
		{374976, 106.3836},
		{375840, 106.3796},
		{376704, 106.3856},
		{377568, 106.3856},
		{380160, 106.3847},
		{381024, 106.3901},
		{381888, 106.3801},
		{382752, 106.3909},
		{383616, 106.3837},
		{386208, 106.3781},
		{387072, 106.3928},
		{387936, 106.3815},
		{388800, 106.3944},
		{389664, 106.3907},
		{392256, 106.389},
		{393120, 106.3815},
		{393984, 106.3796},
		{394848, 106.3813},
		{395712, 106.3935},
		{398304, 106.3912},
		{399168, 106.3785},
		{400032, 106.3827},
		{405216, 106.3855},
		{406080, 106.387},
		{406944, 106.3861},
		{407808, 106.3941},
		{410400, 106.3927},
		{411264, 106.3787},
		{412128, 106.3939},
		{413856, 106.381},
		{416448, 106.3867},
		{417312, 106.3921},
		{418176, 106.3923},
		{419040, 106.3845},
		{419904, 106.3887},
		{422496, 106.396},
		{423360, 106.3845},
		{424224, 106.3784},
		{425088, 106.3945},
		{425952, 106.3858},
		{428544, 106.3924},
		{429408, 106.3823},
		{430272, 106.3878},
		{431136, 106.3795},
		{432000, 106.3935},
		{434592, 106.3869},
		{435456, 106.3848},
		{436320, 106.3885},
		{438048, 106.3891},
		{440640, 106.3806},
		{441504, 106.38},
		{442368, 106.3803},
		{443232, 106.3897},
		{444096, 106.3705},
		{447552, 106.395},
		{448416, 106.4012},
		{449280, 106.3701},
		{450144, 106.3533},
		{452736, 106.3638},
		{453600, 106.3825},
		{454464, 106.3858},
		{455328, 106.3476},
		{456192, 106.3301},
		{458784, 106.3473},
		{459648, 106.3326},
		{460512, 106.3362},
		{462240, 106.3396},
		{464832, 106.3397},
		{465696, 106.3184},
		{466560, 106.3229},
		{467424, 106.3252},
		{468288, 106.3372},
		{470880, 114.9774},
		{471744, 114.9845},
		{472608, 114.9456},
		{473472, 114.9321},
		{474336, 114.9526},
		{476928, 114.9344},
		{477792, 114.9354},
		{478656, 114.9263},
		{479520, 114.9323},
		{480384, 114.9177},
		{482976, 114.8979},
		{483840, 114.909},
		{484704, 114.908},
		{485568, 114.9051},
		{486432, 114.8746},
		{489024, 114.8899},
		{489888, 114.8554},
		{490752, 114.8666},
		{491616, 114.8731},
		{492480, 114.8176},
		{495072, 114.7947},
		{495936, 114.7916},
		{496800, 114.7892},
		{497664, 114.777},
		{498528, 114.761},
		{501984, 114.7346},
		{502848, 114.6764},
		{503712, 114.6507},
		{504576, 114.6442},
		{507168, 114.608},
		{508032, 114.6036},
		{508896, 114.5777},
		{509760, 114.5997},
		{510624, 114.5884},
		{513216, 114.5612},
		{514080, 114.4844},
		{514944, 114.4391},
		{515808, 114.3812},
		{516672, 114.3764},
		{519264, 114.393},
		{520128, 114.3974},
		{520992, 114.4017},
		{521856, 114.3634},
		{522720, 114.3016},
		{525312, 114.2515},
		{526176, 114.2482},
		{527040, 114.1958},
		{527904, 114.1907},
		{528768, 114.1731},
		{531360, 114.1669},
		{532224, 114.1693},
		{533088, 114.1072},
		{533952, 114.1024},
		{534816, 114.0735},
		{537408, 114.0702},
		{538272, 114.0803},
		{539136, 114.0691},
		{540000, 114.0771},
		{540864, 114.0757},
		{543456, 114.0604},
		{544320, 114.0077},
		{545184, 114.0382},
		{546048, 114.0469},
		{546912, 114.0486},
		{549504, 114.0517},
		{550368, 114.051},
		{551232, 114.0539},
		{552096, 114.0527},
		{552960, 114.0833},
		{555552, 114.0489},
		{556416, 114.0399},
		{557280, 114.0304},
		{558144, 113.991},
		{559008, 113.966},
		{561600, 113.9427},
		{562464, 113.8802},
		{563328, 113.8774},
		{564192, 113.8702},
		{565056, 113.8718},
		{567648, 113.8611},
		{568512, 113.8617},
		{569376, 113.8643},
		{570240, 113.8887},
		{571104, 113.8755},
		{573696, 113.8479},
		{574560, 113.8413},
		{575424, 113.8582},
		{576288, 113.8934},
		{577152, 113.8797},
		{579744, 113.9056},
		{580608, 113.8961},
		{581472, 113.9018},
		{582336, 113.882},
		{583200, 113.8696},
		{585792, 113.8311},
		{586656, 113.8509},
		{587520, 113.8489},
		{588384, 113.8419},
		{589248, 113.8432},
		{591840, 113.8469},
		{592704, 113.8621},
		{593568, 113.8548},
		{594432, 113.8804},
		{595296, 113.8998},
		{597888, 113.9299},
		{598752, 114.0812},
		{599616, 114.0825},
		{600480, 114.121},
		{601344, 114.1423},
		{603936, 114.1304},
		{604800, 114.1454},
		{605664, 114.1524},
		{606528, 114.1724},
		{607392, 114.1999},
		{609984, 114.2047},
		{610848, 114.1988},
		{611712, 114.2286},
		{612576, 114.2462},
		{613440, 114.2384},
		{616032, 114.267},
		{616896, 114.2641},
		{617760, 114.267},
		{618624, 114.2949},
		{619488, 114.3368},
		{622080, 114.3602},
		{622944, 114.3704},
		{623808, 114.3617},
		{624672, 114.3807},
		{625536, 114.3809},
		{628128, 114.3669},
		{628992, 114.3764},
		{629856, 114.3967},
		{630720, 114.4546},
		{631584, 114.4774},
		{635040, 114.4942},
		{635904, 114.4962},
		{636768, 114.521},
		{637632, 114.608},
		{640224, 114.6818},
		{641088, 114.7097},
		{641952, 114.7651},
		{642816, 114.7783},
		{643680, 114.7864},
		{646272, 114.7924},
		{647136, 114.827},
		{648000, 114.836},
		{648864, 114.8642},
		{649728, 114.8998},
		{652320, 114.9252},
		{653184, 114.9355},
		{654048, 114.976},
		{654912, 114.955},
		{655776, 115.0086},
		{658368, 115.0375},
		{659232, 115.0676},
		{660096, 115.0659},
		{660960, 115.0562},
		{661824, 115.216},
		{664416, 115.2242},
		{665280, 115.2631},
		{666144, 115.3},
		{667008, 115.3189},
		{667872, 115.3668},
		{670464, 115.3718},
		{671328, 115.3902},
		{672192, 115.4064},
		{673056, 115.4674},
		{673920, 115.5246},
		{676512, 115.5349},
		{677376, 115.526},
		{678240, 115.5305},
		{679104, 115.5676},
		{679968, 115.6666},
		{682560, 115.6712},
		{683424, 115.696},
		{684288, 115.8227},
		{685152, 115.7891},
		{686016, 115.7853},
		{688608, 115.736},
		{689472, 115.7663},
		{690336, 115.7615},
		{691200, 115.7494},
		{692064, 115.7822},
		{694656, 115.795},
		{695520, 115.8461},
		{696384, 115.8235},
		{697248, 115.8499},
		{698112, 115.8514},
		{700704, 115.8539},
		{701568, 115.8595},
		{702432, 115.9832},
		{703296, 115.9467},
		{704160, 115.8855},
		{706752, 115.8601},
		{707616, 115.8416},
		{708480, 115.8306},
		{713664, 115.828},
		{714528, 115.8088},
		{715392, 115.8051},
		{716256, 115.8129},
		{718848, 115.7657},
		{719712, 115.7389},
		{720576, 115.7305},
		{721440, 115.7202},
		{722304, 115.7101},
		{724896, 115.6463},
		{725760, 115.6674},
		{726624, 115.6681},
		{728352, 115.6508},
		{730944, 115.6624},
		{731808, 115.6538},
		{732672, 115.6634},
		{733536, 115.6724},
		{734400, 115.675},
		{736992, 115.6567},
		{737856, 115.6553},
		{738720, 115.6975},
		{739584, 115.8159},
		{740448, 115.8571},
		{743040, 115.8801},
		{743904, 115.8645},
		{744768, 115.8485},
		{746496, 115.8516},
		{749088, 115.8313},
		{749952, 115.8387},
		{750816, 115.8258},
		{751680, 115.8055},
		{752544, 115.8234},
		{756000, 115.8685},
		{756864, 115.8626},
		{757728, 115.8729},
		{758592, 115.9012},
		{761184, 116.0026},
		{762048, 116.006},
		{762912, 115.9997},
		{763776, 116.0022},
		{764640, 115.9955},
		{767232, 115.9985},
		{768096, 116.0084},
		{768960, 115.998},
		{769824, 115.9863},
		{770688, 115.9723},
		{773280, 116.0448},
		{774144, 116.0377},
		{775008, 116.0597},
		{775872, 116.1389},
		{779328, 116.0719},
		{780192, 116.0695},
		{781056, 116.0698},
		{781920, 116.1061},
		{782784, 116.1309},
		{785376, 116.1241},
		{786240, 116.1664},
		{787104, 116.1617},
		{787968, 116.1739},
		{788832, 116.2175},
		{791424, 116.2525},
		{792288, 116.2638},
		{793152, 116.2732},
		{794016, 116.3029},
		{794880, 116.3529},
		{797472, 116.3529},
		{798336, 116.3531},
		{799200, 116.3435},
		{800064, 116.3467},
		{800928, 116.3474},
		{803520, 116.3346},
		{804384, 116.3442},
		{805248, 116.3329},
		{806112, 116.3209},
		{806976, 116.3135},
		{809568, 116.3075},
		{810432, 116.2987},
		{811296, 116.302},
		{812160, 116.306},
		{813024, 116.3154},
		{816480, 116.3266},
		{817344, 116.3292},
		{818208, 116.3365},
		{819072, 116.347},
		{821664, 116.3574},
		{822528, 116.3397},
		{823392, 116.3357},
		{824256, 116.3371},
		{825120, 116.4034},
		{827712, 116.4183},
		{828576, 116.4003},
		{829440, 116.4201},
		{830304, 116.4262},
		{831168, 116.4708},
		{833760, 116.4594},
		{834624, 116.467},
		{835488, 116.4884},
		{836352, 116.4819},
		{837216, 116.4166},
		{839808, 116.2509},
		{840672, 116.2649},
		{841536, 116.2537},
		{842400, 116.2496},
		{843264, 116.2705},
		{845856, 116.258},
		{846720, 116.2603},
		{847584, 116.2721},
		{848448, 116.2668},
		{849312, 116.2721},
		{851904, 116.2693},
		{852768, 116.2625},
		{853632, 116.2639},
		{854496, 116.2821},
		{855360, 116.2592},
		{857952, 116.2599},
		{858816, 116.2463},
		{859680, 116.2414},
		{860544, 116.232},
		{861408, 116.2672},
		{864000, 116.3052},
		{864864, 116.2457},
		{865728, 116.2763},
		{866592, 116.3173},
		{867456, 116.3243},
		{870048, 116.3127},
		{870912, 116.296},
		{871776, 116.346},
		{872640, 116.4735},
		{873504, 116.5554},
		{876096, 116.6067},
		{876960, 116.636},
		{877824, 116.6441},
		{878688, 116.6389},
		{879552, 116.6432},
		{882144, 116.7098},
		{883008, 116.7215},
		{883872, 116.7248},
		{884736, 116.7313},
		{885600, 116.73},
		{888192, 116.7551},
		{889056, 116.8187},
		{889920, 116.648},
		{890784, 116.48},
		{891648, 116.3853},
		{894240, 116.3846},
		{895104, 116.3691},
		{895968, 116.3632},
		{896832, 116.3582},
		{897696, 116.3464},
		{900288, 116.3189},
		{901152, 116.3092},
		{902016, 116.3197},
		{902880, 116.3197},
		{903744, 116.3027},
		{906336, 116.3123},
		{907200, 116.3283},
		{908064, 116.3339},
		{908928, 116.3355},
		{909792, 116.3408},
		{912384, 116.3393},
		{913248, 116.3339},
		{914112, 116.3489},
		{914976, 116.3598},
		{915840, 116.3577},
		{918432, 116.3534},
		{919296, 116.3354},
		{920160, 116.3462},
		{921024, 116.3489},
		{921888, 116.3616},
		{924480, 116.3736},
		{925344, 116.3834},
		{926208, 116.375},
		{927072, 116.3862},
		{927936, 116.3812},
		{930528, 116.3812},
		{931392, 116.3627},
		{932256, 116.3464},
		{933120, 116.3354},
		{933984, 116.3635},
		{936576, 116.3644},
		{937440, 116.3669},
		{938304, 116.3572},
		{939168, 116.3717},
		{940032, 116.3697},
		{943488, 116.4196},
		{944352, 116.4236},
		{945216, 116.4047},
		{946080, 116.4796},
		{949536, 116.4773},
		{950400, 116.481},
		{951264, 116.4885},
		{952128, 116.491},
		{954720, 116.49},
		{955584, 116.5212},
		{956448, 116.5212},
		{957312, 116.5058},
		{958176, 116.5025},
		{960768, 116.5138},
		{961632, 116.5193},
		{962496, 116.5219},
		{963360, 116.5607},
		{964224, 116.5616},
		{966816, 116.5522},
		{967680, 116.5731},
		{968544, 116.5807},
		{969408, 116.5772},
		{970272, 116.5828},
		{972864, 116.5821},
		{973728, 116.5891},
		{974592, 116.2576},
		{975456, 116.0121},
		{976320, 115.7815},
		{978912, 115.4935},
		{979776, 115.4833},
		{980640, 115.4738},
		{981504, 115.4821},
		{982368, 115.4803},
		{984960, 115.4684},
		{985824, 115.3823},
		{986688, 115.3757},
		{987552, 115.4372},
		{988416, 115.4996},
		{991008, 115.6133},
		{991872, 115.6753},
		{992736, 115.7288},
		{993600, 115.7282},
		{994464, 115.7127},
		{997056, 115.7362},
		{997920, 115.7939},
		{998784, 115.8383},
		{999648, 115.8499},
		{1000512, 115.8962},
		{1003104, 115.9555},
		{1003968, 116.0545},
		{1004832, 116.1207},
		{1005696, 116.1379},
		{1006560, 116.1405},
		{1009152, 116.1372},
		{1010016, 116.1306},
		{1010880, 116.1579},
		{1011744, 116.1522},
		{1012608, 116.1572},
		{1015200, 116.1582},
		{1016064, 116.1551},
		{1016928, 116.1419},
		{1017792, 116.1222},
		{1018656, 116.1252},
		{1021248, 116.1178},
		{1022112, 116.1045},
		{1022976, 116.1029},
		{1023840, 116.0845},
		{1024704, 116.0747},
		{1027296, 116.0666},
		{1028160, 116.0669},
		{1029024, 116.0805},
		{1029888, 116.0688},
		{1030752, 116.0735},
		{1033344, 116.0964},
		{1034208, 116.1008},
		{1035072, 116.1036},
		{1040256, 116.0778},
		{1041120, 116.0725},
		{1042848, 116.0824},
		{1045440, 116.0905},
		{1046304, 116.0844},
		{1047168, 116.1241},
		{1048032, 116.1489},
		{1048896, 116.1538},
		{1052352, 116.1572},
		{1053216, 116.165},
		{1054080, 116.1664},
		{1054944, 116.1496},
		{1057536, 116.1594},
		{1058400, 116.1504},
		{1059264, 116.1466},
		{1060128, 116.1444},
		{1060992, 116.1244},
		{1063584, 116.1116},
		{1064448, 116.1214},
		{1065312, 116.112},
		{1066176, 116.1124},
		{1067040, 116.1164},
		{1069632, 116.1137},
		{1070496, 116.1192},
		{1071360, 116.1062},
		{1073088, 116.0942},
		{1075680, 116.1029},
		{1076544, 116.0997},
		{1077408, 116.1112},
		{1078272, 115.9905},
		{1079136, 116.0046},
		{1082592, 116.0006},
		{1083456, 115.9964},
		{1084320, 115.8797},
		{1085184, 115.8801},
		{1087776, 115.8838},
		{1088640, 115.8842},
		{1089504, 115.7584},
		{1090368, 115.756},
		{1091232, 115.7432},
		{1093824, 115.7506},
		{1094688, 115.7494},
		{1095552, 115.7366},
		{1096416, 115.7296},
		{1097280, 115.7237},
		{1099872, 115.7262},
		{1100736, 115.8238},
		{1101600, 115.7272},
		{1102464, 115.722},
		{1103328, 115.7671},
		{1105920, 115.8145},
		{1106784, 115.804},
		{1107648, 115.8264},
		{1108512, 115.8166},
		{1109376, 115.8284},
		{1111968, 115.8292},
		{1112832, 115.831},
		{1113696, 115.8723},
		{1114560, 115.8709},
		{1115424, 115.8595},
		{1118016, 115.8619},
		{1118880, 115.8634},
		{1119744, 115.8759},
		{1120608, 115.9135},
		{1121472, 115.9485},
		{1124064, 115.9395},
		{1124928, 115.9471},
		{1125792, 115.9781},
		{1126656, 116.0212},
		{1127520, 116.0273},
		{1130112, 116.0836},
		{1130976, 116.118},
		{1131840, 116.1305},
		{1132704, 116.1454},
		{1133568, 116.1487},
		{1137024, 116.154},
		{1137888, 116.1611},
		{1138752, 116.1383},
		{1139616, 116.1435},
		{1142208, 116.1349},
		{1143072, 116.1501},
		{1143936, 116.1641},
		{1144800, 116.1548},
		{1145664, 116.1378},
		{1148256, 116.1361},
		{1149120, 116.138},
		{1149984, 116.1322},
		{1150848, 116.1296},
		{1151712, 116.1367},
		{1154304, 116.1362},
		{1155168, 116.1272},
		{1156032, 115.9654},
		{1156896, 115.9874},
		{1157760, 116.007},
		{1160352, 115.9912},
		{1161216, 115.9401},
		{1162080, 115.9528},
		{1162944, 115.9539},
		{1163808, 115.9687},
		{1166400, 115.9676},
		{1167264, 115.9684},
		{1168128, 115.9647},
		{1168992, 115.9742},
		{1169856, 115.9659},
		{1172448, 115.9747},
		{1173312, 115.9745},
		{1174176, 115.978},
		{1175040, 115.9705},
		{1175904, 115.9743},
		{1178496, 116.0468},
		{1179360, 116.0563},
		{1180224, 116.0646},
		{1181088, 116.1037},
		{1181952, 116.0944},
		{1184544, 116.0867},
		{1185408, 116.0868},
		{1186272, 116.0771},
		{1187136, 116.0706},
		{1188000, 116.0656},
		{1190592, 116.0743},
		{1191456, 116.0802},
		{1192320, 116.0899},
		{1193184, 116.0811},
		{1194048, 116.0857},
		{1196640, 116.0885},
		{1197504, 116.0859},
		{1198368, 116.0942},
		{1199232, 116.0877},
		{1200096, 116.0831},
		{1202688, 116.0996},
		{1203552, 116.1005},
		{1204416, 116.1038},
		{1205280, 116.1018},
		{1206144, 116.1005},
		{1208736, 116.1029},
		{1209600, 116.0977},
		{1210464, 116.0981},
		{1211328, 116.1077},
		{1212192, 116.1019},
		{1214784, 116.0848},
		{1215648, 116.0743},
		{1216512, 116.1389},
		{1217376, 116.1456},
		{1218240, 116.187},
		{1220832, 116.1543},
		{1221696, 116.1246},
		{1222560, 116.1248},
		{1223424, 116.1272},
		{1224288, 116.1241},
		{1226880, 116.1248},
		{1227744, 116.1329},
		{1228608, 116.1307},
		{1229472, 116.1338},
		{1230336, 116.1362},
		{1232928, 116.1596},
		{1233792, 116.1677},
		{1234656, 116.1767},
		{1235520, 116.1831},
		{1236384, 116.2192},
		{1238976, 116.2327},
		{1239840, 116.262},
		{1240704, 116.266},
		{1241568, 116.2938},
		{1242432, 116.3011},
		{1245024, 116.303},
		{1245888, 116.3079},
		{1246752, 116.3172},
		{1247616, 116.3182},
		{1248480, 116.3062},
		{1251072, 116.3067},
		{1251936, 116.3041},
		{1252800, 116.3012},
		{1253664, 116.3215},
		{1254528, 116.3337},
		{1258848, 116.3078},
		{1259712, 116.3076},
		{1260576, 116.3078},
		{1264896, 116.3018},
		{1265760, 116.3671},
		{1266624, 116.3815},
		{1269216, 116.3743},
		{1270080, 116.3779},
		{1270944, 116.3754},
		{1271808, 116.3805},
		{1272672, 116.38},
		{1275264, 116.37},
		{1276128, 116.3834},
		{1276992, 116.3767},
		{1277856, 116.3773},
		{1278720, 116.3824},
		{1281312, 116.3569},
		{1282176, 116.3564},
		{1283040, 116.3685},
		{1283904, 116.3677},
		{1284768, 116.4069},
		{1287360, 116.4026},
		{1288224, 116.4082},
		{1289088, 116.4084},
		{1289952, 116.4001},
		{1290816, 116.4},
		{1293408, 116.3619},
		{1294272, 116.3589},
		{1295136, 116.3615},
		{1296000, 116.3773},
		{1296864, 116.3848},
		{1299456, 116.3707},
		{1300320, 116.3752},
		{1301184, 116.3648},
		{1302048, 116.3544},
		{1302912, 116.3328},
		{1305504, 116.3245},
		{1306368, 116.3277},
		{1307232, 116.2676},
		{1308096, 116.2813},
		{1308960, 116.3276},
		{1311552, 116.3214},
		{1312416, 116.3228},
		{1313280, 116.3464},
		{1314144, 116.3036},
		{1315008, 116.2971},
		{1317600, 116.306},
		{1318464, 116.2506},
		{1319328, 116.2414},
		{1320192, 116.2374},
		{1321056, 116.2357},
		{1323648, 116.2416},
		{1324512, 116.2237},
		{1325376, 116.2247},
		{1326240, 116.2231},
		{1327104, 116.1834},
		{1329696, 116.1437},
		{1330560, 116.137},
		{1331424, 116.1419},
		{1332288, 116.0827},
		{1333152, 116.0874},
		{1335744, 116.09},
		{1336608, 116.0784},
		{1337472, 116.0451},
		{1338336, 116.0307},
		{1339200, 116.0318},
		{1341792, 116.0232},
		{1342656, 116.0097},
		{1343520, 116.0104},
		{1348704, 116.004},
		{1349568, 115.949},
		{1350432, 115.9399},
		{1351296, 115.9297},
		{1353888, 115.9353},
		{1354752, 115.9265},
		{1355616, 115.9166},
		{1356480, 115.8912},
		{1357344, 115.8887},
		{1359936, 115.8887},
		{1360800, 115.8901},
		{1361664, 115.8925},
		{1363392, 115.8997},
		{1365984, 115.9164},
		{1366848, 115.9026},
		{1368576, 115.9016},
		{1369440, 115.8695},
		{1372032, 115.8636},
		{1372896, 115.8787},
		{1373760, 115.8821},
		{1374624, 115.906},
		{1375488, 115.9117},
		{1378080, 115.906},
		{1378944, 115.8984},
		{1379808, 115.9001},
		{1381536, 115.9047},
		{1384128, 115.9087},
		{1384992, 115.9542},
		{1385856, 116.0064},
		{1386720, 116.0135},
		{1387584, 116.0212},
		{1391040, 116.0224},
		{1391904, 116.0229},
		{1392768, 116.0383},
		{1393632, 116.0424},
		{1396224, 116.0324},
		{1397088, 116.0315},
		{1397952, 116.032},
		{1398816, 116.0017},
		{1399680, 116.0191},
		{1402272, 116.0216},
		{1403136, 116.033},
		{1404000, 116.0332},
		{1404864, 116.0307},
		{1405728, 116.0304},
		{1409184, 116.0413},
		{1410048, 116.0703},
		{1410912, 116.0647},
		{1411776, 116.0727},
		{1414368, 116.106},
		{1415232, 116.1179},
		{1416096, 116.152},
		{1416960, 116.144},
		{1417824, 116.1764},
		{1420416, 116.1682},
		{1421280, 116.1837},
		{1422144, 116.1928},
		{1423008, 116.1765},
		{1423872, 116.1962},
		{1426464, 116.2084},
		{1427328, 116.2198},
		{1428192, 116.2208},
		{1429056, 116.2358},
		{1429920, 116.1973},
		{1432512, 116.2091},
		{1433376, 116.2143},
		{1434240, 116.2222},
		{1435104, 116.2468},
		{1435968, 116.2435},
		{1438560, 116.2503},
		{1439424, 116.2474},
		{1440288, 116.2436},
		{1441152, 116.2451},
		{1442016, 116.2591},
		{1444608, 116.2573},
		{1445472, 116.2442},
		{1446336, 116.2832},
		{1447200, 116.2875},
		{1448064, 116.3202},
		{1451520, 116.2788},
		{1452384, 116.3264},
		{1453248, 116.3361},
		{1454112, 116.3387},
		{1456704, 116.3308},
		{1457568, 116.2796},
		{1458432, 116.2904},
		{1459296, 116.2356},
		{1460160, 116.2366},
		{1462752, 116.2473},
		{1463616, 116.244},
		{1464480, 116.2334},
		{1465344, 116.2171},
		{1466208, 116.2076},
		{1468800, 116.2135},
		{1469664, 116.217},
		{1470528, 116.2295},
		{1471392, 116.2206},
		{1472256, 116.2297},
		{1474848, 116.2238},
		{1475712, 116.2335},
		{1476576, 116.2569},
		{1477440, 116.2627},
		{1478304, 116.269},
		{1480896, 116.2634},
		{1481760, 116.2785},
		{1482624, 116.2893},
		{1483488, 116.2956},
		{1484352, 116.3066},
		{1486944, 116.3071},
		{1487808, 116.3147},
		{1488672, 116.3256},
		{1489536, 116.3465},
		{1490400, 116.333},
		{1492992, 116.3322},
		{1493856, 116.3254},
		{1494720, 116.3277},
		{1495584, 116.3305},
		{1496448, 116.3264},
		{1499040, 116.3092},
		{1499904, 116.2848},
		{1500768, 116.2642},
		{1501632, 116.2668},
		{1502496, 116.2311},
		{1505088, 116.2374},
		{1505952, 116.2271},
		{1506816, 116.2249},
		{1507680, 116.251},
		{1508544, 116.2366},
		{1511136, 116.2342},
		{1512000, 116.236},
		{1512864, 116.2404},
		{1513728, 116.2109},
		{1514592, 116.2182},
		{1517184, 116.2051},
		{1518048, 116.152},
		{1518912, 116.1169},
		{1519776, 116.1104},
		{1520640, 116.1143},
		{1523232, 116.1121},
		{1524096, 116.0624},
		{1524960, 116.0627},
		{1525824, 115.9888},
		{1526688, 115.9996},
		{1529280, 116.0423},
		{1530144, 116.0656},
		{1531008, 116.0844},
		{1531872, 116.1235},
		{1532736, 116.1837},
		{1535328, 116.1863},
		{1536192, 116.1918},
		{1537056, 116.2448},
		{1537920, 116.2523},
		{1538784, 116.286},
		{1541376, 116.3432},
		{1542240, 116.383},
		{1543104, 116.2967},
		{1543968, 116.2692},
		{1544832, 116.269},
		{1547424, 116.1418},
		{1548288, 116.1074},
		{1549152, 116.0564},
		{1550016, 115.9962},
		{1550880, 116.0287},
		{1553472, 116.041},
		{1554336, 116.0474},
		{1555200, 116.039},
		{1556064, 116.0442},
		{1556928, 115.9869},
		{1559520, 115.9467},
		{1560384, 115.8948},
		{1561248, 115.9024},
		{1562112, 115.9001},
		{1562976, 115.8673},
		{1565568, 115.8442},
		{1566432, 115.769},
		{1567296, 115.6724},
		{1568160, 115.6013},
		{1569024, 115.603},
		{1571616, 115.5851},
		{1572480, 115.5833},
		{1575072, 115.5813},
		{1577664, 115.5921},
		{1578528, 115.6225},
		{1581120, 115.6536},
		{1583712, 115.6777},
		{1584576, 115.6669},
		{1585440, 115.7199},
		{1586304, 115.765},
		{1587168, 115.7763},
		{1589760, 115.7794},
		{1590624, 115.8101},
		{1591488, 115.7966},
		{1592352, 115.7924},
		{1593216, 115.8159},
		{1595808, 115.8093},
		{1596672, 115.8069},
		{1597536, 115.8091},
		{1598400, 115.8251},
		{1599264, 115.8541},
		{1601856, 115.858},
		{1602720, 115.8549},
		{1603584, 115.8711},
		{1604448, 115.9225},
		{1605312, 115.9686},
		{1607904, 116.0083},
		{1608768, 116.0376},
		{1609632, 116.0436},
		{1610496, 116.06},
		{1611360, 116.0606},
		{1613952, 116.0338},
		{1614816, 116.0169},
		{1615680, 116.0014},
		{1616544, 115.9902},
		{1617408, 115.9693},
		{1620000, 115.9662},
		{1620864, 115.9705},
		{1621728, 115.9451},
		{1622592, 115.9393},
		{1623456, 115.9327},
		{1626048, 115.9291},
		{1626912, 115.928},
		{1627776, 115.8632},
		{1628640, 115.8521},
		{1629504, 115.8477},
		{1632096, 115.7842},
		{1632960, 115.7926},
		{1633824, 115.7946},
		{1634688, 115.7929},
		{1635552, 115.8032},
		{1638144, 115.8096},
		{1639008, 115.8202},
		{1639872, 115.8208},
		{1640736, 115.8169},
		{1641600, 115.7822},
		{1644192, 115.756},
		{1645056, 115.7195},
		{1645920, 115.6997},
		{1646784, 115.6664},
		{1647648, 115.7067},
		{1650240, 115.6976},
		{1651104, 115.7196},
		{1651968, 115.6946},
		{1657152, 115.6944},
		{1658016, 115.7075},
		{1658880, 115.7173},
		{1659744, 115.6692},
		{1662336, 115.6647},
		{1663200, 115.5655},
		{1664064, 115.5053},
		{1664928, 115.4567},
		{1665792, 115.4317},
		{1668384, 115.3843},
		{1669248, 115.3145},
		{1670112, 115.2434},
		{1670976, 115.16},
		{1671840, 115.1053},
		{1674432, 115.1046},
		{1675296, 115.1131},
		{1676160, 115.0779},
		{1677888, 115.0834},
		{1680480, 115.046},
		{1681344, 114.9434},
		{1682208, 114.9589},
		{1683936, 114.9139},
		{1686528, 114.9113},
		{1687392, 114.9061},
		{1688256, 114.8241},
		{1689984, 114.8113},
		{1692576, 114.7952},
		{1693440, 114.7997},
		{1694304, 114.8032},
		{1695168, 114.8017},
		{1696032, 114.7321},
		{1699488, 114.7403},
		{1700352, 114.7417},
		{1701216, 114.722},
		{1702080, 114.6778},
		{1704672, 114.6332},
		{1705536, 114.6052},
		{1706400, 114.607},
		{1707264, 114.5527},
		{1708128, 114.5364},
		{1710720, 114.4675},
		{1711584, 114.4068},
		{1712448, 114.4077},
		{1713312, 114.3274},
		{1714176, 114.3202},
		{1716768, 114.3079},
		{1717632, 114.2907},
		{1718496, 114.2935},
		{1719360, 114.2867},
		{1720224, 114.2986},
		{1722816, 114.3096},
		{1724544, 114.3088},
		{1725408, 114.2589},
		{1726272, 114.1713},
		{1728864, 114.0902},
		{1729728, 114.0177},
		{1730592, 113.9982},
		{1731456, 113.9735},
		{1732320, 113.9607},
		{1734912, 113.959},
		{1735776, 113.9623},
		{1736640, 113.9618},
		{1737504, 113.9384},
		{1738368, 113.9411},
		{1740960, 113.7615},
		{1741824, 113.7287},
		{1742688, 113.6797},
		{1743552, 113.6222},
		{1744416, 113.6257},
		{1747008, 113.5956},
		{1747872, 113.5444},
		{1748736, 113.47},
		{1749600, 113.4186},
		{1750464, 113.3116},
		{1753056, 113.2155},
		{1753920, 113.0797},
		{1754784, 112.994},
		{1755648, 112.9599},
		{1756512, 112.9069},
		{1759104, 112.8783},
		{1759968, 112.8125},
		{1760832, 112.8181},
		{1761696, 112.8131},
		{1762560, 112.875},
		{1766016, 112.9323},
		{1766880, 112.928},
		{1767744, 112.9425},
		{1768608, 112.956},
		{1771200, 112.9898},
		{1772064, 113.0505},
		{1772928, 113.0709},
		{1773792, 113.222},
		{1774656, 113.2301},
		{1777248, 113.2623},
		{1778112, 113.2973},
		{1778976, 113.3491},
		{1779840, 113.4408},
		{1780704, 113.5307},
		{1783296, 113.5164},
		{1784160, 113.517},
		{1785024, 113.5128},
		{1785888, 113.5285},
		{1786752, 113.551},
		{1789344, 113.5762},
		{1790208, 113.6111},
		{1791072, 113.5442},
		{1791936, 113.4521},
		{1792800, 113.3712},
		{1795392, 113.2098},
		{1796256, 113.139},
		{1797120, 113.1381},
		{1797984, 113.1036},
		{1798848, 113.2375},
		{1801440, 113.2853},
		{1802304, 113.3153},
		{1803168, 113.3402},
		{1804032, 113.5632},
		{1804896, 113.6026},
		{1807488, 113.7309},
		{1808352, 113.9587},
		{1809216, 114.3627},
		{1810080, 114.3636},
		{1810944, 114.3193},
		{1813536, 114.28},
		{1814400, 114.0171},
		{1815264, 114.0032},
		{1816128, 114.0091},
		{1816992, 113.9701},
		{1819584, 114.0908},
		{1820448, 114.143},
		{1821312, 114.4153},
		{1822176, 114.5279},
		{1823040, 114.5388},
		{1825632, 114.5535},
		{1826496, 114.6705},
		{1827360, 114.7237},
		{1828224, 115.0071},
		{1829088, 115.1181},
		{1831680, 115.1998},
		{1832544, 115.4228},
		{1833408, 115.6866},
		{1834272, 115.6592},
		{1835136, 115.4869},
		{1837728, 115.0525},
		{1838592, 115.4001},
		{1839456, 115.3916},
		{1840320, 115.6551},
		{1841184, 115.6546},
		{1843776, 115.6903},
		{1844640, 115.5716},
		{1845504, 115.6168},
		{1846368, 115.614},
		{1847232, 115.6074},
		{1849824, 115.5989},
		{1850688, 115.6858},
		{1851552, 115.5662},
		{1852416, 115.4809},
		{1853280, 115.4593},
		{1855872, 115.4298},
		{1856736, 115.3197},
		{1857600, 115.2871},
		{1858464, 114.7486},
		{1859328, 114.6698},
		{1861920, 114.6438},
		{1862784, 114.62},
		{1863648, 114.4538},
		{1864512, 114.1553},
		{1865376, 114.1288},
		{1867968, 114.1265},
		{1868832, 114.1241},
		{1869696, 114.2269},
		{1870560, 114.2707},
		{1871424, 114.2148},
		{1874016, 114.234},
		{1874880, 113.699},
		{1875744, 113.662},
		{1876608, 113.886},
		{1877472, 113.937},
		{1880064, 113.7925},
		{1880928, 114.255},
		{1881792, 114.0895},
		{1882656, 114.1317},
		{1883520, 114.1313},
		{1886112, 114.0103},
		{1886976, 113.9},
		{1887840, 113.9984},
		{1892160, 114.069},
		{1893024, 113.8795},
		{1893888, 113.9855},
		{1898208, 113.9454},
		{1899072, 113.616},
		{1899936, 113.5875},
		{1900800, 113.4888},
		{1901664, 113.5129},
		{1904256, 113.4688},
		{1905120, 113.5533},
		{1905984, 113.6223},
		{1906848, 114.009},
		{1907712, 113.9014},
		{1910304, 113.956},
		{1911168, 114.1737},
		{1912032, 114.3161},
		{1912896, 114.7513},
		{1913760, 114.4868},
		{1916352, 114.3867},
		{1917216, 114.5819},
		{1918080, 114.5593},
		{1918944, 114.7098},
		{1919808, 114.3631},
		{1922400, 114.2169},
		{1923264, 114.2169},
		{1924128, 114.019},
		{1924992, 113.8082},
		{1925856, 114.2308},
		{1928448, 114.2735},
		{1929312, 113.5179},
		{1930176, 113.2453},
		{1931040, 113.2326},
		{1931904, 113.2412},
		{1934496, 112.741},
		{1935360, 112.67},
		{1936224, 112.6292},
		{1937088, 112.6494},
		{1937952, 112.7451},
		{1940544, 113.0597},
		{1941408, 113.2121},
		{1942272, 113.2417},
		{1943136, 112.9081},
		{1944000, 113.0045},
		{1946592, 113.1177},
		{1947456, 113.0697},
		{1948320, 113.3713},
		{1949184, 113.5998},
		{1950048, 113.9526},
		{1952640, 113.8947},
		{1953504, 113.6359},
		{1954368, 113.5309},
		{1955232, 113.5165},
		{1956096, 113.4708},
		{1958688, 113.5671},
		{1959552, 113.4964},
		{1960416, 113.4826},
		{1961280, 113.3555},
		{1962144, 113.2594},
		{1964736, 113.1598},
		{1965600, 113.1424},
		{1966464, 113.1387},
		{1967328, 113.1376},
		{1968192, 113.3734},
		{1970784, 113.314},
		{1971648, 113.1493},
		{1972512, 112.9514},
		{1973376, 112.9559},
		{1974240, 112.8745},
		{1976832, 112.7196},
		{1977696, 112.6581},
		{1978560, 112.6646},
		{1983744, 112.7469},
		{1984608, 113.1603},
		{1985472, 113.0951},
		{1986336, 112.7577},
		{1988928, 112.6353},
		{1989792, 112.6327},
		{1990656, 112.7634},
		{1992384, 112.7524},
		{1994976, 112.7585},
		{1995840, 112.7233},
		{1996704, 112.6368},
		{1997568, 112.8787},
		{2001024, 112.9727},
		{2001888, 113.1302},
		{2002752, 113.2838},
		{2003616, 113.3292},
		{2004480, 113.3313},
		{2007072, 112.9855},
		{2007936, 112.7353},
		{2008800, 112.5641},
		{2009664, 112.3796},
		{2010528, 112.3713},
		{2013120, 112.2356},
		{2013984, 112.1497},
		{2014848, 112.161},
		{2016576, 112.2593},
		{2019168, 112.4072},
		{2020032, 112.5289},
		{2020896, 112.5282},
		{2021760, 112.4171},
		{2022624, 112.1477},
		{2026080, 112.1305},
		{2026944, 111.9532},
		{2027808, 111.8427},
		{2028672, 111.8224},
		{2031264, 111.6641},
		{2032128, 111.5543},
		{2032992, 111.513},
		{2033856, 111.3603},
		{2034720, 111.094},
		{2037312, 111.202},
		{2038176, 111.6642},
		{2039904, 112.1371},
		{2040768, 111.9918},
		{2043360, 111.6895},
		{2044224, 111.7099},
		{2045088, 111.5003},
		{2045952, 111.5078},
		{2046816, 111.7995},
		{2049408, 111.9566},
		{2050272, 111.8469},
		{2051136, 111.4424},
		{2052000, 111.2988},
		{2052864, 111.6753},
		{2055456, 111.7959},
		{2056320, 111.9208},
		{2057184, 111.6378},
		{2058048, 111.5404},
		{2058912, 111.4392},
		{2061504, 111.6477},
		{2062368, 111.6434},
		{2063232, 111.4478},
		{2064096, 111.3487},
		{2064960, 111.3381},
		{2067552, 111.5288},
		{2068416, 111.6629},
		{2069280, 111.6794},
		{2070144, 111.8167},
		{2071008, 111.7094},
		{2073600, 111.6888},
		{2074464, 111.6981},
		{2075328, 111.4473},
		{2076192, 111.5982},
		{2077056, 111.8617},
		{2080512, 111.434},
		{2081376, 111.3929},
		{2082240, 111.6721},
		{2083104, 111.4866},
		{2085696, 111.4885},
		{2086560, 111.4989},
		{2087424, 111.7835},
		{2088288, 111.9978},
		{2089152, 111.7688},
		{2091744, 111.4714},
		{2092608, 111.4822},
		{2093472, 111.4238},
		{2094336, 111.2242},
		{2095200, 111.3301},
		{2097792, 111.2889},
		{2098656, 111.3661},
		{2099520, 111.4914},
		{2100384, 111.9124},
		{2101248, 111.9098},
		{2103840, 112.1965},
		{2104704, 112.3094},
		{2105568, 112.3591},
		{2106432, 112.3492},
		{2107296, 112.6605},
		{2109888, 112.5047},
		{2110752, 112.3852},
		{2111616, 111.9696},
		{2112480, 112.1354},
		{2113344, 112.5013},
		{2115936, 112.7086},
		{2116800, 112.9109},
		{2117664, 112.7357},
		{2118528, 112.7868},
		{2119392, 112.9016},
		{2121984, 112.9268},
		{2122848, 112.7062},
		{2123712, 112.6427},
		{2124576, 112.4737},
		{2125440, 112.4259},
		{2128032, 112.3378},
		{2128896, 112.6899},
		{2129760, 112.7121},
		{2130624, 112.6353},
		{2131488, 112.5236},
		{2134080, 112.5442},
		{2134944, 112.6056},
		{2135808, 112.8701},
		{2136672, 112.9136},
		{2137536, 112.9184},
		{2140128, 112.7952},
		{2140992, 112.6566},
		{2141856, 112.9362},
		{2142720, 112.8907},
		{2143584, 112.7667},
		{2146176, 112.5625},
		{2147040, 112.5483},
		{2147904, 112.7752},
		{2148768, 112.8944},
		{2149632, 112.985},
		{2152224, 113.504},
		{2153088, 113.8012},
		{2153952, 113.7016},
		{2154816, 113.8289},
		{2155680, 113.7694},
		{2158272, 114.0804},
		{2159136, 114.1848},
		{2160000, 114.1904},
		{2160864, 114.2675},
		{2161728, 113.8043},
		{2164320, 113.8046},
		{2165184, 114.0655},
		{2166048, 113.879},
		{2166912, 113.7524},
		{2167776, 113.7315},
		{2170368, 113.5645},
		{2171232, 113.9304},
		{2172096, 113.6237},
		{2172960, 113.6091},
		{2173824, 113.6201},
		{2176416, 113.6374},
		{2177280, 113.62},
		{2178144, 113.8442},
		{2179008, 113.7139},
		{2179872, 113.8743},
		{2182464, 113.5809},
		{2183328, 113.8184},
		{2184192, 114.0326},
		{2185056, 113.8991},
		{2185920, 113.4221},
		{2188512, 113.4169},
		{2189376, 113.7661},
		{2190240, 114.0231},
		{2191104, 113.844},
		{2191968, 113.7971},
		{2194560, 113.6751},
		{2195424, 113.9142},
		{2196288, 114.057},
		{2197152, 113.6348},
		{2198016, 113.8036},
		{2200608, 113.6369},
		{2201472, 113.6323},
		{2202336, 113.4385},
		{2203200, 113.242},
		{2206656, 113.0958},
		{2207520, 113.0982},
		{2208384, 113.4068},
		{2209248, 113.1716},
		{2213568, 113.48},
		{2214432, 113.5133},
		{2215296, 113.6726},
		{2216160, 113.4662},
		{2218752, 113.3871},
		{2219616, 113.388},
		{2220480, 113.3815},
		{2221344, 113.22},
		{2222208, 113.1043},
		{2224800, 112.8915},
		{2225664, 113.1014},
		{2226528, 113.144},
		{2227392, 113.1418},
		{2228256, 113.0136},
		{2230848, 112.8994},
		{2231712, 112.6861},
		{2232576, 112.6528},
		{2233440, 112.7963},
		{2234304, 112.7624},
		{2236896, 112.7928},
		{2237760, 112.9143},
		{2238624, 112.9178},
		{2239488, 112.7617},
		{2240352, 112.865},
		{2242944, 112.7477},
		{2243808, 112.834},
		{2244672, 112.7684},
		{2245536, 112.9189},
		{2246400, 113.0087},
		{2248992, 113.1685},
		{2249856, 113.4858},
		{2250720, 113.1997},
		{2251584, 113.3651},
		{2252448, 113.3823},
		{2255040, 113.6557},
		{2255904, 113.8958},
		{2256768, 113.3521},
		{2257632, 113.2299},
		{2258496, 113.4677},
		{2261088, 113.189},
		{2261952, 113.2724},
		{2262816, 113.18},
		{2263680, 113.2787},
		{2264544, 113.2594},
		{2267136, 113.1439},
		{2268000, 113.0418},
		{2268864, 113.0367},
		{2269728, 113.0139},
		{2270592, 112.9855},
		{2273184, 113.0157},
		{2274048, 113.0083},
		{2274912, 113.0315},
		{2275776, 113.0582},
		{2276640, 113.0527},
		{2279232, 113.0235},
		{2280096, 113.1341},
		{2280960, 113.1638},
		{2281824, 113.3341},
		{2282688, 113.1461},
		{2285280, 113.0858},
		{2286144, 113.069},
		{2287008, 113.2522},
		{2292192, 113.455},
		{2293056, 113.7357},
		{2293920, 113.6823},
		{2294784, 113.755},
		{2297376, 113.8771},
		{2298240, 113.9822},
		{2299104, 113.7146},
		{2299968, 113.6099},
		{2300832, 113.571},
		{2303424, 113.4459},
		{2304288, 113.3429},
		{2305152, 113.3611},
		{2306880, 113.5431},
		{2309472, 113.6609},
		{2310336, 113.6476},
		{2311200, 113.7341},
		{2312064, 113.6699},
		{2312928, 113.7573},
		{2315520, 114.0738},
		{2316384, 114.3123},
		{2317248, 114.2876},
		{2318112, 114.3782},
		{2318976, 114.5628},
		{2321568, 114.6141},
		{2322432, 114.6895},
		{2323296, 114.5947},
		{2325024, 114.3091},
		{2327616, 113.9193},
		{2328480, 113.9215},
		{2329344, 113.7215},
		{2330208, 114.0399},
		{2331072, 113.8491},
		{2334528, 113.8874},
		{2335392, 114.0604},
		{2336256, 114.169},
		{2337120, 114.3358},
		{2339712, 114.1685},
		{2340576, 114.3127},
		{2341440, 113.9939},
		{2342304, 113.9263},
		{2343168, 113.9104},
		{2345760, 113.9126},
		{2346624, 113.8308},
		{2347488, 113.8194},
		{2348352, 113.7965},
		{2349216, 113.7033},
		{2351808, 113.928},
		{2352672, 113.9731},
		{2353536, 113.0674},
		{2355264, 113.236},
		{2357856, 113.6718},
		{2358720, 113.6227},
		{2359584, 113.2281},
		{2360448, 113.1163},
		{2361312, 113.3912},
		{2363904, 113.2199},
		{2364768, 113.2106},
		{2365632, 113.4242},
		{2366496, 113.4562},
		{2367360, 113.2792},
		{2369952, 113.3628},
		{2370816, 113.4733},
		{2371680, 113.3569},
		{2372544, 113.6042},
		{2373408, 113.8198},
		{2376000, 113.6272},
		{2376864, 113.6932},
		{2377728, 113.5378},
		{2378592, 113.605},
		{2379456, 113.1918},
		{2382048, 113.4641},
		{2382912, 113.5474},
		{2383776, 113.6323},
		{2384640, 113.7351},
		{2385504, 113.5438},
		{2388096, 113.4812},
		{2388960, 113.4081},
		{2389824, 113.3496},
		{2390688, 113.463},
		{2391552, 113.5413},
		{2395008, 113.3216},
		{2395872, 113.2955},
		{2396736, 113.449},
		{2397600, 113.4903},
		{2400192, 113.4748},
		{2401056, 113.368},
		{2401920, 113.0084},
		{2402784, 113.2377},
		{2403648, 113.0932},
		{2406240, 112.849},
		{2407104, 113.065},
		{2407968, 112.9253},
		{2408832, 112.7213},
		{2409696, 112.6057},
		{2412288, 112.7316},
		{2413152, 112.7443},
		{2414016, 112.814},
		{2414880, 113.1031},
		{2415744, 113.0471},
		{2418336, 113.007},
		{2419200, 112.9169},
		{2420064, 112.81},
		{2420928, 112.8753},
		{2421792, 112.8107},
		{2424384, 112.751},
		{2425248, 112.8041},
		{2426112, 112.8435},
		{2426976, 112.8071},
		{2427840, 112.5604},
		{2430432, 112.5554},
		{2431296, 112.7537},
		{2432160, 112.7501},
		{2433024, 112.9404},
		{2433888, 112.87},
		{2436480, 112.4239},
		{2437344, 112.4615},
		{2438208, 112.3021},
		{2439072, 112.2373},
		{2439936, 112.2463},
		{2442528, 112.3458},
		{2443392, 112.1616},
		{2444256, 112.0795},
		{2445120, 112.2538},
		{2445984, 112.0455},
		{2448576, 111.8063},
		{2449440, 111.9287},
		{2450304, 111.5843},
		{2451168, 111.5323},
		{2452032, 111.3175},
		{2454624, 111.3217},
		{2455488, 111.4877},
		{2456352, 111.4155},
		{2457216, 111.2488},
		{2458080, 111.3365},
		{2460672, 110.9546},
		{2461536, 111.0391},
		{2462400, 111.0748},
		{2463264, 111.0707},
		{2464128, 111.0386},
		{2466720, 111.0133},
		{2467584, 111.0016},
		{2468448, 110.9252},
		{2469312, 110.4579},
		{2470176, 110.5078},
		{2472768, 110.4573},
		{2473632, 110.5053},
		{2474496, 111.1033},
		{2475360, 110.3904},
		{2476224, 110.6321},
		{2478816, 110.781},
		{2479680, 110.6668},
		{2480544, 110.0632},
		{2481408, 110.2068},
		{2482272, 110.6023},
		{2484864, 110.6583},
		{2485728, 110.6476},
		{2486592, 110.3927},
		{2487456, 110.3899},
		{2488320, 110.3326},
		{2490912, 110.3343},
		{2491776, 110.8368},
		{2492640, 110.649},
		{2493504, 110.7764},
		{2494368, 110.4557},
		{2496960, 110.4534},
		{2497824, 110.4717},
		{2498688, 110.5583},
		{2499552, 110.767},
		{2500416, 110.7257},
		{2503008, 110.7946},
		{2503872, 110.8261},
		{2504736, 110.5083},
		{2505600, 110.5969},
		{2506464, 110.6646},
		{2509056, 110.6599},
		{2509920, 110.4771},
		{2510784, 110.4055},
		{2511648, 110.4247},
		{2512512, 110.4944},
		{2515104, 110.482},
		{2515968, 109.8768},
		{2516832, 109.7263},
		{2517696, 109.844},
		{2521152, 109.7689},
		{2522016, 109.3749},
		{2522880, 109.6253},
		{2523744, 110.0464},
		{2528064, 110.1024},
		{2528928, 110.8808},
		{2529792, 110.7331},
		{2530656, 110.9764},
		{2533248, 110.8468},
		{2534112, 111.0046},
		{2534976, 110.0213},
		{2535840, 110.248},
		{2536704, 110.2238},
		{2539296, 110.0944},
		{2540160, 110.1695},
		{2541024, 110.4196},
		{2541888, 110.2671},
		{2542752, 110.3676},
		{2545344, 110.2034},
		{2546208, 110.4198},
		{2547072, 110.2142},
		{2547936, 110.0765},
		{2548800, 110.0272},
		{2551392, 109.9967},
		{2552256, 109.6794},
		{2553120, 109.9269},
		{2553984, 110.2796},
		{2554848, 110.2246},
		{2557440, 110.1111},
		{2558304, 109.9627},
		{2559168, 110.1534},
		{2560032, 110.1868},
		{2560896, 110.0135},
		{2563488, 109.3644},
		{2564352, 108.5471},
		{2565216, 108.423},
		{2566080, 108.6453},
		{2566944, 108.2042},
		{2569536, 108.3584},
		{2570400, 108.4954},
		{2571264, 108.415},
		{2572128, 108.4612},
		{2572992, 108.57},
		{2575584, 108.5554},
		{2576448, 108.5612},
		{2577312, 108.2921},
		{2578176, 108.4981},
		{2579040, 108.3873},
		{2581632, 108.6844},
		{2582496, 108.9046},
		{2583360, 109.1044},
		{2584224, 108.8426},
		{2585088, 108.7425},
		{2587680, 108.7808},
		{2588544, 108.7901},
		{2589408, 108.7937},
		{2590272, 108.7092},
		{2591136, 108.7414},
		{2593728, 108.7915},
		{2594592, 108.7033},
		{2595456, 108.8584},
		{2596320, 108.9372},
		{2597184, 108.5762},
		{2599776, 108.8158},
		{2600640, 108.7469},
		{2601504, 109.0011},
		{2602368, 108.5751},
		{2603232, 108.5766},
		{2605824, 108.3917},
		{2606688, 108.4131},
		{2607552, 108.3128},
		{2608416, 108.4294},
		{2609280, 108.269},
		{2611872, 108.3202},
		{2612736, 108.5158},
		{2613600, 108.338},
		{2614464, 108.2456},
		{2615328, 108.1131},
		{2617920, 108.1202},
		{2618784, 108.1165},
		{2619648, 108.0776},
		{2624832, 107.8391},
		{2625696, 107.9443},
		{2626560, 107.9467},
		{2627424, 107.7575},
		{2630880, 108.2018},
		{2631744, 107.819},
		{2632608, 107.8218},
		{2633472, 107.8627},
		{2636064, 108.3707},
		{2636928, 108.6993},
		{2637792, 108.6034},
		{2638656, 108.3609},
		{2639520, 108.3126},
		{2642112, 108.234},
		{2642976, 108.1498},
		{2643840, 108.2952},
		{2644704, 108.2167},
		{2645568, 108.3239},
		{2648160, 108.2804},
		{2649024, 108.3612},
		{2649888, 108.393},
		{2650752, 108.4996},
		{2651616, 108.6825},
		{2654208, 108.7257},
		{2655072, 108.8014},
		{2655936, 108.6195},
		{2657664, 108.3861},
		{2660256, 108.8775},
		{2661120, 109.5926},
		{2661984, 109.9981},
		{2662848, 109.6085},
		{2663712, 109.7251},
		{2667168, 110.1697},
		{2668032, 110.9966},
		{2668896, 111.5609},
		{2669760, 111.2375},
		{2672352, 111.1238},
		{2673216, 111.049},
		{2674080, 111.0321},
		{2674944, 111.5191},
		{2675808, 112.4762},
		{2678400, 111.2923},
		{2679264, 111.6621},
		{2680128, 111.7617},
		{2680992, 111.7971},
		{2681856, 111.8756},
		{2684448, 111.7542},
		{2685312, 111.8584},
		{2686176, 111.9453},
		{2687040, 112.0547},
		{2687904, 111.9328},
		{2690496, 111.9133},
		{2691360, 111.9869},
		{2692224, 112.088},
		{2693088, 114.5245},
		{2693952, 115.9961},
		{2696544, 114.919},
		{2697408, 114.1297},
		{2698272, 113.5898},
		{2699136, 113.2883},
		{2700000, 113.414},
		{2702592, 113.4199},
		{2703456, 113.2103},
		{2704320, 113.3075},
		{2705184, 112.7348},
		{2706048, 112.3063},
		{2708640, 112.3534},
		{2709504, 112.0152},
		{2710368, 112.0492},
		{2711232, 112.9664},
		{2712096, 114.1402},
		{2715552, 113.3345},
		{2716416, 113.2414},
		{2717280, 113.1459},
		{2718144, 112.9847},
		{2720736, 113.2487},
		{2721600, 113.1925},
		{2722464, 113.1883},
		{2723328, 113.2648},
		{2724192, 113.3837},
		{2726784, 113.47},
		{2727648, 113.0283},
		{2728512, 113.0276},
		{2729376, 113.322},
		{2730240, 113.0445},
		{2732832, 113.1628},
		{2733696, 113.34},
		{2734560, 113.0286},
		{2735424, 113.2075},
		{2736288, 113.1188},
		{2738880, 113.4389},
		{2739744, 113.4326},
		{2740608, 113.9157},
		{2741472, 113.9956},
		{2742336, 114.3092},
		{2744928, 114.1291},
		{2745792, 114.1482},
		{2746656, 114.2283},
		{2747520, 114.3886},
		{2748384, 114.5083},
		{2750976, 114.9301},
		{2751840, 115.3364},
		{2752704, 115.1794},
		{2753568, 115.2525},
		{2754432, 115.5571},
		{2757024, 115.5836},
		{2757888, 115.49},
		{2758752, 115.0913},
		{2759616, 115.5781},
		{2760480, 115.4162},
		{2763072, 115.1532},
		{2763936, 115.4479},
		{2764800, 115.4886},
		{2765664, 115.5709},
		{2766528, 115.7011},
		{2769120, 115.7987},
		{2769984, 115.5742},
		{2770848, 115.8169},
		{2771712, 116.0072},
		{2772576, 115.9143},
		{2775168, 115.9303},
		{2776032, 116.0189},
		{2776896, 116.1929},
		{2777760, 116.3825},
		{2778624, 116.4877},
		{2781216, 116.3929},
		{2782080, 116.8781},
		{2782944, 117.1953},
		{2783808, 117.7512},
		{2784672, 118.4291},
		{2787264, 118.3907},
		{2788128, 118.0651},
		{2788992, 117.3937},
		{2789856, 117.1411},
		{2790720, 117.6525},
		{2793312, 117.4278},
		{2794176, 118.0546},
		{2795040, 118.4746},
		{2795904, 118.7573},
		{2796768, 118.8396},
		{2799360, 119.0389},
		{2800224, 119.2117},
		{2801088, 119.219},
		{2801952, 119.6332},
		{2802816, 119.766},
		{2805408, 119.9194},
		{2806272, 120.9478},
		{2807136, 121.0997},
		{2808000, 121.4032},
		{2808864, 118.5507},
		{2811456, 118.6964},
		{2812320, 119.3429},
		{2813184, 119.4878},
		{2814048, 119.5726},
		{2814912, 119.6963},
		{2817504, 119.8701},
		{2818368, 120.2079},
		{2819232, 120.3476},
		{2820096, 120.4768},
		{2820960, 120.5896},
		{2823552, 120.6789},
		{2824416, 120.6496},
		{2825280, 120.5042},
		{2826144, 120.7827},
		{2827008, 120.8682},
		{2829600, 120.885},
		{2830464, 120.882},
		{2831328, 120.5533},
		{2832192, 120.677},
		{2833056, 120.5622},
		{2837376, 120.4305},
		{2838240, 120.731},
		{2839104, 120.8381},
		{2843424, 121.308},
		{2844288, 121.4054},
		{2845152, 121.3531},
		{2847744, 121.4046},
		{2848608, 121.3434},
		{2849472, 121.3413},
		{2850336, 121.4999},
		{2851200, 121.4231},
		{2853792, 121.4786},
		{2854656, 121.5333},
		{2855520, 121.5846},
		{2856384, 121.7858},
		{2857248, 121.7167},
		{2859840, 121.7247},
		{2860704, 122.0386},
		{2861568, 121.8044},
		{2862432, 121.9575},
		{2863296, 121.7368},
		{2865888, 121.9166},
		{2866752, 121.9325},
		{2867616, 122.0234},
		{2868480, 122.1653},
		{2869344, 121.1232},
		{2871936, 120.9555},
		{2872800, 121.2014},
		{2873664, 121.3724},
		{2874528, 121.6089},
		{2875392, 121.1408},
		{2877984, 121.144},
		{2878848, 121.2233},
		{2879712, 121.2845},
		{2880576, 121.5399},
		{2881440, 121.2799},
		{2884032, 121.541},
		{2884896, 121.5504},
		{2885760, 121.5063},
		{2886624, 121.5112},
		{2887488, 121.5651},
		{2890080, 121.5693},
		{2890944, 121.7167},
		{2891808, 121.7106},
		{2892672, 121.6876},
		{2893536, 121.7452},
		{2896128, 121.7545},
		{2896992, 121.9073},
		{2897856, 121.8854},
		{2898720, 121.6981},
		{2899584, 121.8195},
		{2902176, 121.8604},
		{2903040, 121.9211},
		{2903904, 121.9278},
		{2904768, 122.0311},
		{2905632, 122.0424},
		{2908224, 121.9836},
		{2909088, 121.9395},
		{2909952, 121.8388},
		{2910816, 121.905},
		{2911680, 121.9895},
		{2914272, 122.5622},
		{2915136, 123.6267},
		{2916000, 126.9841},
		{2916864, 124.5116},
		{2917728, 125.1708},
		{2920320, 125.7317},
		{2921184, 126.2179},
		{2922048, 127.1836},
		{2922912, 128.255},
		{2923776, 128.1813},
		{2926368, 128.181},
		{2927232, 128.1886},
		{2928096, 128.0534},
		{2933280, 128.4793},
		{2934144, 128.8226},
		{2935872, 129.5025},
		{2938464, 129.9954},
		{2939328, 131.0379},
		{2940192, 130.1948},
		{2941056, 130.3927},
		{2941920, 130.886},
		{2944512, 133.0513},
		{2946240, 138.2855},
		{2947104, 141.3542},
		{2947968, 142.011},
		{2950560, 140.4958},
		{2951424, 138.8886},
		{2952288, 134.4775},
		{2953152, 134.2289},
		{2954016, 135.0235},
		{2956608, 134.1363},
		{2957472, 135.4095},
		{2958336, 136.9224},
		{2959200, 137.1598},
		{2960064, 137.6977},
		{2962656, 137.6599},
		{2963520, 137.4048},
		{2964384, 137.2735},
		{2966112, 137.114},
		{2968704, 137.3708},
		{2969568, 139.1772},
		{2970432, 140.2391},
		{2971296, 141.4},
		{2972160, 138.9315},
		{2975616, 141.4776},
		{2976480, 140.6586},
		{2977344, 139.475},
		{2978208, 139.6327},
		{2980800, 139.9893},
		{2981664, 140.1341},
		{2982528, 140.6983},
		{2983392, 141.5151},
		{2984256, 142.3905},
		{2986848, 142.4275},
		{2987712, 142.7807},
		{2988576, 144.5008},
		{2989440, 139.9157},
		{2990304, 141.9614},
		{2992896, 140.9045},
		{2993760, 139.844},
		{2994624, 139.6881},
		{2995488, 140.1637},
		{2996352, 140.6041},
		{2998944, 139.1072},
		{2999808, 139.2206},
		{3000672, 139.7105},
		{3001536, 138.4455},
		{3002400, 137.6209},
		{3004992, 137.7953},
		{3005856, 137.7704},
		{3006720, 137.8026},
		{3007584, 137.9669},
		{3008448, 138.1516},
		{3011040, 138.1219},
		{3011904, 138.1223},
		{3012768, 138.1211},
		{3013632, 138.1279},
		{3014496, 138.0916},
		{3017088, 138.1252},
		{3017952, 138.2231},
		{3018816, 138.1269},
		{3019680, 137.8723},
		{3020544, 136.9635},
		{3023136, 137.0931},
		{3024000, 137.0439},
		{3024864, 135.0664},
		{3025728, 134.9446},
		{3026592, 134.8158},
		{3030048, 134.8713},
		{3030912, 134.8674},
		{3031776, 134.9767},
		{3032640, 135.7456},
		{3035232, 135.944},
		{3036096, 137.5219},
		{3036960, 137.3946},
		{3037824, 137.3927},
		{3038688, 136.3548},
		{3041280, 136.5402},
		{3042144, 137.3973},
		{3043008, 137.8485},
		{3043872, 137.7699},
		{3044736, 138.4426},
		{3047328, 138.5117},
		{3048192, 138.5739},
		{3049056, 138.502},
		{3049920, 138.5105},
		{3050784, 138.2437},
		{3053376, 137.9167},
		{3054240, 138.0589},
		{3055104, 137.7508},
		{3055968, 138.1793},
		{3056832, 138.2161},
		{3059424, 138.6033},
		{3060288, 138.6038},
		{3061152, 138.9526},
		{3062016, 139.3756},
		{3062880, 138.8471},
		{3065472, 139.5007},
		{3066336, 141.0292},
		{3067200, 140.9081},
		{3068064, 141.3428},
		{3068928, 141.3684},
		{3071520, 142.7627},
		{3072384, 142.6969},
		{3073248, 142.5168},
		{3074112, 142.8512},
		{3074976, 142.2506},
		{3077568, 140.8693},
		{3078432, 141.2097},
		{3079296, 141.9374},
		{3080160, 142.2896},
		{3081024, 144.4438},
		{3083616, 143.7232},
		{3084480, 144.4013},
		{3085344, 143.4504},
		{3086208, 140.5713},
		{3087072, 140.1787},
		{3089664, 140.3381},
		{3090528, 141.32},
		{3091392, 142.3151},
		{3092256, 142.9711},
		{3093120, 142.9502},
		{3095712, 142.7184},
		{3096576, 144.0945},
		{3097440, 144.4004},
		{3098304, 144.4145},
		{3099168, 144.2934},
		{3101760, 145.0342},
		{3102624, 144.832},
		{3103488, 144.4861},
		{3104352, 145.9012},
		{3105216, 145.4278},
		{3107808, 146.4131},
		{3108672, 147.4409},
		{3109536, 146.7752},
		{3110400, 146.2453},
		{3111264, 144.6746},
		{3113856, 145.3614},
		{3114720, 147.6478},
		{3115584, 147.5782},
		{3116448, 146.5677},
		{3117312, 146.6047},
		{3119904, 147.9653},
		{3120768, 147.8702},
		{3121632, 149.4729},
		{3122496, 149.9376},
		{3123360, 150.9327},
		{3125952, 150.9962},
		{3126816, 150.7508},
		{3127680, 151.1638},
		{3128544, 150.5726},
		{3129408, 148.1361},
		{3132000, 147.1669},
		{3132864, 147.755},
		{3133728, 149.2046},
		{3134592, 149.4084},
		{3135456, 149.827},
		{3138048, 148.5366},
		{3138912, 147.5436},
		{3139776, 146.4789},
		{3140640, 144.0321},
		{3141504, 144.0029},
		{3144096, 141.9136},
		{3144960, 141.4663},
		{3145824, 140.8824},
		{3146688, 140.3786},
		{3147552, 140.4286},
		{3152736, 139.7471},
		{3153600, 142.0917},
		{3156192, 141.7985},
		{3158784, 140.6689},
		{3159648, 139.6841},
		{3162240, 139.9567},
		{3163104, 140.9447},
		{3163968, 140.4108},
		{3164832, 140.6212},
		{3165696, 141.0972},
		{3168288, 140.6849},
		{3169152, 141.559},
		{3170016, 141.4094},
		{3170880, 141.2019},
		{3171744, 141.2044},
		{3174336, 139.8486},
		{3175200, 139.3625},
		{3176064, 139.9944},
		{3176928, 139.8449},
		{3177792, 139.9896},
		{3180384, 139.897},
		{3181248, 139.9027},
		{3182112, 139.9008},
		{3182976, 139.4521},
		{3183840, 139.2058},
		{3186432, 138.3235},
		{3187296, 138.3286},
		{3188160, 138.3422},
		{3189024, 138.515},
		{3189888, 138.58},
		{3192480, 138.207},
		{3193344, 137.7748},
		{3194208, 138.0336},
		{3195072, 137.6356},
		{3195936, 137.4821},
		{3198528, 137.4975},
		{3199392, 136.7532},
		{3200256, 137.3394},
		{3201120, 137.4909},
		{3201984, 137.6646},
		{3204576, 137.9274},
		{3205440, 137.0897},
		{3206304, 136.8181},
		{3207168, 136.353},
		{3208032, 136.2494},
		{3210624, 136.0512},
		{3211488, 136.7267},
		{3212352, 137.3948},
		{3213216, 138.7829},
		{3214080, 137.2056},
		{3216672, 137.0108},
		{3217536, 137.7899},
		{3218400, 137.8014},
		{3219264, 137.307},
		{3220128, 137.3086},
		{3222720, 137.4206},
		{3223584, 138.1813},
		{3224448, 137.8363},
		{3225312, 138.4193},
		{3226176, 136.9616},
		{3228768, 137.2109},
		{3229632, 136.6725},
		{3230496, 136.3313},
		{3235680, 136.3672},
		{3236544, 136.2992},
		{3237408, 136.4056},
		{3238272, 136.401},
		{3240864, 136.5355},
		{3241728, 135.6009},
		{3242592, 135.4242},
		{3243456, 135.1919},
		{3244320, 135.0344},
		{3246912, 133.7338},
		{3247776, 133.7509},
		{3248640, 133.7303},
		{3249504, 134.2455},
		{3250368, 133.5826},
		{3252960, 133.2552},
		{3253824, 133.0066},
		{3254688, 130.6994},
		{3256416, 130.1272},
		{3259008, 131.0534},
		{3259872, 131.0892},
		{3261600, 130.0803},
		{3262464, 129.4923},
		{3265056, 128.5419},
		{3265920, 127.4912},
		{3266784, 128.2552},
		{3268512, 128.7958},
		{3271104, 128.1293},
		{3271968, 128.2949},
		{3272832, 128.0075},
		{3273696, 128.4391},
		{3274560, 129.1861},
		{3278016, 129.3558},
		{3278880, 131.2189},
		{3279744, 131.8463},
		{3280608, 130.5234},
		{3283200, 130.3813},
		{3284064, 131.2028},
		{3284928, 131.206},
		{3285792, 131.2588},
		{3286656, 130.6761},
		{3289248, 130.308},
		{3290112, 129.6497},
		{3290976, 129.6778},
		{3291840, 129.1615},
		{3292704, 128.6083},
		{3295296, 128.5664},
		{3296160, 128.7604},
		{3297024, 129.2721},
		{3297888, 128.838},
		{3298752, 128.7319},
		{3302208, 128.5528},
		{3303072, 129.4577},
		{3303936, 129.2019},
		{3304800, 129.2967},
		{3307392, 129.5627},
		{3308256, 129.4144},
		{3309120, 130.453},
		{3309984, 129.3627},
		{3310848, 128.9138},
		{3313440, 128.8975},
		{3314304, 128.5007},
		{3315168, 127.5711},
		{3316032, 127.6018},
		{3316896, 127.7288},
		{3319488, 127.9991},
		{3320352, 128.165},
		{3321216, 128.0475},
		{3322080, 128.7246},
		{3322944, 128.2747},
		{3325536, 127.6523},
		{3326400, 126.4012},
		{3327264, 125.6637},
		{3328128, 125.9671},
		{3328992, 126.0163},
		{3331584, 127.2136},
		{3332448, 127.2903},
		{3333312, 127.0398},
		{3334176, 127.8542},
		{3335040, 127.6552},
		{3337632, 126.8205},
		{3338496, 126.2915},
		{3339360, 126.4602},
		{3340224, 125.1854},
		{3341088, 126.3431},
		{3344544, 125.7236},
		{3345408, 124.6461},
		{3346272, 125.3036},
		{3347136, 125.5725},
		{3349728, 125.5432},
		{3350592, 127.2147},
		{3351456, 127.2573},
		{3352320, 126.6158},
		{3353184, 126.569},
		{3355776, 126.2057},
		{3356640, 126.3582},
		{3357504, 126.5639},
		{3358368, 126.6054},
		{3359232, 126.7131},
		{3361824, 127.3487},
		{3362688, 128.1251},
		{3363552, 129.3343},
		{3364416, 130.6416},
		{3365280, 130.3313},
		{3367872, 130.6792},
		{3368736, 131.5964},
		{3369600, 131.6842},
		{3370464, 130.0391},
		{3371328, 129.5268},
		{3373920, 129.4847},
		{3374784, 128.2482},
		{3375648, 127.8167},
		{3376512, 127.9602},
		{3377376, 128.1118},
		{3379968, 128.4113},
		{3380832, 129.5024},
		{3381696, 130.0045},
		{3382560, 130.5971},
		{3383424, 130.647},
		{3386016, 129.6465},
		{3386880, 128.7095},
		{3387744, 128.4669},
		{3388608, 129.5423},
		{3389472, 128.5912},
		{3392064, 128.7483},
		{3392928, 128.1725},
		{3393792, 128.0104},
		{3394656, 128.1182},
		{3395520, 128.3756},
		{3398112, 128.6883},
		{3398976, 129.273},
		{3399840, 129.1143},
		{3400704, 129.6349},
		{3401568, 129.6071},
		{3404160, 129.6426},
		{3405024, 129.8439},
		{3405888, 130.3858},
		{3406752, 130.4912},
		{3407616, 130.36},
		{3410208, 130.3086},
		{3411072, 130.2872},
		{3411936, 130.6322},
		{3412800, 130.5882},
		{3413664, 131.0175},
		{3416256, 130.2441},
		{3417120, 130.7293},
		{3417984, 130.0691},
		{3418848, 129.8581},
		{3419712, 129.3853},
		{3422304, 129.6086},
		{3423168, 130.5527},
		{3424032, 130.1307},
		{3424896, 130.4049},
		{3425760, 129.8928},
		{3428352, 129.544},
		{3429216, 128.839},
		{3430080, 128.55},
		{3430944, 128.5851},
		{3431808, 128.6215},
		{3434400, 128.3232},
		{3435264, 128.3066},
		{3436128, 128.6416},
		{3436992, 129.1036},
		{3437856, 129.0061},
		{3440448, 129.1746},
		{3441312, 128.8219},
		{3442176, 128.4391},
		{3443040, 128.359},
		{3443904, 128.1995},
		{3446496, 127.7512},
		{3447360, 127.5153},
		{3448224, 127.6107},
		{3449088, 127.5459},
		{3449952, 128.1371},
		{3452544, 127.3369},
		{3453408, 127.3617},
		{3454272, 127.4478},
		{3455136, 126.3095},
		{3456000, 126.0781},
		{3458592, 126.323},
		{3459456, 126.2051},
		{3460320, 125.8779},
		{3461184, 126.2242},
		{3462048, 126.4897},
		{3464640, 126.6198},
		{3468096, 125.8253},
		{3470688, 124.5218},
		{3471552, 124.8994},
		{3474144, 124.75},
		{3476736, 125.3781},
		{3477600, 124.7236},
		{3478464, 124.9246},
		{3479328, 125.2559},
		{3480192, 124.8857},
		{3482784, 123.7997},
		{3483648, 123.5872},
		{3484512, 123.8527},
		{3485376, 123.6026},
		{3486240, 123.4713},
		{3488832, 123.6211},
		{3489696, 123.787},
		{3490560, 123.9803},
		{3491424, 123.7573},
		{3492288, 123.675},
		{3494880, 123.8098},
		{3495744, 123.5505},
		{3496608, 123.447},
		{3497472, 122.5108},
		{3498336, 121.3595},
		{3500928, 121.5087},
		{3501792, 121.237},
		{3502656, 119.8295},
		{3503520, 121.1471},
		{3504384, 119.4581},
		{3506976, 121.1472},
		{3507840, 120.954},
		{3508704, 121.1423},
		{3509568, 120.7453},
		{3510432, 121.5537},
		{3513024, 122.3537},
		{3513888, 124.0349},
		{3514752, 123.7568},
		{3515616, 122.6474},
		{3516480, 122.6246},
		{3519072, 122.4524},
		{3519936, 122.685},
		{3520800, 122.1624},
		{3521664, 121.5517},
		{3522528, 121.4139},
		{3525120, 121.2098},
		{3525984, 121.6202},
		{3526848, 121.6272},
		{3527712, 121.2404},
		{3528576, 121.6183},
		{3531168, 121.5707},
		{3532032, 122.0048},
		{3532896, 122.4509},
		{3533760, 121.7186},
		{3534624, 122.0289},
		{3537216, 122.2259},
		{3538080, 122.6311},
		{3538944, 122.9345},
		{3539808, 123.1201},
		{3540672, 123.8982},
		{3543264, 123.0161},
		{3544128, 121.5277},
		{3544992, 122.2764},
		{3545856, 122.2163},
		{3546720, 119.9507},
		{3549312, 120.2248},
		{3550176, 121.1938},
		{3551040, 120.5845},
		{3551904, 120.7213},
		{3552768, 121.8138},
		{3555360, 121.8295},
		{3556224, 121.8122},
		{3557088, 121.3739},
		{3557952, 120.8074},
		{3558816, 120.8723},
		{3561408, 120.7523},
		{3562272, 120.1766},
		{3563136, 119.7259},
		{3568320, 119.5149},
		{3569184, 119.2795},
		{3570912, 119.8023},
		{3573504, 119.1297},
		{3574368, 119.5964},
		{3575232, 119.5992},
		{3576960, 119.029},
		{3579552, 118.8891},
		{3580416, 118.9573},
		{3581280, 118.7188},
		{3582144, 119.3959},
		{3583008, 119.3027},
		{3585600, 118.5781},
		{3586464, 118.2927},
		{3587328, 118.4783},
		{3588192, 118.7817},
		{3589056, 118.5882},
		{3591648, 120.3533},
		{3592512, 120.7073},
		{3593376, 120.4309},
		{3594240, 119.1115},
		{3595104, 117.9764},
		{3597696, 118.2525},
		{3598560, 118.9715},
		{3599424, 119.4222},
		{3601152, 118.3679},
		{3603744, 118.7421},
		{3604608, 119.2922},
		{3605472, 120.1625},
		{3606336, 119.8336},
		{3607200, 120.191},
		{3610656, 119.856},
		{3611520, 120.3854},
		{3612384, 121.1697},
		{3613248, 120.9595},
		{3615840, 121.3996},
		{3617568, 120.9145},
		{3618432, 120.6036},
		{3619296, 121.2223},
		{3621888, 122.3733},
		{3622752, 122.5891},
		{3623616, 124.8332},
		{3624480, 123.7012},
		{3625344, 123.4993},
		{3627936, 123.6959},
		{3628800, 124.0046},
		{3629664, 123.9035},
		{3630528, 124.3301},
		{3631392, 124.0705},
		{3633984, 123.8636},
		{3634848, 124.823},
		{3635712, 124.7933},
		{3636576, 124.21},
		{3637440, 123.4177},
		{3640032, 123.7399},
		{3640896, 123.8721},
		{3641760, 123.7967},
		{3642624, 123.6678},
		{3643488, 123.9307},
		{3646080, 124.1615},
		{3646944, 124.6556},
		{3647808, 124.2223},
		{3648672, 124.4156},
		{3649536, 124.2016},
		{3652128, 123.791},
		{3652992, 123.6183},
		{3653856, 123.708},
		{3654720, 124.3642},
		{3655584, 124.397},
		{3659040, 124.4366},
		{3659904, 124.6904},
		{3660768, 124.7064},
		{3661632, 125.4477},
		{3664224, 125.5091},
		{3665088, 126.579},
		{3665952, 127.9456},
		{3666816, 127.3315},
		{3667680, 127.1863},
		{3670272, 126.775},
		{3671136, 126.5761},
		{3672000, 127.2702},
		{3672864, 127.4281},
		{3673728, 127.7714},
		{3676320, 128.6392},
		{3677184, 128.7018},
		{3678048, 127.3913},
		{3678912, 124.9947},
		{3679776, 125.7191},
		{3682368, 126.0664},
		{3683232, 127.6282},
		{3684096, 127.4744},
		{3684960, 127.1156},
		{3685824, 126.9528},
		{3688416, 127.3961},
		{3689280, 127.836},
		{3690144, 128.2038},
		{3691008, 126.5903},
		{3691872, 125.8359},
		{3694464, 126.1633},
		{3695328, 126.5027},
		{3696192, 126.9416},
		{3697056, 126.3683},
		{3697920, 125.9111},
		{3700512, 125.9123},
		{3701376, 125.7212},
		{3702240, 125.4043},
		{3703104, 126.6575},
		{3703968, 125.6327},
		{3706560, 125.7106},
		{3707424, 125.2589},
		{3708288, 125.0133},
		{3709152, 125.1687},
		{3710016, 125.7377},
		{3712608, 125.6509},
		{3713472, 126.1754},
		{3714336, 125.7004},
		{3715200, 125.8563},
		{3716064, 125.2531},
		{3718656, 124.8772},
		{3719520, 125.4245},
		{3720384, 125.4098},
		{3721248, 126.1954},
		{3722112, 126.1639},
		{3724704, 125.9217},
		{3725568, 125.6998},
		{3726432, 125.4646},
		{3727296, 126.1141},
		{3728160, 126.0712},
		{3730752, 126.4749},
		{3731616, 126.152},
		{3732480, 125.9113},
		{3733344, 125.2099},
		{3734208, 125.6253},
		{3736800, 125.5602},
		{3737664, 125.0187},
		{3738528, 124.1579},
		{3739392, 124.3492},
		{3740256, 124.6251},
		{3742848, 124.9292},
		{3743712, 124.9952},
		{3744576, 124.9349},
		{3745440, 125.1333},
		{3746304, 125.2141},
		{3748896, 125.6063},
		{3749760, 125.5167},
		{3750624, 125.7398},
		{3751488, 124.9157},
		{3752352, 124.9295},
		{3754944, 125.1318},
		{3755808, 125.6197},
		{3756672, 125.6658},
		{3757536, 125.6254},
		{3758400, 125.1435},
		{3760992, 125.3541},
		{3761856, 125.1925},
		{3762720, 124.5474},
		{3763584, 124.8814},
		{3764448, 124.6826},
		{3767040, 124.9739},
		{3767904, 125.1151},
		{3768768, 124.861},
		{3769632, 124.9525},
		{3770496, 125.1021},
		{3773088, 124.817},
		{3773952, 124.9067},
		{3774816, 124.6607},
		{3775680, 124.6099},
		{3776544, 124.4061},
		{3779136, 123.6731},
		{3780000, 124.1408},
		{3785184, 123.5385},
		{3786048, 123.5254},
		{3786912, 123.4179},
		{3791232, 122.3595},
		{3792096, 121.4876},
		{3792960, 121.8182},
		{3793824, 121.9627},
		{3794688, 121.8998},
		{3797280, 121.7167},
		{3798144, 121.9102},
		{3799008, 121.6172},
		{3799872, 121.3443},
		{3800736, 120.9295},
		{3803328, 119.4377},
		{3804192, 119.0607},
		{3805056, 119.6062},
		{3805920, 119.8405},
		{3806784, 119.2679},
		{3809376, 119.2815},
		{3810240, 119.27},
		{3811104, 119.6542},
		{3811968, 119.9627},
		{3812832, 119.5193},
		{3815424, 119.4732},
		{3816288, 119.3735},
		{3817152, 119.3316},
		{3818016, 118.6983},
		{3818880, 118.781},
		{3821472, 119.2347},
		{3822336, 119.3518},
		{3823200, 119.5184},
		{3824064, 118.96},
		{3824928, 118.9339},
		{3827520, 119.2067},
		{3828384, 119.5357},
		{3829248, 118.8675},
		{3830112, 118.9685},
		{3830976, 118.6718},
		{3833568, 119.5109},
		{3834432, 119.7737},
		{3835296, 119.6246},
		{3836160, 120.2886},
		{3837024, 119.9252},
		{3839616, 119.9625},
		{3840480, 120.3869},
		{3841344, 121.0625},
		{3842208, 121.2805},
		{3843072, 121.2327},
		{3845664, 121.1307},
		{3846528, 120.7201},
		{3847392, 120.1093},
		{3848256, 120.2009},
		{3849120, 120.2605},
		{3851712, 120.001},
		{3852576, 120.377},
		{3853440, 120.4228},
		{3854304, 120.4043},
		{3855168, 121.2472},
		{3857760, 121.782},
		{3858624, 122.9173},
		{3859488, 122.9467},
		{3860352, 122.5262},
		{3861216, 122.3985},
		{3863808, 122.2702},
		{3864672, 123.3464},
		{3865536, 123.6194},
		{3866400, 123.1327},
		{3867264, 123.1651},
		{3869856, 122.9644},
		{3870720, 122.955},
		{3871584, 122.923},
		{3876768, 122.9029},
		{3877632, 123.442},
		{3878496, 123.9203},
		{3879360, 122.8593},
		{3881952, 122.8146},
		{3882816, 122.8654},
		{3883680, 122.9452},
		{3885408, 123.0632},
		{3888000, 123.2668},
		{3888864, 123.1571},
		{3889728, 123.2945},
		{3890592, 123.6371},
		{3891456, 123.7635},
		{3894048, 123.8674},
		{3894912, 123.8227},
		{3895776, 124.7766},
		{3896640, 124.5635},
		{3897504, 123.9242},
		{3900096, 124.0257},
		{3900960, 124.0016},
		{3901824, 124.0626},
		{3902688, 123.6701},
		{3903552, 123.4633},
		{3906144, 123.2765},
		{3907008, 123.3057},
		{3907872, 123.4013},
		{3909600, 123.4246},
		{3912192, 123.2117},
		{3913056, 123.014},
		{3913920, 123.0944},
		{3914784, 122.4991},
		{3915648, 122.4237},
		{3919104, 122.3346},
		{3919968, 122.6931},
		{3920832, 122.5136},
		{3921696, 122.0681},
		{3924288, 121.8124},
		{3925152, 122.1447},
		{3926016, 122.1668},
		{3926880, 122.3798},
		{3927744, 122.5626},
		{3930336, 122.4644},
		{3931200, 122.4352},
		{3932064, 122.5977},
		{3933792, 122.6199},
		{3936384, 123.1052},
		{3937248, 123.329},
		{3938112, 123.205},
		{3938976, 123.3043},
		{3939840, 123.2938},
		{3942432, 123.513},
		{3943296, 123.1792},
		{3944160, 123.4471},
		{3945024, 123.522},
		{3945888, 123.4046},
		{3948480, 123.5251},
		{3949344, 123.3207},
		{3950208, 123.2073},
		{3951072, 123.0205},
		{3951936, 122.2175},
		{3954528, 122.725},
		{3955392, 122.6154},
		{3956256, 122.5277},
		{3957120, 122.6123},
		{3957984, 122.7106},
		{3960576, 122.6076},
		{3961440, 122.709},
		{3962304, 121.8253},
		{3963168, 121.8297},
		{3964032, 121.2245},
		{3966624, 121.4679},
		{3967488, 121.0638},
		{3968352, 121.242},
		{3969216, 121.1113},
		{3970080, 120.8515},
		{3973536, 121.0123},
		{3974400, 121.3208},
		{3975264, 121.07},
		{3976128, 121.3238},
		{3978720, 121.4657},
		{3979584, 121.9693},
		{3980448, 121.9142},
		{3981312, 121.5825},
		{3982176, 120.9695},
		{3984768, 120.9029},
		{3985632, 121.3381},
		{3986496, 122.0074},
		{3987360, 121.8743},
		{3988224, 122.1097},
		{3990816, 121.7391},
		{3991680, 121.7545},
		{3992544, 122.2242},
		{3993408, 122.0068},
		{3994272, 122.1235},
		{3996864, 122.1958},
		{3997728, 122.4332},
		{3998592, 122.6897},
		{3999456, 122.6143},
		{4000320, 122.2055},
		{4002912, 122.7206},
		{4003776, 122.3013},
		{4004640, 122.484},
		{4005504, 122.6169},
		{4006368, 122.5985},
		{4008960, 122.1156},
		{4009824, 122.2009},
		{4010688, 122.1388},
		{4011552, 122.0147},
		{4012416, 121.8176},
		{4015008, 121.5919},
		{4015872, 121.8656},
		{4016736, 121.585},
		{4017600, 121.9763},
		{4018464, 121.873},
		{4021056, 121.571},
		{4021920, 121.6617},
		{4022784, 121.8562},
		{4023648, 121.6704},
		{4024512, 121.9326},
		{4027104, 121.8168},
		{4027968, 122.0034},
		{4028832, 121.715},
		{4029696, 121.5129},
		{4030560, 121.6114},
		{4033152, 121.4208},
		{4034016, 121.5218},
		{4034880, 121.7178},
		{4035744, 121.3161},
		{4036608, 121.2178},
		{4039200, 120.5982},
		{4040064, 120.9701},
		{4040928, 121.0628},
		{4041792, 120.9611},
		{4042656, 120.5155},
		{4045248, 120.6269},
		{4046112, 121.1028},
		{4046976, 121.9351},
		{4047840, 121.7072},
		{4048704, 121.5204},
		{4051296, 120.9042},
		{4052160, 120.9099},
		{4053024, 121.0775},
		{4053888, 120.2811},
		{4054752, 119.7343},
		{4057344, 119.9177},
		{4058208, 119.9089},
		{4059072, 120.3204},
		{4059936, 120.3312},
		{4060800, 119.9249},
		{4063392, 119.5151},
		{4064256, 119.5899},
		{4065120, 119.3089},
		{4065984, 119.1463},
		{4066848, 119.4988},
		{4069440, 119.3093},
		{4070304, 119.3234},
		{4071168, 119.6216},
		{4072032, 119.2373},
		{4072896, 118.6018},
		{4075488, 118.4156},
		{4076352, 117.9687},
		{4077216, 118.0186},
		{4078080, 116.5758},
		{4078944, 115.1666},
		{4081536, 114.8133},
		{4082400, 114.1683},
		{4083264, 112.7771},
		{4084128, 114.5834},
		{4084992, 114.203},
		{4087584, 114.2858},
		{4088448, 114.8959},
		{4089312, 115.4436},
		{4090176, 115.6776},
		{4091040, 115.3647},
		{4093632, 114.619},
		{4094496, 114.5904},
		{4095360, 114.1795},
		{4096224, 113.8346},
		{4099680, 113.6091},
		{4100544, 113.522},
		{4101408, 113.8245},
		{4102272, 112.8354},
		{4103136, 113.0158},
		{4106592, 113.5866},
		{4107456, 113.4374},
		{4108320, 113.3607},
		{4109184, 113.8162},
		{4111776, 113.8201},
		{4112640, 113.6648},
		{4113504, 113.152},
		{4114368, 113.3123},
		{4115232, 112.4713},
		{4117824, 112.3591},
		{4118688, 111.8671},
		{4119552, 111.5276},
		{4120416, 111.8164},
		{4121280, 111.9229},
		{4123872, 111.5147},
		{4124736, 111.7154},
		{4125600, 111.6965},
		{4126464, 111.4955},
		{4127328, 111.2493},
		{4129920, 111.2178},
		{4130784, 110.4899},
		{4131648, 110.2572},
		{4132512, 110.7134},
		{4133376, 111.4535},
		{4135968, 111.9235},
		{4136832, 111.5993},
		{4137696, 111.3618},
		{4138560, 111.2988},
		{4139424, 111.0153},
		{4142016, 111.2348},
		{4142880, 111.2184},
		{4143744, 111.0943},
		{4144608, 110.9434},
		{4145472, 110.703},
		{4148064, 109.5634},
		{4148928, 110.0508},
		{4149792, 109.9694},
		{4150656, 110.0158},
		{4151520, 109.6976},
		{4154112, 109.6569},
		{4154976, 109.8921},
		{4155840, 109.545},
		{4156704, 109.5438},
		{4157568, 109.2673},
		{4160160, 109.3634},
		{4161024, 108.7245},
		{4161888, 108.5539},
		{4162752, 107.7974},
		{4163616, 108.291},
		{4166208, 107.3283},
		{4167072, 107.1584},
		{4167936, 107.0026},
		{4168800, 106.5525},
		{4169664, 106.6056},
		{4172256, 106.4228},
		{4173120, 107.2432},
		{4173984, 109.5266},
		{4179168, 109.4474},
		{4180032, 109.2162},
		{4180896, 108.0787},
		{4181760, 108.0025},
		{4184352, 107.9872},
		{4185216, 107.8795},
		{4186080, 108.3077},
		{4186944, 108.4865},
		{4187808, 108.5576},
		{4190400, 109.2021},
		{4191264, 109.8216},
		{4192128, 112.1743},
		{4192992, 111.8057},
		{4193856, 111.4653},
		{4196448, 112.009},
		{4197312, 112.2046},
		{4198176, 111.9253},
		{4199904, 111.9039},
		{4202496, 112.4895},
		{4203360, 113.3006},
		{4204224, 113.2325},
		{4205088, 113.278},
		{4205952, 112.9063},
		{4208544, 112.0574},
		{4209408, 113.4303},
		{4210272, 114.1321},
		{4212000, 114.4335},
		{4214592, 114.416},
		{4215456, 114.5869},
		{4216320, 115.0592},
		{4217184, 116.3585},
		{4218048, 116.8131},
		{4221504, 115.764},
		{4222368, 114.9518},
		{4223232, 113.3864},
		{4224096, 113.1045},
		{4226688, 113.1399},
		{4227552, 112.4087},
		{4228416, 112.6405},
		{4229280, 112.5137},
		{4230144, 112.5042},
		{4232736, 112.1112},
		{4233600, 111.2877},
		{4234464, 110.6268},
		{4235328, 111.3997},
		{4236192, 113.0542},
		{4238784, 110.928},
		{4239648, 110.2662},
		{4240512, 109.9718},
		{4241376, 110.2159},
		{4242240, 110.4064},
		{4244832, 110.8801},
		{4245696, 111.8831},
		{4246560, 112.3271},
		{4247424, 110.9502},
		{4250880, 111.6702},
		{4251744, 112.2009},
		{4252608, 111.8221},
		{4253472, 112.0588},
		{4254336, 112.1553},
		{4256928, 110.3672},
		{4257792, 110.9584},
		{4258656, 110.3606},
		{4259520, 110.2633},
		{4260384, 110.1805},
		{4262976, 110.0204},
		{4263840, 110.3004},
		{4264704, 109.9525},
		{4265568, 110.1328},
		{4266432, 110.4653},
		{4269024, 110.2376},
		{4269888, 110.1041},
		{4270752, 110.1277},
		{4271616, 110.0286},
		{4272480, 109.5038},
		{4275072, 110.0094},
		{4275936, 109.7154},
		{4276800, 109.809},
		{4277664, 109.2192},
		{4278528, 109.1079},
		{4281120, 109.1013},
		{4281984, 109.0029},
		{4282848, 108.8078},
		{4283712, 108.7011},
		{4284576, 109.5642},
		{4288032, 109.3156},
		{4288896, 108.6261},
		{4289760, 108.5595},
		{4290624, 109.1413},
		{4293216, 109.1193},
		{4294080, 110.3837},
		{4294944, 110.4684},
		{4295808, 110.3078},
		{4296672, 109.8528},
		{4299264, 109.7388},
		{4300128, 109.6338},
		{4300992, 109.4952},
		{4301856, 109.0119},
		{4302720, 108.9703},
		{4305312, 108.9594},
		{4306176, 108.6613},
		{4307040, 109.101},
		{4307904, 108.91},
		{4308768, 108.6327},
		{4311360, 108.304},
		{4312224, 108.0502},
		{4313088, 107.7073},
		{4313952, 106.83},
		{4314816, 106.3506},
		{4317408, 106.8203},
		{4318272, 107.465},
		{4319136, 107.5125},
		{4320000, 107.9077},
		{4320864, 108.8949},
		{4323456, 107.6025},
		{4324320, 106.9893},
		{4325184, 107.1079},
		{4326048, 106.3284},
		{4326912, 105.8445},
		{4329504, 105.5182},
		{4330368, 105.329},
		{4331232, 104.741},
		{4332096, 105.2186},
		{4332960, 104.9294},
		{4335552, 106.3276},
		{4336416, 105.7706},
		{4337280, 106.6156},
		{4338144, 106.8333},
		{4339008, 104.2646},
		{4341600, 103.3093},
		{4342464, 103.329},
		{4343328, 103.0206},
		{4344192, 103.7792},
		{4345056, 104.0208},
		{4347648, 103.8688},
		{4348512, 103.3492},
		{4349376, 103.2449},
		{4350240, 103.2219},
		{4351104, 102.9773},
		{4353696, 102.7761},
		{4354560, 102.6924},
		{4355424, 102.3033},
		{4356288, 101.4377},
		{4357152, 101.6498},
		{4359744, 101.3133},
		{4360608, 101.6143},
		{4361472, 101.8009},
		{4362336, 102.7486},
		{4363200, 103.0884},
		{4365792, 103.1052},
		{4366656, 102.3125},
		{4367520, 100.61},
		{4368384, 100.6062},
		{4369248, 100.5898},
		{4371840, 100.8037},
		{4372704, 100.9038},
		{4373568, 102.1183},
		{4374432, 102.7174},
		{4375296, 102.5038},
		{4377888, 102.9698},
		{4378752, 102.4408},
		{4379616, 102.3065},
		{4380480, 101.9105},
		{4381344, 102.0777},
		{4383936, 102.5575},
		{4384800, 104.1244},
		{4385664, 104.3251},
		{4386528, 104.5727},
		{4387392, 104.5603},
		{4389984, 104.592},
		{4390848, 104.8966},
		{4391712, 104.9115},
		{4392576, 104.6155},
		{4393440, 105.105},
		{4396032, 106.7008},
		{4396896, 107.0113},
		{4397760, 107.0653},
		{4398624, 107.3151},
		{4399488, 106.5586},
		{4402080, 106.0282},
		{4402944, 105.4811},
		{4403808, 105.0054},
		{4404672, 104.9155},
		{4405536, 104.7986},
		{4408128, 105.6238},
		{4408992, 105.9612},
		{4409856, 105.6835},
		{4410720, 106.1023},
		{4411584, 106.0225},
		{4415040, 106.0396},
		{4415904, 106.1686},
		{4416768, 105.7012},
		{4417632, 104.9002},
		{4421088, 104.6979},
		{4421952, 104.9033},
		{4422816, 104.2058},
		{4423680, 103.1916},
		{4426272, 103.3088},
		{4427136, 103.2039},
		{4428000, 103.4172},
		{4428864, 103.1908},
		{4429728, 103.3046},
		{4432320, 103.5088},
		{4433184, 103.3415},
		{4434048, 103.9022},
		{4434912, 104.4168},
		{4435776, 104.1561},
		{4438368, 104.3602},
		{4439232, 104.6974},
		{4440096, 104.6187},
		{4440960, 104.3273},
		{4441824, 104.8025},
		{4444416, 104.5172},
		{4445280, 105.5128},
		{4446144, 106.3029},
		{4447008, 107.0495},
		{4447872, 106.6614},
		{4450464, 106.1577},
		{4451328, 105.4127},
		{4452192, 105.1116},
		{4453056, 105.307},
		{4453920, 106.0024},
		{4456512, 106.5154},
		{4457376, 106.4691},
		{4458240, 106.3918},
		{4459104, 106.2094},
		{4459968, 105.9202},
		{4462560, 105.8585},
		{4463424, 106.9487},
		{4464288, 114.9475},
		{4465152, 110.9776},
		{4466016, 110.7183},
		{4468608, 110.3704},
		{4469472, 109.323},
		{4470336, 108.3552},
		{4471200, 109.8133},
		{4472064, 110.75},
		{4474656, 111.1007},
		{4475520, 111.3095},
		{4476384, 115.2111},
		{4477248, 115.2127},
		{4478112, 117.6202},
		{4480704, 119.6527},
		{4481568, 118.6033},
		{4482432, 117.8102},
		{4483296, 117.5232},
		{4484160, 116.1034},
		{4486752, 117.8671},
		{4487616, 118.1098},
		{4488480, 121.2677},
		{4489344, 119.9668},
		{4490208, 122.9995},
		{4492800, 122.2739},
		{4493664, 121.1571},
		{4494528, 119.979},
		{4495392, 118.3584},
		{4496256, 119.5127},
		{4498848, 122.6043},
		{4499712, 122.0105},
		{4500576, 123.0136},
		{4501440, 124.546},
		{4502304, 122.8137},
		{4504896, 123.9253},
		{4505760, 125.1086},
		{4506624, 127.8039},
		{4511808, 129.2191},
		{4512672, 133.012},
		{4514400, 133.4757},
		{4516992, 132.1157},
		{4517856, 128.018},
		{4518720, 129.2496},
		{4519584, 127.5226},
		{4520448, 129.8152},
		{4523904, 128.8522},
		{4524768, 129.8949},
		{4525632, 126.1538},
		{4526496, 126.0492},
		{4529088, 126.172},
		{4529952, 124.1685},
		{4530816, 124.154},
		{4531680, 123.8988},
		{4532544, 126.4145},
		{4535136, 126.0306},
		{4536000, 125.8165},
		{4536864, 124.1185},
		{4537728, 125.3162},
		{4538592, 126.3217},
		{4541184, 126.5205},
		{4542048, 128.7086},
		{4542912, 128.7262},
		{4544640, 128.2582},
		{4547232, 127.9537},
		{4548096, 127.7045},
		{4548960, 127.2229},
		{4549824, 127.4115},
		{4550688, 126.7556},
		{4554144, 129.4194},
		{4555008, 129.6654},
		{4555872, 129.1178},
		{4556736, 129.2766},
		{4559328, 129.5617},
		{4560192, 130.2942},
		{4561056, 132.4199},
		{4561920, 131.4563},
		{4562784, 131.2253},
		{4565376, 130.9102},
		{4566240, 130.307},
		{4567104, 130.3053},
		{4567968, 130.9367},
		{4568832, 131.529},
		{4571424, 131.5983},
		{4572288, 131.8593},
		{4573152, 132.9201},
		{4574016, 132.8675},
		{4574880, 134.2338},
		{4577472, 132.8254},
		{4578336, 132.1127},
		{4579200, 132.2123},
		{4580064, 132.2102},
		{4580928, 133.1667},
		{4583520, 132.8628},
		{4584384, 131.711},
		{4585248, 130.3226},
		{4586112, 130.4029},
		{4586976, 130.9663},
		{4589568, 131.0616},
		{4590432, 131.5229},
		{4591296, 129.6199},
		{4592160, 128.0652},
		{4593024, 129.3089},
		{4595616, 129.2042},
		{4596480, 127.9713},
		{4597344, 126.075},
		{4598208, 126.4215},
		{4599072, 127.9599},
		{4601664, 128.0195},
		{4602528, 127.695},
		{4603392, 126.1532},
		{4604256, 124.9115},
		{4605120, 124.7985},
		{4608576, 125.4581},
		{4609440, 126.1362},
		{4610304, 125.2569},
		{4611168, 125.1323},
		{4613760, 125.3734},
		{4614624, 124.8114},
		{4615488, 123.214},
		{4616352, 122.4788},
		{4617216, 122.9028},
		{4619808, 123.2182},
		{4620672, 124.1951},
		{4621536, 123.7943},
		{4622400, 124.1575},
		{4623264, 123.3903},
		{4625856, 123.3483},
		{4626720, 123.3757},
		{4627584, 122.752},
		{4628448, 122.0226},
		{4629312, 121.4527},
		{4631904, 121.9623},
		{4632768, 121.8546},
		{4633632, 122.0622},
		{4634496, 122.746},
		{4635360, 125.2255},
		{4637952, 125.3103},
		{4638816, 124.5541},
		{4639680, 123.619},
		{4640544, 122.155},
		{4641408, 123.0054},
		{4644000, 123.0039},
		{4644864, 122.6211},
		{4645728, 122.6797},
		{4646592, 123.1116},
		{4647456, 124.9934},
		{4650048, 123.1544},
		{4650912, 123.0018},
		{4651776, 122.3617},
		{4652640, 122.6182},
		{4653504, 122.4036},
		{4656096, 122.6116},
		{4656960, 121.5721},
		{4657824, 120.3335},
		{4658688, 119.246},
		{4659552, 118.8299},
		{4662144, 119.303},
		{4663008, 119.4339},
		{4663872, 118.7077},
		{4664736, 119.0911},
		{4665600, 118.2062},
		{4668192, 117.4203},
		{4669056, 118.3867},
		{4669920, 118.4444},
		{4670784, 118.1458},
		{4671648, 118.7348},
		{4674240, 119.412},
		{4675104, 118.9053},
		{4675968, 118.4238},
		{4676832, 118.7211},
		{4677696, 118.7107},
		{4680288, 119.9011},
		{4681152, 119.4761},
		{4682016, 118.8102},
		{4682880, 119.1079},
		{4683744, 119.2079},
		{4686336, 119.5152},
		{4687200, 119.8127},
		{4688064, 119.9105},
		{4688928, 119.874},
		{4689792, 119.5097},
		{4692384, 120.322},
		{4693248, 121.8562},
		{4694112, 124.4026},
		{4694976, 123.1316},
		{4695840, 122.9202},
		{4698432, 124.4081},
		{4699296, 124.3339},
		{4700160, 127.264},
		{4701024, 126.0731},
		{4701888, 126.0169},
		{4704480, 125.822},
		{4705344, 125.5143},
		{4706208, 124.1972},
		{4707072, 122.7957},
		{4707936, 122.6835},
		{4710528, 123.206},
		{4711392, 123.7516},
		{4712256, 125.0401},
		{4713120, 125.0136},
		{4713984, 124.909},
		{4716576, 125.4188},
		{4717440, 125.3626},
		{4718304, 125.508},
		{4719168, 125.2147},
		{4720032, 123.4369},
		{4722624, 122.7997},
		{4723488, 124.0078},
		{4724352, 124.8507},
		{4725216, 125.6024},
		{4726080, 125.7233},
		{4730400, 128.6764},
		{4731264, 128.0206},
		{4732128, 129.1819},
		{4736448, 125.9513},
		{4737312, 125.4304},
		{4738176, 126.1028},
		{4740768, 126.0146},
		{4741632, 126.3699},
		{4742496, 128.424},
		{4743360, 128.5989},
		{4744224, 126.3137},
		{4746816, 126.1298},
		{4747680, 125.5019},
		{4748544, 125.2762},
		{4749408, 124.3974},
		{4750272, 123.8134},
		{4752864, 123.9708},
		{4753728, 123.0216},
		{4754592, 123.0243},
		{4755456, 122.6242},
		{4756320, 123.8912},
		{4758912, 122.6187},
		{4759776, 122.1371},
		{4760640, 121.6152},
		{4761504, 122.12},
		{4762368, 122.3012},
		{4764960, 121.8933},
		{4765824, 121.8467},
		{4766688, 121.4696},
		{4767552, 121.2176},
		{4768416, 120.7431},
		{4771008, 121.1998},
		{4771872, 121.5134},
		{4772736, 121.0184},
		{4773600, 120.9111},
		{4774464, 120.7878},
		{4777056, 119.4185},
		{4777920, 119.0058},
		{4778784, 119.0029},
		{4779648, 119.2198},
		{4780512, 119.372},
		{4783104, 117.8456},
		{4783968, 118.921},
		{4784832, 121.1367},
		{4785696, 119.5324},
		{4786560, 120.2721},
		{4789152, 122.9212},
		{4790016, 121.7263},
		{4790880, 121.2556},
		{4791744, 120.3236},
		{4792608, 120.6374},
		{4795200, 121.0242},
		{4796064, 120.8574},
		{4796928, 121.8987},
		{4797792, 121.1486},
		{4798656, 121.564},
		{4801248, 121.2224},
		{4802112, 120.8981},
		{4802976, 121.1966},
		{4803840, 120.4989},
		{4804704, 120.4073},
		{4807296, 120.212},
		{4808160, 120.0712},
		{4809024, 119.962},
		{4809888, 120.0543},
		{4810752, 119.1544},
		{4813344, 119.8144},
		{4814208, 120.311},
		{4815072, 120.8434},
		{4820256, 121.8026},
		{4821120, 121.0294},
		{4821984, 120.269},
		{4822848, 120.4336},
		{4825440, 119.6466},
		{4826304, 119.6708},
		{4827168, 119.5582},
		{4828896, 118.8043},
		{4831488, 118.405},
		{4832352, 118.5436},
		{4833216, 118.1724},
		{4834080, 118.2242},
		{4834944, 118.4943},
		{4837536, 117.9618},
		{4839264, 117.5534},
		{4840128, 117.008},
		{4840992, 116.8998},
		{4843584, 116.6918},
		{4844448, 117.3953},
		{4845312, 117.7108},
		{4846176, 116.5803},
		{4847040, 116.724},
		{4849632, 116.6103},
		{4850496, 116.1477},
		{4851360, 115.6081},
		{4853088, 115.105},
		{4855680, 115.5614},
		{4856544, 114.195},
		{4857408, 113.2292},
		{4858272, 113.5723},
		{4859136, 113.4666},
		{4862592, 112.8278},
		{4863456, 113.2095},
		{4864320, 112.4049},
		{4865184, 112.4123},
		{4867776, 113.3578},
		{4868640, 115.0122},
		{4869504, 114.5952},
		{4870368, 114.604},
		{4871232, 117.2002},
		{4873824, 116.0283},
		{4874688, 115.0183},
		{4875552, 115.1363},
		{4876416, 114.4581},
		{4877280, 113.8043},
		{4879872, 113.1192},
		{4880736, 113.069},
		{4881600, 113.1181},
		{4882464, 113.4134},
		{4883328, 113.7954},
		{4885920, 114.5367},
		{4886784, 115.0599},
		{4887648, 114.9225},
		{4888512, 114.2711},
		{4889376, 114.0066},
		{4891968, 113.4758},
		{4892832, 114.5641},
		{4893696, 114.2303},
		{4894560, 114.501},
		{4895424, 113.1922},
		{4898016, 112.36},
		{4898880, 111.7588},
		{4899744, 112.6405},
		{4900608, 112.2974},
		{4901472, 111.7641},
		{4904064, 111.6976},
		{4904928, 111.9227},
		{4905792, 111.2047},
		{4906656, 110.6658},
		{4907520, 110.7191},
		{4910112, 111.1526},
		{4910976, 110.4218},
		{4911840, 110.9281},
		{4912704, 111.4678},
		{4913568, 113.7187},
		{4916160, 114.7718},
		{4917024, 113.1068},
		{4917888, 116.1711},
		{4918752, 116.2128},
		{4919616, 115.7093},
		{4923072, 117.575},
		{4923936, 117.603},
		{4924800, 118.9688},
		{4925664, 121.8356},
		{4928256, 121.5073},
		{4929120, 120.8047},
		{4929984, 123.3077},
		{4930848, 126.7513},
		{4931712, 126.5606},
		{4934304, 123.8451},
		{4935168, 123.4},
		{4936032, 120.2296},
		{4936896, 118.169},
		{4937760, 119.85},
		{4940352, 118.2069},
		{4941216, 118.5708},
		{4942080, 118.8583},
		{4942944, 117.6038},
		{4943808, 116.911},
		{4946400, 118.0399},
		{4947264, 119.4764},
		{4948128, 119.3108},
		{4948992, 119.2249},
		{4949856, 119.182},
		{4952448, 121.6323},
		{4953312, 121.6331},
		{4954176, 120.4348},
		{4955040, 119.462},
		{4955904, 120.2988},
		{4958496, 121.784},
		{4959360, 121.4957},
		{4960224, 119.0658},
		{4961088, 118.7165},
		{4961952, 118.0944},
		{4964544, 117.9035},
		{4965408, 117.9937},
		{4966272, 117.4557},
		{4967136, 116.9291},
		{4968000, 117.5558},
		{4970592, 117.4034},
		{4971456, 117.2752},
		{4972320, 116.928},
		{4973184, 116.3999},
		{4974048, 115.8819},
		{4976640, 115.2923},
		{4977504, 114.7119},
		{4978368, 114.6265},
		{4979232, 114.1166},
		{4980096, 114.3398},
		{4982688, 114.2737},
		{4983552, 115.6033},
		{4984416, 115.023},
		{4985280, 114.3544},
		{4986144, 114.3106},
		{4988736, 117.0243},
		{4989600, 115.4135},
		{4990464, 115.8769},
		{4991328, 116.1749},
		{4992192, 116.0612},
		{4994784, 115.917},
		{4995648, 116.1088},
		{4996512, 115.388},
		{4997376, 113.1519},
		{4998240, 113.6952},
		{5000832, 114.1135},
		{5001696, 113.6682},
		{5002560, 114.0092},
		{5003424, 117.089},
		{5004288, 117.022},
		{5006880, 117.4864},
		{5007744, 117.4104},
		{5008608, 115.9641},
		{5009472, 117.2779},
		{5010336, 118.1038},
		{5012928, 117.8733},
		{5013792, 120.0945},
		{5014656, 121.7467},
		{5015520, 122.2104},
		{5016384, 122.4064},
		{5018976, 122.0745},
		{5019840, 122.9697},
		{5020704, 122.1354},
		{5021568, 119.2659},
		{5022432, 118.6192},
		{5025024, 119.3144},
		{5025888, 119.6948},
		{5026752, 120.7124},
		{5027616, 118.7649},
		{5028480, 119.0918},
		{5031072, 119.1925},
		{5031936, 118.6671},
		{5032800, 118.6159},
		{5033664, 117.8893},
		{5034528, 119.5766},
		{5037120, 121.3286},
		{5037984, 120.3109},
		{5038848, 120.7373},
		{5039712, 121.018},
		{5040576, 121.4348},
		{5045760, 121.0152},
		{5046624, 119.9007},
		{5049216, 119.9954},
		{5051808, 121.3622},
		{5052672, 118.9995},
		{5055264, 119.713},
		{5056128, 119.8053},
		{5056992, 121.6664},
		{5057856, 121.0189},
		{5058720, 121.933},
		{5061312, 124.2274},
		{5062176, 124.6095},
		{5063040, 126.3363},
		{5063904, 126.1198},
		{5064768, 125.6062},
		{5067360, 126.3174},
		{5068224, 128.39},
		{5069088, 127.6012},
		{5069952, 127.9944},
		{5070816, 125.8651},
		{5073408, 125.9478},
		{5074272, 125.3678},
		{5075136, 125.0678},
		{5076000, 126.7234},
		{5076864, 126.1138},
		{5079456, 125.6872},
		{5080320, 126.0},
		{5081184, 126.9569},
		{5082048, 128.5455},
		{5082912, 129.1872},
		{5085504, 131.1699},
		{5086368, 131.3003},
		{5087232, 129.1076},
		{5088096, 128.8077},
		{5088960, 128.9687},
		{5091552, 128.8935},
		{5092416, 129.204},
		{5093280, 129.6093},
		{5094144, 129.9141},
		{5095008, 130.2087},
		{5097600, 129.5118},
		{5098464, 128.6169},
		{5099328, 128.5887},
		{5100192, 128.8604},
		{5101056, 129.9155},
		{5103648, 131.3773},
		{5104512, 131.0179},
		{5105376, 131.0144},
		{5106240, 132.0172},
		{5107104, 135.993},
		{5109696, 136.707},
		{5110560, 136.4637},
		{5111424, 136.6812},
		{5112288, 141.716},
		{5113152, 140.8457},
		{5115744, 152.279},
		{5116608, 157.6591},
		{5117472, 157.5528},
		{5122656, 150.2834},
		{5123520, 154.7375},
		{5124384, 151.2994},
		{5125248, 157.7845},
		{5127840, 155.5023},
		{5128704, 155.2227},
		{5129568, 151.16},
		{5130432, 150.3135},
		{5131296, 150.3984},
		{5133888, 145.9293},
		{5134752, 146.7342},
		{5135616, 145.221},
		{5136480, 146.9964},
		{5137344, 148.096},
		{5139936, 150.4997},
		{5140800, 151.2567},
		{5141664, 151.005},
		{5142528, 150.6251},
		{5143392, 153.6749},
		{5145984, 152.5092},
		{5146848, 151.1824},
		{5147712, 150.0292},
		{5149440, 147.9984},
		{5152032, 147.2289},
		{5152896, 148.3362},
		{5153760, 149.1537},
		{5155488, 149.7958},
		{5158080, 152.6918},
		{5158944, 153.9971},
		{5159808, 153.2971},
		{5160672, 153.3181},
		{5161536, 158.011},
		{5164992, 157.4689},
		{5165856, 158.7133},
		{5166720, 154.7626},
		{5167584, 147.7172},
		{5170176, 146.3585},
		{5171040, 148.6551},
		{5171904, 148.438},
		{5172768, 147.3325},
		{5173632, 145.8501},
		{5176224, 145.9253},
		{5177088, 146.1207},
		{5177952, 147.8522},
		{5178816, 148.2986},
		{5179680, 149.2329},
		{5182272, 150.5012},
		{5183136, 153.301},
		{5184000, 154.2188},
		{5184864, 153.5053},
		{5185728, 152.0602},
		{5188320, 152.4684},
		{5189184, 152.8753},
		{5190048, 155.0305},
		{5190912, 155.8811},
		{5191776, 157.1712},
		{5194368, 157.2632},
		{5196096, 159.7244},
		{5196960, 163.7717},
		{5197824, 161.1595},
		{5200416, 164.2721},
		{5201280, 168.8934},
		{5202144, 165.9207},
		{5203008, 162.35},
		{5203872, 164.5723},
		{5206464, 160.4153},
		{5207328, 160.9275},
		{5208192, 158.7194},
		{5209056, 159.2223},
		{5209920, 155.545},
		{5212512, 154.351},
		{5213376, 155.0526},
		{5214240, 152.299},
		{5215104, 151.9111},
		{5215968, 153.2111},
		{5218560, 155.3119},
		{5219424, 158.6655},
		{5220288, 158.5155},
		{5221152, 155.694},
		{5222016, 160.4258},
		{5224608, 158.8205},
		{5225472, 160.5394},
		{5226336, 160.1212},
		{5227200, 161.5202},
		{5228064, 163.6289},
		{5230656, 165.7673},
		{5231520, 161.9275},
		{5232384, 160.1269},
		{5233248, 158.9515},
		{5234112, 158.7835},
		{5237568, 157.6715},
		{5238432, 156.755},
		{5239296, 158.2687},
		{5240160, 162.0081},
		{5242752, 158.7652},
		{5243616, 158.8034},
		{5244480, 159.0102},
		{5245344, 156.7428},
		{5246208, 157.7371},
		{5248800, 157.2125},
		{5249664, 158.2334},
		{5250528, 158.6768},
		{5251392, 158.9171},
		{5252256, 157.2593},
		{5254848, 157.0668},
		{5255712, 159.1269},
		{5256576, 158.4522},
		{5257440, 158.416},
		{5258304, 158.8986},
		{5260896, 159.9148},
		{5261760, 159.18},
		{5262624, 160.3788},
		{5263488, 160.9912},
		{5264352, 164.798},
		{5266944, 162.7312},
		{5267808, 166.3355},
		{5268672, 169.1668},
		{5269536, 167.6603},
		{5270400, 168.4152},
		{5272992, 170.9641},
		{5273856, 171.5335},
		{5274720, 172.7131},
		{5275584, 177.2699},
		{5276448, 173.0016},
		{5279040, 170.3129},
		{5279904, 180.018},
		{5280768, 182.15},
		{5281632, 177.399},
		{5282496, 184.1072},
		{5285088, 187.0252},
		{5285952, 190.4998},
		{5286816, 201.241},
		{5287680, 204.6839},
		{5288544, 206.7204},
		{5291136, 207.1209},
		{5292000, 181.6678},
		{5292864, 230.0012},
		{5293728, 191.6819},
		{5294592, 199.9866},
		{5297184, 199.9911},
		{5298048, 199.6994},
		{5298912, 199.2305},
		{5299776, 200.418},
		{5300640, 201.1017},
		{5303232, 200.5673},
		{5304096, 201.267},
		{5304960, 202.3754},
		{5305824, 202.8365},
		{5306688, 205.6808},
		{5309280, 205.6906},
		{5310144, 207.3961},
		{5311008, 205.6388},
		{5311872, 205.1982},
		{5312736, 207.7121},
		{5315328, 216.0081},
		{5316192, 221.392},
		{5317056, 220.4588},
		{5317920, 222.8941},
		{5318784, 223.2843},
		{5321376, 222.6054},
		{5322240, 230.1081},
		{5323104, 230.9533},
		{5323968, 230.4237},
		{5324832, 228.7913},
		{5327424, 229.8377},
		{5328288, 233.6497},
		{5329152, 236.1288},
		{5330016, 236.5712},
		{5330880, 237.5162},
		{5333472, 237.4835},
		{5334336, 239.6188},
		{5335200, 240.6292},
		{5336064, 243.1769},
		{5336928, 243.644},
		{5339520, 250.6977},
		{5340384, 250.1019},
		{5341248, 250.0794},
		{5342112, 243.8797},
		{5342976, 214.6956},
		{5345568, 196.3379},
		{5346432, 190.7223},
		{5347296, 195.8442},
		{5348160, 199.8842},
		{5349024, 204.102},
		{5351616, 203.5171},
		{5352480, 202.9084},
		{5353344, 206.6133},
		{5354208, 207.2367},
		{5355072, 217.6413},
		{5357664, 217.5214},
		{5358528, 221.8308},
		{5363712, 227.2573},
		{5364576, 217.2919},
		{5365440, 216.2925},
		{5367168, 216.1348},
	},
}
